const CONTROLLER_BASE_URL = "admin_controller.php";
const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';
const DEFAULT_SORT = 'Created_Time';

/**
 * @type {HTMLCollectionOf<HTMLImageElement>}
 */
const sort_icons = document.getElementsByClassName('sort_icon');

/**
 * @type {HTMLButtonElement}
 */
const saveButtons = document.getElementsByClassName('save_button');

/**
 * @type {HTMLTableElement}
 */
const record_table = document.getElementById('record_table');

/**
 * @type {HTMLHeadingElement}
 */
const title_header = document.getElementById('title');

const columnDefinitions = {
	owner: new ColumnDefinition('Sales', 'salesFirstName', {isSortable: true}),
	mts: new ColumnDefinition('MTS', 'mtsNumber', {isSortable: true, linkBase: 'https://crm.zoho.com/crm/org26161413/tab/Invoices/LINKTO', linksTo: 'Invoice_Id'}),
	account_name: new ColumnDefinition('School', 'accountName', {isSortable: true, linkBase: 'https://crm.zoho.com/crm/org26161413/tab/Accounts/LINKTO', linksTo: 'Account_Id'}),
	created_time: new ColumnDefinition('Invoice Date', 'invoiceDate', {isSortable: true}),
	products: new ColumnDefinition('SKU', 'lineItemSku'),
	quantity: new ColumnDefinition('Quantity', 'lineItemQuantity'),
	notes: new ColumnDefinition('Notes', 'notes'),
	ap: new ColumnDefinition('AP', 'apDataPresent'),
	at: new ColumnDefinition('AT', 'assetTags'),
	subtotal: new ColumnDefinition('Subtotal', 'subtotal'),
	status: new ColumnDefinition('Status', 'status'),
	newCustomer: new ColumnDefinition('NC', 'newCustomer')
};

/**
 * @type {ColumnDefinition[]}
 */
const columns = [
	columnDefinitions.owner,
	columnDefinitions.mts,
	columnDefinitions.account_name,
	columnDefinitions.created_time,
	columnDefinitions.products,
	columnDefinitions.quantity,
	columnDefinitions.notes,
	columnDefinitions.ap,
	columnDefinitions.at,
	columnDefinitions.newCustomer,
	columnDefinitions.status,
	columnDefinitions.subtotal,
];


const ORDER_DICT = {};
const ORDER_ARRAYS = {
	/**
	 * @type {OpenOrder[]}
	 */
	new_order : [],
	/**
	 * @type {OpenOrder[]}
	 */
	back_order : [],
	/**
	 * @type {OpenOrder[]}
	 */
	on_order : [],
	/**
	 * @type {OpenOrder[]}
	 */
	picked : [],
	/**
	 * @type {OpenOrder[]}
	 */
	ready_to_ship : []
}


const STATUS_DICT = {};
/**
 * @type {OpenOrderStatus[]}
 */
const STATUS_ARRAY = [];

const ORDER_VALUES = {
	total: 0,
	new_order: 0,
	back_order: 0,
	on_order: 0,
	picked: 0,
	ready_to_ship: 0
}

const sort = {
	by: 'invoiceDate',
	order: 1
};

let listenerAttached = false;
let updateElementsVisible = false;

init();

async function init() {
	console.log("init begin");
	const returnedData = await getOpenOrders();
	console.log("got data");
	console.log(returnedData);
	if (returnedData) {
		clearArrays();
	}
	processStatuses(returnedData['statuses']);
	processOrders(returnedData['openOrders']);
	renderPage();
	setLoadingModalVisible(false);

	// if (returnedData['statuses']) {
	// 	const statuses = returnedData['statuses'];
	// 	STATUS_ARRAY.length = 0;
	// 	for(const status of statuses) {
	// 		STATUS_ARRAY.push(new Status(status));
	// 	}
	// }


	// if (returnedData['openQuotes']) {
	// 	const openQuotes = returnedData['openQuotes'];
	// 	processOpenQuotes(openQuotes);
	// 	sortQuoteArrays();
	// }
	// getStatuses();

	function processOrders(orders) {
		for (const orderData of orders) {
			const order = new OpenOrder(orderData);
			const orderValue = order.getSubtotal();
			const orderStatusId = order.getStatusId();
			/**
			 * @type {OpenOrderStatus}
			 */
			const orderStatus = STATUS_DICT[orderStatusId];
			const orderStatusIdentifier = orderStatus.getIdentifier();
			ORDER_VALUES.total += orderValue;
			ORDER_VALUES[orderStatusIdentifier] += orderValue;

			ORDER_DICT[order.getId()] = order;
			/**
			 * @type {OpenOrder[]}
			 */
			const orderArray = ORDER_ARRAYS[orderStatusIdentifier];
			try {
				orderArray.push(order);

			}
			catch (err) {
				console.error(err);
				console.log(orderStatusIdentifier);
				console.log(orderArray);
				console.log(ORDER_ARRAYS[orderStatusIdentifier]);
			}
		}
	}

	function processStatuses(statuses) {
		for (const statusData of statuses) {
			const status = new OpenOrderStatus(statusData);
			const statusId = status.getId();
			STATUS_DICT[statusId] = status;
			STATUS_ARRAY.push(status);
		}
	}
}

async function zohoRefreshOpenOrders() {
	setLoadingModalVisible(true);
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=open_orders_refresh`)
		.then(response => response.text())
		.then(data => { return data }
	);
	return response;
}

async function getRecords() {
	init();
}

async function getOpenOrders() {
	const updateResponse = await fetch(`${CONTROLLER_BASE_URL}?m=open_orders_update`)
		.then(response => response.text())
		.then(data => {return data});
	console.log(updateResponse);
	const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=open_orders_fetch`)
		.then(response => response.text())
		.then(data => {return data});
	try {
		const data = JSON.parse(responseText);
		return data;
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
		return null;
	}
}

function clearArrays() {
	for (const status in ORDER_ARRAYS) {
		/**
		 * @type {OpenOrder[]}
		 */
		const arr = ORDER_ARRAYS[status];
		arr.length = 0;
	}
	for (const element in ORDER_DICT) {
		delete ORDER_DICT[element];
	}
	STATUS_ARRAY.length = 0;
	for (const element in STATUS_DICT) {
		delete STATUS_DICT[element];
	}
	for (const status in ORDER_VALUES) {
		ORDER_VALUES[status] = 0;
	}
}


function addResizeListener() {
	if (!listenerAttached) {
		window.addEventListener('resize', () => {
			title_header.style.maxWidth = `${record_table.clientWidth}px`;
		});
	}
	listenerAttached = true;
}

function renderPage() {
	/**
	 * @type {HTMLSpanElement}
	 */
	const total_orders_span = document.getElementById('total_orders_span');
	/**
	 * @type {HTMLSpanElement}
	 */
	const total_orders_header_number = document.getElementById('total_orders_header_number');
	/**
	 * @type {HTMLSpanElement}
	 */
	const total_value_header_number = document.getElementById('total_value_header_number');
	total_orders_header_number.innerHTML = '';
	total_orders_header_number.innerHTML = Object.keys(ORDER_DICT).length.toLocaleString();

	total_value_header_number.innerHTML = numberToCurrencyString(Math.round(ORDER_VALUES.total));

	// clearPage();
	renderHeaders();
	renderBody();
	title_header.style.maxWidth = `${record_table.clientWidth}px`;
	// addResizeListener();
	total_orders_span.classList.remove('pseudo_hidden');
	// setLoadingModalVisible(false);
}

function renderHeaders() {
	const sections = [
		'new_order',
		'back_order',
		'on_order',
		'picked',
		'ready_to_ship'
	];

	for (const section of sections) {
		renderHeader(section);
	}

	renderValues();
	function renderHeader(section) {
		/**
		 * @type {HTMLTableRowElement}
		 */
		const headerRow = document.getElementById(`${section}_header_row`);
		headerRow.innerHTML = '';
		for (const column of columns) {
			headerRow.appendChild(column.getHeaderCellElement());
		}
		const saveAllCell = document.createElement('th');
		const saveAllButton = document.createElement('button');
		saveAllButton.id = 'save_button_all';
		saveAllButton.setAttribute('onclick', 'saveAllChanges()');
		saveAllButton.innerText = "Save All Changes";
		saveAllCell.appendChild(saveAllButton);
		headerRow.appendChild(saveAllCell);
	}

	function renderValues() {
		const valueHeaders = {
			new_order: document.getElementById('value_new_order'),
			back_order: document.getElementById('value_back_order'),
			on_order: document.getElementById('value_on_order'),
			picked: document.getElementById('value_picked'),
			ready_to_ship: document.getElementById('value_ready_to_ship'),
		}
		for (const header in valueHeaders) {
			valueHeaders[header].innerHTML = numberToCurrencyString(Math.round(ORDER_VALUES[header]));
		}
	}
}

function renderBody() {
	for (const status in ORDER_ARRAYS) {
		/**
		 * @type {OpenOrder[]}
		 */
		const currentArray = ORDER_ARRAYS[status];
		/**
		 * @type {HTMLTableSectionElement}
		 */
		const currentTableSection = document.getElementById(`${status}_body`);
		currentTableSection.innerHTML = '';
		for (const order of currentArray) {
			const rows = order.generateRows(columns);
			for (const row of rows) {
				currentTableSection.appendChild(row);
			}
		}
	}
}

/** helper functions start **/

function numberToCurrencyString(num) {
	try {
		return num.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
	}
	catch (err) {
		console.warn(err);
		return 'N/A';
	}
}

/** helper functions end **/

async function performSaveChanges(saveButton) {
	saveButton.disabled = true;
	const orderId = saveButton.getAttribute('order_id');
	const record = ORDER_DICT[orderId];
	console.log(record);
	const result = await saveChanges(record);
	saveButton.disabled = false;
	init();
}

/**
 *
 * @param {OpenOrder} record Open order to save
 * @returns
 */
async function saveChanges(record) {
	const dataToSave = record.getSaveableFieldsAsObject();
	if (!dataToSave) {
		return false;
	}
	console.log(dataToSave);
	setLoadingModalVisible(true);
	console.log(dataToSave);
	const stringified = JSON.stringify(dataToSave).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");
	const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=open_orders_saveChanges`, {
		method: 'POST',
		headers: {
			'Accept' : 'application/json',
			'Content-Type': 'application/json'
		},
		body: stringified
	});
	const content = await rawResponse.text();

	console.log(content);

	return true;

}

async function saveAllChanges() {
	/**
	 * @type {HTMLButtonElement}
	 */
	const saveButton = document.getElementById(`save_button_all`);
	saveButton.disabled = true;

	for (const status in ORDER_ARRAYS) {
		/**
		 * @type {OpenOrder[]}
		 */
		const orderArray = ORDER_ARRAYS[status];
		for (const record of orderArray) {
			if (record.getIsChanged()) {
				const result = await saveChanges(record);
			}
		}
	}

	saveButton.disabled = false;
	init();
}


/**
 *
 * @param {HTMLTextAreaElement} obj
 */
function updateNote(obj) {
	const invoiceId = obj.getAttribute('invoice_id');
	/**
	 * @type {OpenOrder}
	 */
	const invoice = ORDER_DICT[invoiceId];
	invoice.setNotes(obj.value.trim());
	invoice.setIsChanged(true);
}

/**
 *
 * @param {HTMLInputElement} input Checkbox of the AT status
 */
function updateATStatus(input) {
	const invoiceId = input.getAttribute('invoice_id');
	/**
	 * @type {OpenOrder}
	 */
	const invoice = ORDER_DICT[invoiceId];
	invoice.setAssetTags(input.checked);
	invoice.setIsChanged(true);
}

/**
 *
 * @param {HTMLSelectElement} obj
 */
function updateStatus(obj) {
	const invoiceId = obj.getAttribute('invoice_id');
	/**
	 * @type {OpenOrder}
	 */
	const invoice = ORDER_DICT[invoiceId];
	invoice.setStatusId(obj.value);
	invoice.setIsChanged(true);
}

function toggleUpdateElementsVisible() {
	updateElementsVisible = !updateElementsVisible;
	setUpdateElementsVisible(updateElementsVisible);
}

/**
 *
 * @param {boolean} visible
 */
function setUpdateElementsVisible(visible) {
	/**
	 * @type {HTMLCollectionOf<HTMLTableCellElement>}
	 */
	const columnElements = document.getElementsByClassName('updateElement');
	for (const element of columnElements) {
		element.hidden = !visible;
	}
}

/**
 *
 * @param {string} id
 */
async function addToDeductions(id) {
	const url = `${CONTROLLER_BASE_URL}?m=addToDeductions&module=tts&view=open&id=${id}`;
	const response = await fetch(url)
		.then(response => response.text())
		.then(data => {
			console.log(data);
			return data;
		});
	init();
}

// sort functions below


function setSort(newSort) {
	if (newSort == sort.by) {
		sort.order *= -1;
	} else {
		sort.by = newSort;
	}
	sortRecords();
	renderBody();
}

function sortRecords() {
	for (const status in ORDER_ARRAYS) {
		/**
		 * @type {OpenOrder[]}
		 */
		const orderArray = ORDER_ARRAYS[status];
		orderArray.sort((a, b) => sortFunction(a,b));
	}
	for (const icon of sort_icons) {
		if (icon.parentElement.getAttribute('sort_order') != sort.by) {
			icon.src = SORT_NONE_URL;
		} else {
			if (sort.order == 1) {
				icon.src = SORT_ASC_URL;
			} else if (sort.order == -1) {
				icon.src = SORT_DESC_URL;
			}
		}
	}
	/**
	 *
	 * @param {OpenOrder} a First open order to compare
	 * @param {OpenOrder} b Second open order to compare
	 * @returns Sort value
	 */
	function sortFunction(a,b) {
		const aValue = a.getFieldByName(sort.by);
		const bValue = b.getFieldByName(sort.by);
		if(isNaN(aValue) || (aValue instanceof Date && bValue instanceof Date)) {
			if(aValue < bValue) {
				return -1 * sort.order;
			}
			if(aValue > bValue) {
				return 1 * sort.order;
			}
		} else {
			if(parseInt(aValue) < parseInt(bValue)) {
				return -1 * sort.order;
			}
			if(parseInt(aValue) > parseInt(bValue)) {
				return 1 * sort.order;
			}
		}
	}
}
