class OpenOrder {
	constructor(data) {
		/**
		 * @type {string}
		 */
		this.id = data['id'];
		if (data['dbId']) {
			this.id = data['dbId'];
		}
		/**
		 * @type {string}
		 */
		this.zohoId = data['zohoId'];
		/**
		 * @type {string}
		 */
		this.mtsNumber = data['mtsNumber'];
		/**
		 * @type {string}
		 */
		this.notes = data['notes'];
		/**
		 * @type {string}
		 */
		this.statusId = data['statusId'];
		/**
		 * @type {string}
		 */
		this.initialStatusId = data['statusId'];
		/**
		 * @type {string}
		 */
		const dateStr = data['invoiceDate'];
		this.invoiceDate = new Date(dateStr.replace(' ', 'T'));
		/**
		 * @type {boolean}
		 */
		this.apDataPresent = data['apDataPresent'];
		/**
		 * @type {boolean}
		 */
		this.assetTags = data['assetTags'];
		/**
		 * @type {boolean}
		 */
		this.newCustomer = data['newCustomerOrder'];
		/**
		 * @type {number}
		 */
		this.total = data['total'];

		this.account = new Account(data['account']);
		this.salesperson = new Salesperson(data['salesperson']);
		/**
		 * @type {OpenOrderLineItem[]}
		 */
		this.lineItems = [];

		for (const lineItem of data['lineItems']) {
			this.lineItems.push(new OpenOrderLineItem(lineItem));
		}

		this.changed = false;
		this.changedFields = {
			notes : false,
			assetTags : false,
			statusId : false
		}

		this.rows = {
			main : null,
			subordinate : []
		}
	}

	/**
	 *
	 * @param {string} fieldName
	 * @returns
	 */
	getFieldByName(fieldName) {
		if (fieldName == 'salesFirstName') {
			return this.getSalesperson().getFirstName();
		}
		else if (fieldName == 'mtsNumber') {
			return this.getMtsNumber();
		}
		else if (fieldName == 'accountName') {
			return this.getAccount().getName();
		}
		else if (fieldName == 'invoiceDate') {
			return this.getInvoiceDate();
		}
		else if (fieldName == 'subtotal') {
			return this.getSubtotal();
		}
		else {
			return '';
		}
	}


	/**
	 * Fetches order ID
	 * @returns ID of order
	 */
	getId() {
		return this.id;
	}
	/**
	 * Sets id to new value
	 * @param {string} id New ID value
	 */
	setId(id) {
		this.id = id;
	}

	/**
	 * Fetches order zoho ID
	 * @returns Zoho ID of order
	 */
	getZohoId() {
		return this.zohoId;
	}
	/**
	 * Sets id to new value
	 * @param {string} zohoId New ID value
	 */
	setZohoId(zohoId) {
		this.zohoId = zohoId;
	}

	/**
	 * Fetches order MTS Number
	 * @returns MTS Number of order
	 */
	getMtsNumber() {
		return this.mtsNumber;
	}
	/**
	 * Sets MTS Number to new value
	 * @param {string} mtsNumber New MTS Number value
	 */
	setMtsNumber(mtsNumber) {
		this.mtsNumber = mtsNumber;
	}

	/**
	 * Fetches order notes
	 * @returns notes of order
	 */
	getNotes() {
		return this.notes;
	}
	/**
	 * Sets notes to new value
	 * @param {string} notes New notes value
	 */
	setNotes(notes) {
		this.notes = notes;
	}

	/**
	 * Fetches order statusId
	 * @returns statusId of order
	 */
	getStatusId() {
		return this.statusId;
	}
	/**
	 * Sets statusId to new value
	 * @param {string} statusId New statusId value
	 */
	setStatusId(statusId) {
		this.statusId = statusId;
	}

	/**
	 * Fetches order invoice date
	 * @returns invoice date of order
	 */
	getInvoiceDate() {
		return this.invoiceDate;
	}
	/**
	 * Sets invoice date to new value
	 * @param {Date} invoiceDate New invoice date value
	 */
	setInvoiceDate(invoiceDate) {
		this.invoiceDate = invoiceDate;
	}

	/**
	 * Fetches order AP Data Present
	 * @returns AP Data Present of order
	 */
	getApDataPresent() {
		return this.apDataPresent;
	}
	/**
	 * Sets AP Data Present to new value
	 * @param {boolean} apDataPresent New AP Data Present value
	 */
	setApDataPresent(apDataPresent) {
		this.apDataPresent = apDataPresent;
	}

	/**
	 * Fetches order asset tags
	 * @returns asset tags of order
	 */
	getAssetTags() {
		return this.assetTags;
	}
	/**
	 * Sets asset tags to new value
	 * @param {boolean} assetTags New asset tags value
	 */
	setAssetTags(assetTags) {
		this.assetTags = assetTags;
	}

	/**
	 * Fetches order new customer
	 * @returns new customer of order
	 */
	getNewCustomer() {
		return this.newCustomer;
	}
	/**
	 * Sets new customer to new value
	 * @param {boolean} newCustomer New new customer value
	 */
	setNewCustomer(newCustomer) {
		this.newCustomer = newCustomer;
	}

	/**
	 * Fetches order total
	 * @returns total of order
	 */
	getTotal() {
		return this.total;
	}
	/**
	 * Sets total to new value
	 * @param {number} total New total value
	 */
	setTotal(total) {
		this.total = total;
	}

	/**
	 * Fetches order account
	 * @returns account of order
	 */
	getAccount() {
		return this.account;
	}
	/**
	 * Sets account to new value
	 * @param {Account} account New account value
	 */
	setAccount(account) {
		this.account = account;
	}

	/**
	 * Fetches order salesperson
	 * @returns salesperson of order
	 */
	getSalesperson() {
		return this.salesperson;
	}
	/**
	 * Sets salesperson to new value
	 * @param {Salesperson} salesperson New salesperson value
	 */
	setSalesperson(salesperson) {
		this.salesperson = salesperson;
	}

	/**
	 * Fetches order line items
	 * @returns line items of order
	 */
	getLineItems() {
		return this.lineItems;
	}
	/**
	 * Sets line items to new value
	 * @param {OpenOrderLineItem[]} lineItems New line items value
	 */
	setLineItems(lineItems) {
		this.lineItems = lineItems;
	}

	getSubtotal() {
		let subtotal = 0.0;
		for (const lineItem of this.getLineItems()) {
			if (!lineItem.getDeducted()) {
				subtotal += lineItem.getTotal();
			}
		}
		return subtotal;
	}

	getIsChanged() {
		return this.changed;
	}

	setIsChanged(changed=true) {
		this.changed = changed;
	}

	getChangedFields() {
		return this.changedFields;
	}

	/**
	 *
	 * @param {number} index
	 */
	getLineItemByIndex(index) {
		const lineItems = this.getLineItems();
		// Make sure the index isn't out of bounds
		if (index >= lineItems.length) {
			index = lineItems.length - 1;
		}
		// Make sure the index is 0 or positive
		if (index >= 0) {
			return lineItems[index]; // If in bounds, return line item at index
		}
		else {
			return null; // Else return null
		}
	}

	getSaveableFieldsAsObject() {
		if (!this.getIsChanged()) {
			return null;
		}
		const toSave = {
			dbId: this.getId(),
			fields: {
				notes : this.getNotes(),
				assetTags : this.getAssetTags(),
				statusId : this.getStatusId()
			}
		};
		return toSave;
	}

	/** end of getters and setters **/

	/** row generators **/

	/**
	 *
	 * @param {ColumnDefinition[]} columns Column definitions used to generate the row
	 * @returns Array of table rows
	 */
	generateRows(columns) {
		this.rows.main = null;
		this.rows.subordinate.length = 0;
		const rows = [];
		const rowspan = this.getVisibleLineItemIndexes().length;
		const mainRow = this.generateMainRow(columns, rowspan);
		this.rows.main = mainRow;
		rows.push(mainRow);
		const visibleLineItemIndexes = this.getVisibleLineItemIndexes();
		for (let i = 1; i < visibleLineItemIndexes.length; i++) {
			const lineItemIndex = visibleLineItemIndexes[i];
			const itemRow = this.generateLineItemRow(lineItemIndex);
			rows.push(itemRow);
			this.rows.subordinate.push(itemRow);
		}
		return rows;
	}

	/**
	 * Generates main order row
	 * @param {ColumnDefinition[]} columns Column definitions from which to generate the row
	 * @param {number} rowspan rowspan of the main row
	 * @returns Main order row
	 */
	generateMainRow(columns, rowspan) {
		const row = document.createElement('tr');
		const firstVisibleLineItem = this.getFirstVisibleLineItemIndex();
		for (const column of columns) {
			const identifier = column.getIdentifier();
			row.appendChild(this.generateTableCellByIdentifier(identifier, firstVisibleLineItem, rowspan));
		}
		row.appendChild(this.generateSaveChangesCell(rowspan));
		return row;
	}

	/**
	 * Build line item row
	 * @param {number} index Line item index to build the row from
	 * @returns Line item row
	 */
	generateLineItemRow(index) {
		const row = document.createElement('tr');
		const identifiers = ['lineItemSku', 'lineItemQuantity'];
		for (const identifier of identifiers) {
			row.appendChild(this.generateTableCellByIdentifier(identifier, index));
		}
		return row;
	}

	generateNotesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const textarea = document.createElement('textarea');
		cell.appendChild(textarea);
		textarea.rows = Math.max(3, rowspan);
		textarea.classList.add('note-input');
		textarea.setAttribute('invoice_id', this.getId());
		textarea.setAttribute('oninput', 'updateNote(this)');
		textarea.innerText = this.getNotes();
		textarea.placeholder = "Enter notes...";
		return cell;
	}

	/**
	 *
	 * @param {number} rowspan
	 */
	generateAssetTagsCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const checkbox = document.createElement('input');
		cell.appendChild(checkbox);
		checkbox.type = 'checkbox';
		checkbox.name = `at_${this.getZohoId()}`;
		checkbox.id = `at_${this.getZohoId()}`;
		checkbox.setAttribute('invoice_id', this.getId());
		checkbox.setAttribute('onchange', 'updateATStatus(this)');
		checkbox.checked = this.getAssetTags();
		return cell;
	}

	/**
	 *
	 * @param {number} rowspan
	 */
	getNewCustomerCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const checkbox = document.createElement('input');
		cell.appendChild(checkbox);
		checkbox.type = 'checkbox';
		checkbox.name = `nc_${this.getZohoId()}`;
		checkbox.id = `nc_${this.getZohoId()}`;
		checkbox.setAttribute('invoice_id', this.getId());
		checkbox.checked = this.getNewCustomer();
		// checkbox.disabled = true;
		return cell;
	}

	/**
	 *
	 * @param {number} rowspan
	 */
	generateStatusCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const select = document.createElement('select');
		cell.appendChild(select);
		select.classList.add('borderless');
		select.id = `status_select_${this.getId()}`;
		select.setAttribute('invoice_id', this.getId());
		select.setAttribute('onchange', 'updateStatus(this)');
		for (const status of STATUS_ARRAY) {
			const selected = (this.getStatusId() == status.getId());
			select.appendChild(status.getStatusOption(selected));
		}
		return cell;
	}

	/**
	 * Generate the cell with the save changes button
	 * @param {number} rowspan Number of visible orders
	 * @returns cell with save changes button
	 */
	generateSaveChangesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const button = document.createElement('button');
		cell.appendChild(button);
		button.type = 'button';
		button.id = `save_button_${this.getId()}`;
		button.classList.add('save_button');
		button.setAttribute('order_id', this.getId());
		button.setAttribute('onclick', `performSaveChanges(this)`);
		button.innerText = `Save Changes`;
		return cell;
	}

	getHideSkuLink(zohoId) {
		const small = document.createElement('small');
		const span = document.createElement('span');
		small.appendChild(span);
		span.innerText = " | ";
		const link = document.createElement('a');
		small.appendChild(link);
		small.classList.add('updateElement');
		small.hidden = !updateElementsVisible;
		link.href = "javascript:void(0)";
		link.innerText = "(deduct)";
		link.setAttribute('onclick', `addToDeductions('${zohoId}')`);
		return small;
	}

	getFirstVisibleLineItemIndex() {
		const lineItems = this.getLineItems();
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (!lineItem.getDeducted()) {
				return i;
			}
		}
	}

	getVisibleLineItemIndexes() {
		const lineItems = this.getLineItems();
		const visibleIndexes = [];
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (!lineItem.getDeducted()) {
				visibleIndexes.push(i);
			}
		}
		return visibleIndexes;
	}

	getVisibleLineItems() {
		const indexes = this.getVisibleLineItemIndexes();
		const visibleLineItems = [];
		for (const index of indexes) {
			visibleLineItems.push(this.getLineItemByIndex(index));
		}
		return visibleLineItems;
	}

	/**
	 *
	 * @param {string} identifier
	 * @param {number} index
	 * @param {number} rowspan
	 */
	generateTableCellByIdentifier(identifier, index=0, rowspan=null) {
		if (identifier == 'salesFirstName') {
			return this.generateCellWithText(this.getSalesperson().getFirstName(), rowspan);
		}
		else if (identifier == 'mtsNumber') {
			return this.generateCellWithLink(this.getMtsNumber(), `https://crm.zoho.com/crm/org26161413/tab/Invoices/${this.getZohoId()}`, rowspan);
		}
		else if (identifier == 'accountName') {
			return this.generateCellWithLink(this.getAccount().getName(), `https://crm.zoho.com/crm/org26161413/tab/Accounts/${this.getAccount().getZohoId()}`, rowspan);
		}
		else if (identifier == 'invoiceDate') {
			return this.generateCellWithText(this.getInvoiceDate().toLocaleDateString(), rowspan);
		}
		else if (identifier == 'lineItemSku') {
			try {
				return this.getLineItemByIndex(index).getSkuCell();
			}
			catch (err) {
				console.error(err);
				console.log(this.getId(), this.getLineItems());
				return this.generatePlaceholderCell();
			}
		}
		else if (identifier == 'lineItemQuantity') {
			try {
				return this.getLineItemByIndex(index).getQuantityCell();
			}
			catch (err) {
				console.error(err);
				console.log(this.getId(), this.getLineItems());
				return this.generatePlaceholderCell();
			}
		}
		else if (identifier == 'notes') {
			return this.generateNotesCell(rowspan);
		}
		else if (identifier == 'apDataPresent') {
			return this.generateCheckboxCell(this.getApDataPresent(), rowspan);
		}
		else if (identifier == 'assetTags') {
			return this.generateAssetTagsCell(rowspan);
		}
		else if (identifier == 'newCustomer') {
			return this.generateCheckboxCell(this.getNewCustomer(), rowspan);
		}
		else if (identifier == 'status') {
			return this.generateStatusCell(rowspan);
		}
		else if (identifier == 'subtotal') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getSubtotal()), rowspan);
		}
		else {
			return this.generatePlaceholderCell(rowspan);
		}
	}

	/* generic generators */

	generatePlaceholderCell(rowspan=null) {
		return this.generateCellWithText('placeholder', rowspan);
	}

	/**
	 *
	 * @param {string|boolean} innerText
	 * @param {number} rowspan
	 */
	generateCellWithText(innerText, rowspan=null) {
		const cell = document.createElement('td');
		if (typeof innerText == 'boolean') {
			cell.innerText = (innerText) ? 'Yes' : 'No';
		}
		else {
			cell.innerText = (innerText) ? innerText : 'N/A';
		}
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	generateCellWithLink(innerText='', href='', rowspan=null) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		if (href) {
			link.href = href;
		}
		link.target = "_blank";
		if (innerText) {
			link.innerText = innerText;
		}
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	generateCheckboxCell(checked=false, rowspan=null) {
		const cell = document.createElement('td');
		const checkbox = document.createElement('input');
		checkbox.type = 'checkbox';
		cell.appendChild(checkbox);
		checkbox.checked = checked;
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	/**
	 *
	 * @param {number} num
	 */
	displayNumberAsCurrency(num) {
		try {
			return num.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		catch (err) {
			console.warn(err);
			return 'N/A';
		}
	}
}
