class ShipLogRecord {
	constructor(jsonRecord) {
		/**
		 * @type {string}
		 */
		this.order_id = jsonRecord['order_id'];
		/**
		 * @type {string}
		 */
		this.account_id = jsonRecord['account_id'];
		/**
		 * @type {string}
		 */
		this.account_name = jsonRecord['account_name'];
		this.invoice_date = new Date(jsonRecord['invoice_date'].replace(' ', 'T'));
		/**
		 * @type {string}
		 */
		this.mts_number = jsonRecord['mts_number'];
		/**
		 * @type {string}
		 */
		this.notes = jsonRecord['notes'] || '';
		/**
		 * @type {string}
		 */
		this.sales_id = jsonRecord['sales_id'];
		/**
		 * @type {string}
		 */
		this.salesperson = jsonRecord['salesperson'];
		this.ship_date = new Date(`${jsonRecord['ship_date']}T12:00:00.000Z`);
		this.total = parseFloat(jsonRecord['total']);
		// this.deductions = parseFloat(jsonRecord['deductions']);
		/**
		 * @type {string}
		 */
		this.tracking_numbers = jsonRecord['tracking_numbers'];
		this.products = [
			{
				product_name : jsonRecord['product_name'],
				product_id : jsonRecord['product_id'],
				product_code : jsonRecord['product_code'],
				product_description : jsonRecord['product_description'],
				quantity : parseInt(jsonRecord['quantity']),
				total : parseFloat(jsonRecord['product_total']),
				/**
				 * @type {string}
				 */
				filter : (jsonRecord['product_code']) ? jsonRecord['product_code'] : jsonRecord['product_name'],
				deducted : (jsonRecord['deducted'] == '1') ? true : false,
				visible : true
			}
		];
		this.deductions = this.calculateDeductions();
		this.subtotal = this.total - this.deductions;
		/**
		 * @type {string}
		 */
		this.stateAbbreviation = jsonRecord['stateAbbreviation']// || "UK";

		/**
		 * @type {string}
		 */
		this.stateName = jsonRecord['stateName'];
	}

	getOrderId() {
		return this.order_id;
	}

	getAccountId() {
		return this.account_id;
	}

	getAccountName() {
		return this.account_name;
	}

	getInvoiceDate() {
		return this.invoice_date;
	}

	getMtsNumber() {
		return this.mts_number;
	}

	getNotes() {
		return this.notes;
	}

	getSalesId() {
		return this.sales_id;
	}

	getSalespersonFirstName() {
		return this.salesperson.substring(0, this.salesperson.indexOf(' '));
	}

	getShipDate() {
		return this.ship_date;
	}

	setSubtotal(subtotal) {
		this.subtotal = subtotal;
	}

	getSubtotal() {
		return this.subtotal;
	}

	setDeductions(deductions) {
		this.deductions = deductions;
	}

	getDeductions() {
		return this.deductions;
	}

	getTotal() {
		return this.total;
	}

	getTrackingNumbers() {
		return this.tracking_numbers;
	}

	getProducts() {
		return this.products;
	}

	getNumProducts() {
		return this.products.length;
	}

	getStateAbbreviation() {
		return this.stateAbbreviation;
	}

	getStateName() {
		return this.stateName;
	}

	getProductId(index) {
		try {
			return this.products[index].product_id;
		}
		catch (err) {
			console.log(this);
			console.error(err);
			return '';
		}
	}

	getProductName(index) {
		try {
			return this.products[index].product_name;
		}
		catch (err) {
			console.log(this);
			console.error(err);
			return '';
		}
	}

	getProductCode(index) {
		try {
			return this.products[index].product_code;
		}
		catch (err) {
			console.log(this);
			console.error(err);
			return '';
		}
	}

	getProductDescription(index) {
		try {
			return this.products[index].product_description;
		}
		catch (err) {
			console.log(this);
			console.error(err);
			return '';
		}
	}

	addProduct({product_name, product_id, product_code, product_description, quantity, total, deducted}) {
		const filter = (product_code) ? product_code : product_name;
		const visible = true;
		this.products.push({product_name, product_id, product_code, product_description, quantity, total, filter, deducted, visible});
		this.setDeductions(this.calculateDeductions());
		this.setSubtotal(this.calculateSubtotal());
	}

	/**
	 *
	 * @param {boolean} showSalesperson Whether or not to show the salesperson cell
	 * @param {number} rowNum Which number row this is, for determining if odd or even
	 * @param {boolean} isUnknownDateItem For displaying unknown date options or not
	 */
	getMainRecordRow(showSalesperson=true, rowNum, isUnknownDateItem=false) {
		const firstVisibleProductIndex = this.getFirstVisibleProduct();
		const rowspan = Math.max(this.getVisibleProducts().length, 1);
		const rowClass = (rowNum % 2 == 0) ? 'even' : 'odd';
		const mainRow = document.createElement('tr');
		mainRow.classList.add(rowClass);
		if (showSalesperson) {
			mainRow.appendChild(this.getSalespersonCell(rowspan));
		}
		mainRow.appendChild(this.getMtsNumberCell(rowspan));
		// mainRow.appendChild(this.getStateAbbreviationCell(rowspan));
		mainRow.appendChild(this.getAccountCell(rowspan));
		(isUnknownDateItem) ? mainRow.appendChild(this.getInvoiceDateCell(rowspan)) : mainRow.appendChild(this.getShipDateCell(rowspan));
		mainRow.appendChild(this.getProductNameCell(firstVisibleProductIndex));
		mainRow.appendChild(this.getProductQuantityCell(firstVisibleProductIndex));
		mainRow.appendChild(this.getNotesCell(rowspan));
		mainRow.appendChild(this.getTrackingNumbersCell(rowspan));
		if (!isUnknownDateItem) {
			mainRow.appendChild(this.getRefreshRecordCell(rowspan));
		}
		mainRow.appendChild(this.getSubtotalCell(rowspan));
		if (isUnknownDateItem) {
			mainRow.appendChild(this.getUpdatedShipDateButtonCell(rowspan));
		}
		return mainRow;
	}

	/**
	 *
	 * @param {number} rowNum Row number for determining even/odd
	 * @param {number} index Product index
	 */
	getProductRow(rowNum, index) {
		const rowClass = (rowNum % 2 == 0) ? 'even' : 'odd';
		const row = document.createElement('tr');
		row.classList.add(rowClass);
		row.appendChild(this.getProductNameCell(index));
		row.appendChild(this.getProductQuantityCell(index));
		return row;
	}

	/**
	 *
	 * @param {number} rowspan Number of rows to span
	 */
	getSalespersonCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getSalespersonFirstName();
		return cell;
	}

	/**
	 *
	 * @param {number} rowspan Number of rows to span
	 */
	getMtsNumberCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const link = document.createElement('a');
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Invoices/${this.getOrderId()}`;
		link.target = '_blank';
		link.innerText = this.getMtsNumber();
		cell.appendChild(link);
		return cell;
	}

	getStateAbbreviationCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getStateAbbreviation();
		// const innerSpan = document.createElement('span');
		// cell.appendChild(innerSpan);
		// innerSpan.innertText = this.getStateAbbreviation();
		return cell;
	}

	/**
	 *
	 * @param {number} rowspan Number of rows to span
	 */
	getAccountCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const link = document.createElement('a');
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Accounts/${this.getAccountId()}`;
		link.target = '_blank';
		link.innerText = this.getAccountName();
		cell.appendChild(link);
		return cell;
	}

	getInvoiceDateCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const dateDisplay = document.createElement('p');
		dateDisplay.id = `invoice_date_display_${this.getOrderId()}`;
		dateDisplay.style.margin = '0';
		dateDisplay.innerText = this.getInvoiceDate().toLocaleDateString();
		cell.appendChild(dateDisplay);
		return cell;
	}

	getShipDateCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;

		const shipDateDisplay = document.createElement('p');
		cell.appendChild(shipDateDisplay);
		shipDateDisplay.id = `ship_date_display_${this.getOrderId()}`;
		shipDateDisplay.style.margin = '0';

		const shipDateEditToggle = document.createElement('a');
		shipDateDisplay.appendChild(shipDateEditToggle);
		shipDateEditToggle.href = "javascript:void(0)";
		shipDateEditToggle.setAttribute('order_id', this.getOrderId());
		shipDateEditToggle.setAttribute('onclick', "toggleDisplayEdit(this)");
		shipDateEditToggle.innerText = this.getShipDate().toLocaleDateString();

		const shipDateEdit = document.createElement('p');
		cell.appendChild(shipDateEdit);
		shipDateEdit.style.margin = '0';

		const shipDateEditInput = document.createElement('input');
		shipDateEdit.appendChild(shipDateEditInput);
		shipDateEditInput.type = 'date';
		shipDateEditInput.hidden = true;
		shipDateEditInput.id = `ship_date_edit_${this.getOrderId()}`;
		shipDateEditInput.style.margin = '0';
		shipDateEditInput.value = this.getShipDate().toISOString().substr(0, 10);

		cell.appendChild(getSaveEditsSpan(this.getOrderId()));

		return cell;

		/**
		 *
		 * @param {string} orderId
		 */
		function getSaveEditsSpan(orderId) {
			const span = document.createElement('small');
			span.hidden = true;
			span.id = `ship_date_edit_controls_${orderId}`;

			const cancelLink = document.createElement('a');
			span.appendChild(cancelLink);
			cancelLink.href = "javascript:void(0)";
			cancelLink.id = `ship_date_cancel_${orderId}`;
			cancelLink.setAttribute('order_id', orderId);
			cancelLink.setAttribute('onclick', "toggleDisplayEdit(this)");
			cancelLink.innerText = 'Cancel';

			const separatorSpan = document.createElement('span');
			span.appendChild(separatorSpan);
			separatorSpan.innerText = ' | ';

			const saveLink = document.createElement('a');
			span.appendChild(saveLink);
			saveLink.href = "javascript:void(0)";
			saveLink.id = `ship_date_cancel_${orderId}`;
			saveLink.setAttribute('order_id', orderId);
			saveLink.setAttribute('onclick', "saveShipDate(this)");
			saveLink.innerText = 'Save';



			return span;
		}

	}

	getProductCodeCell(index) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Invoices/${this.getOrderId()}#secDiv_Product_Details`;
		link.target = "_blank";
		link.innerText = (this.getProducts()[index]) ? this.getProductCode(index) : '';
		return cell;
	}

	/**
	 *
	 * @param {number} index
	 */
	getProductNameCell(index) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		const hideSkuLink = this.getHideSkuLink(this.getProductId(index));
		cell.appendChild(hideSkuLink);
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Invoices/${this.getOrderId()}#secDiv_Product_Details`;
		link.target = "_blank";
		link.innerText = (this.getProductCode(index)) ? this.getProductCode(index) : this.getProductName(index);
		return cell;
	}

	getProductQuantityCell(index) {
		const cell = document.createElement('td');
		cell.classList.add('halign-center');
		cell.innerText = (this.getProducts()[index]) ? this.getProducts()[index].quantity : '';
		return cell;
	}

	getNotesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const textarea = document.createElement('textarea');
		cell.appendChild(textarea);
		textarea.rows = Math.max(3, rowspan);
		textarea.classList.add('note-input');
		textarea.setAttribute('invoice_id', this.getOrderId());
		textarea.innerText = this.getNotes();
		return cell;
	}

	getTrackingNumbersCell(rowspan) {
		const trackingNumbers = this.getTrackingNumbers();
		const upsPattern = /^(1Z.{16})$/;
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		try {
			for(const trackingNumber of trackingNumbers.split(' ')) {
				if (upsPattern.test(trackingNumber)) {
					cell.appendChild(getUpsTrackingNumberLink(trackingNumber))
				}
				else {
					cell.appendChild(getTrackingNumberSpan(trackingNumber));
				}
			}
		}
		catch (err) {
			console.log(this);
			console.error(err);
		}
		return cell;

		/**
		 *
		 * @param {string} trackingNumber Tracking number to link to
		 */
		function getUpsTrackingNumberLink(trackingNumber) {
			const link = document.createElement('a');
			link.href = `https://www.ups.com/track?loc=en_US&tracknum=${trackingNumber}`;
			link.target = '_blank';
			link.style.display = 'block';
			link.innerText = trackingNumber;
			return link;
		}

		/**
		 *
		 * @param {string} trackingNumber
		 */
		function getTrackingNumberSpan(trackingNumber) {
			const span = document.createElement('span');
			span.style.display = 'block';
			span.innerText = trackingNumber;
			return span;
		}
	}

	/**
	 *
	 * @param {number} rowspan
	 */
	getRefreshRecordCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		cell.classList.add('updateElement');
		const refreshImg = document.createElement('img');
		cell.appendChild(refreshImg);
		refreshImg.alt = "Refresh Record";
		refreshImg.src = "../../img/refresh.png";
		refreshImg.style.width = '16px';
		refreshImg.style.cursor = 'pointer';
		refreshImg.setAttribute('onclick', `refreshRecord('${this.getOrderId()}')`);
		return cell;
	}

	getSubtotalCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getSubtotal().toLocaleString('en-US', {style: 'currency', currency: 'USD'});
		return cell;
	}

	getUpdatedShipDateButtonCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const button = document.createElement('button');
		cell.appendChild(button);
		button.innerText = 'Get Updated Date';
		button.setAttribute('onclick', `refreshRecord('${this.getOrderId()}')`);
		return cell;
	}

	getHideSkuLink(zohoId) {
		const small = document.createElement('small');
		const span = document.createElement('span');
		small.appendChild(span);
		span.innerText = " | ";
		const link = document.createElement('a');
		small.appendChild(link);
		small.classList.add('updateElement');
		link.href = "javascript:void(0)";
		link.innerText = "(deduct)";
		link.setAttribute('onclick', `addToDeductions('${zohoId}')`);
		return small;
	}

	getCsvRows() {
		const rows = [];
		const newRow = [];
		newRow.push(`"${this.getSalespersonFirstName()}"`);
		newRow.push(`"${this.getMtsNumber()}"`);
		newRow.push(`"${this.getStateAbbreviation()}"`);
		newRow.push(`"${this.getShipDate().toLocaleDateString()}"`);
		newRow.push(`"${this.getSubtotal().toLocaleString('en-US', {style: 'currency', currency: 'USD'})}"`);
		newRow.push(`"${this.getTotal().toLocaleString('en-US', {style: 'currency', currency: 'USD'})}"`);
		rows.push(newRow);
		return rows;
	}

	getFirstVisibleProduct() {
		const products = this.getProducts();
		for (let i = 0; i < products.length; i++) {
			const product = products[i];
			if (product.visible && !product.deducted) {
				return i;
			}
		}
	}

	getVisibleProducts() {
		const products = this.getProducts();
		const visibleIndexes = [];
		for (let i = 0; i < products.length; i++) {
			const product = products[i];
			if (product.visible && !product.deducted) {
				visibleIndexes.push(i);
			}
		}
		return visibleIndexes;
	}

	calculateDeductions() {
		const products = this.getProducts();
		let deductions = 0.0;
		if (products) {
			for (const product of products) {
				if (product.deducted) {
					deductions += parseFloat(product.total);
				}
			}
		}
		return deductions;
	}

	calculateSubtotal() {
		const total = this.getTotal();
		const deductions = this.getDeductions();
		return total - deductions;
	}
}
