class ShipLogRecord {
	constructor(jsonRecord) {
		/**
		 * @type {string}
		 */
		this.order_id = jsonRecord['order_id'];
		/**
		 * @type {string}
		 */
		this.notes = jsonRecord['notes'] || '';
		/**
		 * @type {string}
		 */
		this.sales_channel_identifier = jsonRecord['sales_channel_identifier'];
		/**
		 * @type {string}
		 */
		this.sales_channel_name = jsonRecord['sales_channel_name'];
		this.ship_date = (jsonRecord['ship_date'] instanceof Date) ? jsonRecord['ship_date'] : new Date(`${jsonRecord['ship_date'].replace(' ', 'T')}`);
		/**
		 * @type {string}
		 */
		this.ship_method = jsonRecord['ship_method'] || '';
		this.subtotal = parseFloat(jsonRecord['subtotal']);
		/**
		 * @type {string[]}
		 */
		this.tracking_numbers = (jsonRecord['tracking_numbers']) ? jsonRecord['tracking_numbers'].split(',') : [];
		/**
		 * @type {boolean}
		 */
		this.isReturn = jsonRecord['isReturn'] || false;
		this.orderType = (jsonRecord['isReturn']) ? 'Return' : 'Sale';
		this.products = [];
		// this.products = [{sku : jsonRecord['sku'], quantity : jsonRecord['quantity']}];
	}
	getSalesChannelName() {
		return this.sales_channel_name;
	}

	getOrderIdLink() {
		const channels = {
			'back_market' : `https://www.backmarket.com/bo_merchant/orders?date_type=creation&order=${this.getOrderId()}`,
			'amazon' : `https://sellercentral.amazon.com/orders-v3/order/${this.getOrderId()}`,
			'ebay_others' : `https://www.ebay.com/sh/ord/details?orderid=${this.getOrderId()}`,
			'ebay_auctions' : `https://www.ebay.com/sh/ord/details?orderid=${this.getOrderId()}`,
			'newegg' : `https://sellerportal.newegg.com/manage-order/orderlist`
		}
		return `<a href='${channels[this.sales_channel_identifier]}' target='_blank'>${this.getOrderId()}</a>`;
	}

	getOrderIdAnchor() {
		const channels = {
			'back_market' : `https://www.backmarket.com/bo_merchant/orders?date_type=creation&order=${this.getOrderId()}`,
			'amazon' : `https://sellercentral.amazon.com/orders-v3/order/${this.getOrderId()}`,
			'ebay_others' : `https://www.ebay.com/sh/ord/details?orderid=${this.getOrderId()}`,
			'ebay_auctions' : `https://www.ebay.com/sh/ord/details?orderid=${this.getOrderId()}`,
			'newegg' : `https://sellerportal.newegg.com/manage-order/orderlist`
		}
		const link = document.createElement('a');
		link.href = channels[this.sales_channel_identifier];
		link.innerText = this.getOrderId();
		link.target = '_blank';
		return link;
	}

	getShipDate() {
		return this.ship_date;
	}

	getShipDateString() {
		return this.ship_date.toLocaleDateString();
	}

	getShipMethod() {
		return this.ship_method;
	}

	getDecodedShipMethod() {
		const upsShippingMethodDict = {
			'01' : 'UPS Next Day Air',
			'02' : 'UPS 2nd Day Air',
			'03' : 'UPS Ground',
			'29' : 'UPS Next Day Air',
			'42' : 'UPS Ground'
		};
		if (!this.tracking_numbers) {
			return this.getShipMethod();
		}
		if (this.isUpsTrackingNum(this.tracking_numbers[0])) {
			const serviceCode = this.tracking_numbers[0].substr(8,2);
			return (upsShippingMethodDict[serviceCode]) ? upsShippingMethodDict[serviceCode] : this.getShipMethod();
		}
		else {
			return this.getShipMethod();
		}
	}

	getOrderId() {
		return this.order_id;
	}

	getSubtotal() {
		return this.subtotal;
	}

	addProduct({sku, quantity}) {
		if (sku && quantity) {
			const visible = true;
			this.products.push({sku, quantity, visible});
		}
	}

	addTrackingNumber(trackingNumber) {
		this.tracking_numbers.push(trackingNumber);
	}

	getTrackingNumbers() {
		return this.tracking_numbers;
	}

	getProductSku(index) {
		return (this.products && this.products[index]) ? this.products[index].sku : '';
	}

	getProductQuantity(index) {
		return (this.products && this.products[index]) ? this.products[index].quantity : '';
	}

	getTrackingNumberUrls() {
		let str = '';
		for (const number of this.getTrackingNumbers()) {
			if (this.isUpsTrackingNum(number)) {
				str += `<a style='display: block;' href='https://www.ups.com/track?loc=en_US&tracknum=${number}' target='_blank'>${number}</a>`;
			} else {
				str += `<span style='display: block;'>${number}</span>`;
			}
		}
		return str;
	}

	createRecordRow(rowNum) {
		// for (const product of this.products) {
		// 	if(!product.sku && product.quantity) {
		// 		delete product;
		// 	}
		// }
		const rowspan = Math.max(this.products.length, 1);
		// console.log(this.products, this.products.length);
		const rowClass = (rowNum % 2 == 0) ? 'even' : 'odd';
		const returnClass = (this.isReturn) ? 'return' : '';
		let returnString =  `
			<tr class='${rowClass} ${returnClass}'>
				<td style='padding: 6px;' rowspan='${rowspan}'>${this.getSalesChannelName()}</td>
				<td style='padding: 6px;' rowspan='${rowspan}'>${this.getOrderIdLink()}</td>
				<td style='padding: 6px;' rowpsan='${rowspan}'>${this.getOrderType()}</td>
				<td style='padding: 6px;' rowspan='${rowspan}'>${this.getShipDateString()}</td>
				<td style='padding: 6px;'>${this.getProductSku(0)}</td>
				
				<td style='padding: 6px;' rowspan='${rowspan}'>${this.getDecodedShipMethod()}</td>
				<td style='padding: 6px;' rowspan='${rowspan}'>${this.getTrackingNumberUrls()}</td>
				<td style='padding: 6px;' rowspan='${rowspan}'>${(this.isReturn) ? '-' : ''}$${Math.abs(Math.round(this.subtotal)).toLocaleString()}</td>
			</tr>
		`;
		for (let i=1; i < this.products.length; i++) {
			returnString += `
				<tr class='${rowClass}'>
					<td style='padding: 6px;'>${this.getProductSku(i)}</td>
					<td style='padding: 6px;' class='halign-center'>${this.getProductQuantity(i)}</td>
				</tr>
			`;
		}
		return returnString;
	}

	isUpsTrackingNum(trackingNum) {
		const upsPattern = /^(1Z.{16})$/;
		return upsPattern.test(trackingNum);
	}

	/**
	 *
	 * @param {number} rowspan
	 */
	getOrderTypeCell(rowspan) {
		const cell = document.createElement('td');
		cell.style.padding = '6px';
		cell.rowSpan = rowspan;
		cell.innerText = this.getOrderType();
		return cell;
	}

	getOrderType() {
		return this.orderType;
	}

	getCsvRow() {
		const rows = [];
		if (this.products.length > 0) {
			for (const product of this.products) {
				const newRow = [];
				newRow.push(`"${this.getSalesChannelName()}"`);
				newRow.push(`"${this.getOrderId()}"`);
				newRow.push(`"${this.getOrderType()}"`);
				newRow.push(`"${this.getShipDate().toLocaleDateString()}"`);
				newRow.push(`"${product['sku']}"`);
				newRow.push(`"${product['quantity']}"`);
				newRow.push(`"${this.getSubtotal().toLocaleString('en-US', {style: 'currency', currency: 'USD'})}"`);
				rows.push(newRow);
			}
		}
		else {
			const newRow = [];
			newRow.push(`"${this.getSalesChannelName()}"`);
			newRow.push(`"${this.getOrderId()}"`);
			newRow.push(`"${this.getOrderType()}"`);
			newRow.push(`"${this.getShipDate().toLocaleDateString()}"`);
			newRow.push(`""`);
			newRow.push(`""`);
			newRow.push(`"${this.getSubtotal().toLocaleString('en-US', {style: 'currency', currency: 'USD'})}"`);
			rows.push(newRow);
		}
		return rows;
	}
}
