class ServiceRequest {
	constructor(data) {
		/**
		 * @type {string}
		 */
		this.status = data.status;
		/**
		 * @type {string}
		 */
		this.oNum = data.oNum;
		/**
		 * @type {string}
		 */
		this.sales = data.sales;
		/**
		 * @type {string}
		 */
		this.organization = data.organization;
		/**
		 * @type {Date}
		 */
		this.dateCreation = new Date(`${data.dateCreation}T12:00:00`);
		/**
		 * @type {string}
		 */
		this.warranty = data.warranty;
		/**
		 * @type {string}
		 */
		this.warrantyName = (!data.warrantyId || data.warrantyId == '1') ? 'None' : (data.warrantyId == '2') ? 'Warranty' : (data.warrantyId == '3') ? 'Warranty+' : 'None';
		/**
		 * @type {string}
		 */
		this.notes = data.notes || '';
		/**
		 * @type {string}
		 */
		this.problem = data.problem.trim();
		/**
		 * @type {string}
		 */
		this.customerId = data.customerId;
		/**
		 * @type {string}
		 */
		this.warrantyId = data.warrantyId;
		/**
		 * @type {Salesperson}
		 */
		this.salesRep = new Salesperson(data);
		/**
		 * @type {boolean}
		 */
		this.changed = false;
		/**
		 * @type {HTMLTableRowElement}
		 */
		this.displayRow = null;

		this.dateCompleted = (data.dateCompleted) ?  new Date(`${data.dateCompleted}T12:00:00`) : null;

		this.trackingOut = data.trackingOut;

		this.servicesPerformed = data.servicesPerformed;

		/**
		 * @type {boolean}
		 */
		this.oACAdapter = data.oACAdapter;

		this.originalValues = {
			status : this.status,
			oNum : this.oNum,
			sales : this.sales,
			organization : this.organization,
			dateCreation : this.dateCreation,
			warrantyId : this.warrantyId,
			oACAdapter : this.oACAdapter,
			notes : this.notes,
			problem : this.problem,
			customerId : this.customerId,
			salesRep : this.salesRep,
		};
	}

	/**
	 *
	 * @param {ColumnDefinition[]} columns
	 */
	generateRow() {
		const row = document.createElement('tr');
		row.appendChild(this.getSalesRepCell(false));
		row.appendChild(this.getONumCell());
		row.appendChild(this.getOrganizationCell());
		row.appendChild(this.getDateCreatedCell());
		row.appendChild(this.getProblemCell());
		// row.appendChild(this.getWarrantyCell());
		row.appendChild(this.getCameWithACCheckboxCell());
		row.appendChild(this.getWarrantyCheckboxCell());
		row.appendChild(this.getWarrantyPlusCheckboxCell());
		row.appendChild(this.getNotesCell());
		row.appendChild(this.getStatusCell());
		row.appendChild(this.getActionCell());
		if (this.isChanged()) {
			row.classList.add('changed');
		}
		if (this.getOrderAge() > 60) {
			row.classList.add('oldOrder');
		}
		this.displayRow = row;
		return row;
	}

	generateCompletedRequestRow() {
		const row = document.createElement('tr');
		row.appendChild(this.getSalesRepCell(false));
		row.appendChild(this.getONumCell());
		row.appendChild(this.getOrganizationCell());
		row.appendChild(this.getDateCompletedCell());
		// row.appendChild(this.getNotesCell(true));
		row.appendChild(this.getServicesPerformedCell());
		row.appendChild(this.getTrackingNumberCell());
		this.displayRow = row;
		return row;
	}

	getActionCell() {
		const cell = document.createElement('td');
		const button = document.createElement('button');
		cell.appendChild(button);
		button.innerText = "Save Changes";
		button.setAttribute('oNum', this.getONum());
		button.setAttribute('onclick', 'saveChanges(this)');
		return cell;
	}

	getCustomerId() {
		return this.customerId;
	}

	getCustomerIdLink() {
		const link = document.createElement('a');
		link.href = `${CONTROLLER_BASE_URL}?m=customer_detail&cString=${this.getCustomerId()}`;
		link.target = "_blank";
		link.innerText = this.getOrganization();
		return link;
	}

	getSalesRepCell(generateSelect=true) {
		const cell = document.createElement('td');
		if (generateSelect) {
			cell.appendChild(this.getSalesRepSelect());
		}
		else {
			try {
				cell.innerText = this.salesRep.getFirstName();
			}
			catch (e) {
				// console.error(e);
				cell.innerText = "None Selected";
			}
		}
		return cell;
	}

	getSalesRepSelect() {
		const select = document.createElement('select');
		select.classList.add('borderless');
		const placeholderOption = document.createElement('option');
		placeholderOption.value='';
		placeholderOption.innerText = "None Selected";
		select.appendChild(placeholderOption);
		let found = false;
		for (const salesperson of salespeople) {
			let selected = (salesperson.getId() == this.salesRep.getId());
			select.appendChild(salesperson.getSalespersonOption(selected));
			if (selected) {
				found = true;
			}
		}
		if (!found && this.salesRep.getId()) {
			select.appendChild(this.salesRep.getSalespersonOption(true));
		}
		select.setAttribute('oNum', this.getONum());
		select.setAttribute('onchange', 'setSales(this)');
		return select;
	}

	getONum() {
		return this.oNum;
	}

	getONumLink() {
		const link = document.createElement('a');
		link.href = `${CONTROLLER_BASE_URL}?m=order_detail&oNum=${this.getONum()}`;
		link.innerText = this.getONum();
		link.target = "_blank";
		return link;
	}

	getONumCell() {
		const cell = document.createElement('td');
		cell.appendChild(this.getONumLink());
		return cell;
	}

	getStatus() {
		return this.status;
	}

	/**
	 *
	 * @param {string} newStatus
	 */
	setStatus(newStatus) {
		this.status = newStatus;
	}

	getStatusCell() {
		const cell = document.createElement('td');
		cell.appendChild(this.getStatusSelect());
		return cell;
	}

	getTrackingNumber() {
		return this.trackingOut;
	}

	getTrackingNumberCell() {
		const cell = document.createElement('td');
		if (this.getTrackingNumber()) {
			const trackingNumberLink = document.createElement('a');
			trackingNumberLink.href = `https://www.ups.com/track?loc=en_US&tracknum=${this.getTrackingNumber()}`;
			trackingNumberLink.innerText = this.getTrackingNumber();
			cell.appendChild(trackingNumberLink);
		}
		else {
			cell.innerText = "";
		}
		cell.classList.add('halign-center');
		return cell;
	}

	getOrganization() {
		return this.organization;
	}

	getOrganizationCell() {
		const cell = document.createElement('td');
		cell.appendChild(this.getCustomerIdLink());
		return cell;
	}

	getProblem() {
		return this.problem.trim();
	}

	setProblem(problem) {
		this.problem = problem;
	}

	getProblemCell() {
		const cell = document.createElement('td');
		const textArea = document.createElement('textarea');
		textArea.value = this.getProblem();
		textArea.rows = 6;
		textArea.cols = 40;
		textArea.setAttribute('oNum', this.getONum());
		textArea.setAttribute('oninput', 'setProblem(this)');
		cell.appendChild(textArea);
		// textArea.disabled = true;
		// textArea.classList.add('borderless');
		// textArea.classList.add('pseudoparagraph');

		return cell;
	}

	getNotes() {
		return this.notes.trim();
	}

	setNote(newNote) {
		this.notes = newNote;
	}

	getNotesCell(disabled) {
		const cell = document.createElement('td');
		const textArea = document.createElement('textarea');
		textArea.value = this.getNotes();
		textArea.rows = 6;
		textArea.cols = 40
		cell.appendChild(textArea);
		textArea.setAttribute('oNum', this.getONum());
		textArea.setAttribute('oninput', 'setNotes(this)');
		textArea.disabled = disabled;
		return cell;
	}

	getServicesPerformed() {
		return this.servicesPerformed;
	}

	getServicesPerformedCell() {
		const cell = document.createElement('td');
		// const textArea = document.createElement('textarea');
		// textArea.value = this.getServicesPerformed();
		// textArea.rows = 6;
		// textArea.cols = 40
		const textArea = document.createElement('p');
		textArea.innerText = this.getServicesPerformed();
		cell.appendChild(textArea);
		textArea.disabled = true;
		return cell;
	}

	getWarranty() {
		return this.warranty;
	}

	getWarrantyId() {
		return this.warrantyId;
	}

	getCameWithACAdapter() {
		return this.oACAdapter;
	}

	setCameWithACAdapter(oACAdapter) {
		this.oACAdapter = oACAdapter;
	}

	getCameWithACCheckboxCell() {
		const cell = document.createElement('td');
		const checkbox = document.createElement('input');
		cell.appendChild(checkbox);
		checkbox.type = 'checkbox';
		checkbox.checked = this.getCameWithACAdapter();
		checkbox.setAttribute('oNum', this.getONum());
		checkbox.setAttribute('onchange', 'toggleCameWithACAdapter(this)');
		checkbox.id = `${this.getONum()}_camewithacadaptercheckbox`;
		return cell;
	}

	getWarrantyCheckboxCell() {
		const cell = document.createElement('td');
		const warrantyId = this.getWarrantyId();
		const warrantyCheckbox = document.createElement('input');
		cell.appendChild(warrantyCheckbox);
		warrantyCheckbox.type = 'checkbox';
		warrantyCheckbox.checked = (parseInt(warrantyId) > 1);
		warrantyCheckbox.value = '2';
		warrantyCheckbox.setAttribute('oNum', this.getONum());
		warrantyCheckbox.setAttribute('onchange', 'toggleWarranty(this)');
		warrantyCheckbox.id = `${this.getONum()}_warrantycheckbox`;
		return cell;
	}

	getWarrantyPlusCheckboxCell() {
		const cell = document.createElement('td');
		const warrantyId = this.getWarrantyId();
		const warrantyCheckbox = document.createElement('input');
		cell.appendChild(warrantyCheckbox);
		warrantyCheckbox.type = 'checkbox';
		warrantyCheckbox.checked = (parseInt(warrantyId) > 2);
		warrantyCheckbox.value = '3';
		warrantyCheckbox.setAttribute('oNum', this.getONum());
		warrantyCheckbox.setAttribute('onchange', 'toggleWarrantyPlus(this)');
		warrantyCheckbox.id = `${this.getONum()}_warrantypluscheckbox`;
		return cell;
	}

	getWarrantyCell(generateSelect=true) {
		const cell = document.createElement('td');
		if (generateSelect) {
			cell.appendChild(this.getWarrantySelect());
		}
		else {
			const warrantyId = this.getWarrantyId();
			const types = {
				'1' : 'None',
				'2' : 'Warranty',
				'3' : 'Warranty+'
			}
			cell.innerText = types[warrantyId] || 'None';
		}
		return cell;
	}

	getWarrantySelect() {
		const select = document.createElement('select');
		select.classList.add('borderless');
		for (const type of warrantyTypes) {
			const option = type.getSelectOption(this.getWarrantyId());
			select.appendChild(option);
		}
		select.setAttribute('oNum', this.getONum());
		select.setAttribute('onchange', 'setWarranty(this)');
		return select;
	}

	getDateCreated() {
		return this.dateCreation.toLocaleDateString();
	}

	getDateCreatedCell() {
		const cell = document.createElement('td');
		cell.innerText = this.getDateCreated();
		return cell;
	}

	getDateCompleted() {
		return (this.dateCompleted) ? this.dateCompleted.toLocaleDateString() : "Unknown";
	}

	getDateCompletedCell() {
		const cell = document.createElement('td');
		cell.innerText = this.getDateCompleted();
		return cell;
	}

	getStatusSelect() {
		const statuses = [
			{
				value : 'Service - New',
				display : 'New'
			},
			{
				value : 'Service - Label Sent',
				display : 'Label Sent'
			},
			{
				value : 'Service - Box Sent',
				display : 'Box Sent'
			},
			{
				value : 'Service - Received',
				display : 'Received'
			},
			{
				value : 'Service - Diagnostic Complete',
				display : 'Diagnostic Complete'
			},
			{
				value : 'Service - Approved/Declined',
				display : 'Approved/Declined'
			},
			{
				value : 'Service - Ready To Ship',
				display : 'Ready To Ship'
			},
			{
				value : 'Service - Replacement',
				display : 'Replacement'
			},
		];
		// const statuses = ['Service - New', 'Service - Label Sent', 'Service - Box Sent', 'Service - Received', 'Service - Ready To Ship'];
		const selectElement = document.createElement('select');
		selectElement.classList.add('borderless');
		for (const status of statuses) {
			const optionElement = document.createElement('option');
			optionElement.value = status.value;
			optionElement.innerText = status.display;
			if (status.value == this.getStatus()) {
				optionElement.selected = true;
			}
			selectElement.appendChild(optionElement);
		}
		selectElement.setAttribute('oNum', this.getONum());
		selectElement.setAttribute('onchange', 'setStatus(this)');
		return selectElement;
	}

	getChanges() {
		if (!this.isChanged()) {
			return false;
		}
		const propertiesToEncode = {
			oNum : this.getONum()
		};
		for (const property in this.originalValues) {
			if (typeof this[property] == "string") {
				if (this[property] != this.originalValues[property]) {
					propertiesToEncode[property] = encodeURIComponent(this[property]);
				}
			}
			else if (typeof this[property] == "boolean") {
				if (this[property] != this.originalValues[property]) {
					propertiesToEncode[property] = (this[property]) ? '1' : '0';
				}
			}
			else if (this[property] instanceof Salesperson) {
				/**
				 * @type {Salesperson}
				 */
				const thisSales = this[property];
				/**
				 * @type {Salesperson}
				 */
				const originalSales = this.originalValues[property];
				if (!thisSales.checkSameId(originalSales)) {
					propertiesToEncode['salesRepId'] = thisSales.getId();
				}
			}
		}
		console.log(propertiesToEncode);
		return propertiesToEncode;
	}

	getEncodedChanges() {
		const changes = this.getChanges();
		if (!changes) {
			return false;
		}
		else {
			return JSON.stringify(changes);
		}
	}

	isChanged() {
		return this.changed;
	}

	/**
	 *
	 * @param {boolean} changed
	 */
	setChanged(changed=true) {
		this.changed = changed;
		if (changed) {
			this.displayRow.classList.add('changed');
		}
		else {
			this.displayRow.classList.remove('changed');
		}
	}

	determineIfChanged() {
		let changed = false;
		for (const field in this.originalValues) {
			if (typeof this[field] == 'string') {
				if (this[field] != this.originalValues[field]) {
					return true;
				}
			}
			else if (typeof this[field] == "boolean") {
				if (this[field] != this.originalValues[field]) {
					return true;
				}
			}
			else if (this[field] instanceof Salesperson) {
				/**
				 * @type {Salesperson}
				 */
				const thisSales = this[field];
				/**
				 * @type {Salesperson}
				 */
				const originalSales = this.originalValues[field];
				if (!thisSales.checkSameId(originalSales)) {
					return true;
				}
			}
		}
		return changed;
	}

	getOrderAge() {
		const today = new Date();
		const diff = today - this.dateCreation;
		return Math.round(diff / 1000 / 60 / 60 / 24);
	}

	getDisplayRow() {
		return this.displayRow;
	}
}
