const CONTROLLER_BASE_URL = "admin_controller.php";

const regions = {
	dict : {},
	/**
	 * @type {Region[]}
	 */
	arr : [],
	empty : () => {
		for (const member in regions.dict) {
			/**
			 * @type {Region}
			 */
			const region = regions.dict[member];
			if (region.row) {
				region.row.remove();
			}
			delete regions.dict[member];
		}
		regions.arr.length = 0;
	},
};
const states = {
	dict : {},
	/**
	 * @type {State[]}
	 */
	arr : [],
	empty : () => {
		for (const member in states.dict) {
			/**
			 * @type {State}
			 */
			const state = states.dict[member];
			if (state.row) {
				state.row.remove();
			}
			delete states.dict[member];
		}
		states.arr.length = 0;
	},
};
const salespersons = {
	dict : {},
	/**
	 * @type {Salesperson[]}
	 */
	arr : [],
	empty : () => {
		for (const member in salespersons.dict) {
			delete salespersons.dict[member];
		}
		salespersons.arr.length = 0;
	},
};

init();

async function init() {
	regions.empty();
	states.empty();
	salespersons.empty();
	await getRegions();
	await getStates();

	await getSalespersons();
	generateRegionInputs();
	generateStateInputs();
	setLoadingModalVisible(false);
}

async function generateRegionInputs() {
	/**
	 * @type {HTMLTableSectionElement}
	 */
	const tableBody = document.getElementById("regions_tablebody");
	for(const region of regions.arr) {
		tableBody.appendChild(generateRegionInputRow(region));
	}
}

function generateStateInputs() {
	const tableBody = document.getElementById("states_tablebody");
	for (const state of states.arr) {
		tableBody.appendChild(generateStateInputRow(state));
	}
}

async function getRegions() {
	const regionText = await fetch(`${CONTROLLER_BASE_URL}?m=ajax_getRegions`)
		.then(response => response.text())
		.then(data => {
			return data
		});
	// console.log(regionText);
	const rcvdRegions = JSON.parse(regionText);
	for (const regionData of rcvdRegions) {
		const region = new Region(regionData);
		regions.dict[region.id] = region;
		regions.arr.push(region);
	}
	console.log(regions);
}

async function getStates() {
	const stateText = await fetch(`${CONTROLLER_BASE_URL}?m=ajax_getStates`)
		.then(response => response.text())
		.then(data => {
			return data
		});
	// console.log(stateText);
	const rcvdStates = JSON.parse(stateText);
	console.log(rcvdStates);
	for (const stateData of rcvdStates) {
		const state = new State(stateData);
		state.regionId = stateData['regionid'];
		states.dict[state.id] = state;
		states.arr.push(state);
	}
	console.log(states);
}

async function getSalespersons() {
	const salespersonText = await fetch(`${CONTROLLER_BASE_URL}?m=ajax_getSalespersons`)
		.then(response => response.text())
		.then(data => {
			return data
		});
	// console.log(salespersonText);
	const rcvdSalespersons = JSON.parse(salespersonText);
	for (const salespersonData of rcvdSalespersons) {
		if (salespersonData['name'] == null) {
			continue;
		}
		const salesperson = new Salesperson(salespersonData);
		salespersons.dict[salesperson.id] = salesperson;
		salespersons.arr.push(salesperson);
	}
	console.log(salespersons);
}


/**
 * Generates region input row
 * @param {Region} region Region to generate the row for
 * @returns Row
 */
function generateRegionInputRow(region) {
	if (document.getElementById(`region_${region.id}_input_row`)) {
		/**
		 * @type {HTMLTableRowElement}
		 */
		const foundRow = document.getElementById(`region_${region.id}_input_row`);
		region.row = foundRow;
		return foundRow;
	}

	const row = document.createElement("tr");
	row.id = `region_${region.id}_input_row`;
	const nameCell = document.createElement("td");
	const identifierCell = document.createElement("td");
	const salespersonsCell = document.createElement("td");
	const actionsCell = document.createElement("td");
	const nameInput = document.createElement("input");
	const identifierInput = document.createElement("input");
	const salespersonsInput = document.createElement("select");
	salespersonsInput.classList.add("borderless");

	const option = document.createElement('option');
	option.value = "";
	option.innerText = "Select";
	option.disabled = true;
	salespersonsInput.appendChild(option);

	let hasAssignedSales = false;
	for (const sales of salespersons.arr) {
		let isAssigned = false;
		for (const regSales of region.salespersons) {
			if (sales.id == regSales.id) {
				isAssigned = true;
				hasAssignedSales = true;
			}
		}
		salespersonsInput.appendChild(sales.getSalespersonOption(isAssigned));
	}
	option.selected = !hasAssignedSales;
	const saveButton = document.createElement("button");

	saveButton.id = `region_${region.id}_save`;
	saveButton.innerText = "Save";
	nameInput.id = `region_${region.id}_name`
	identifierInput.id = `region_${region.id}_identifier`
	nameInput.name = `region_${region.id}_name`
	identifierInput.name = `region_${region.id}_identifier`
	nameInput.value = region.name;
	identifierInput.value = region.identifier;
	nameInput.placeholder = "Region Name";
	identifierInput.placeholder = "Region Identifier";

	nameInput.classList.add("borderless");
	nameInput.setAttribute("regionid", region.id);
	nameInput.setAttribute("oninput", "updateRegionName(this)");
	identifierInput.classList.add("borderless");
	identifierInput.setAttribute("regionid", region.id);
	identifierInput.setAttribute("oninput", "updateRegionIdentifier(this)");
	salespersonsInput.setAttribute("regionid", region.id);
	salespersonsInput.setAttribute("onchange", "setSalespersonId(this)");

	row.appendChild(nameCell);
	row.appendChild(identifierCell);
	row.appendChild(salespersonsCell);
	// row.appendChild(actionsCell);
	nameCell.appendChild(nameInput);
	identifierCell.appendChild(identifierInput);
	salespersonsCell.appendChild(salespersonsInput);
	actionsCell.appendChild(saveButton);
	region.row = row;
	return row;
}

/**
 *
 * @param {State} state
 * @returns
 */
function generateStateInputRow(state) {
	if (document.getElementById(`state_${state.id}_input_row`)) {
		/**
		 * @type {HTMLTableRowElement}
		 */
		const foundRow = document.getElementById(`state_${state.id}_input_row`);
		state.row = foundRow;
		return foundRow;
	}

	const row = document.createElement("tr");
	row.id = `state_${state.id}_input_row`;
	const nameCell = document.createElement("td");
	const abbreviationCell = document.createElement("td");
	const regionCell = document.createElement("td");
	const actionsCell = document.createElement("td");
	const nameInput = document.createElement("input");
	nameInput.readOnly = true;
	const abbreviationInput = document.createElement("input");
	abbreviationInput.readOnly = true;
	const regionSelect = document.createElement("select");
	regionSelect.classList.add("borderless");
	regionSelect.setAttribute("onchange", "setStateRegionId(this)");
	for (const region of regions.arr) {
		let isAssigned = region.id == state.regionId;

		regionSelect.appendChild(region.getRegionOption(isAssigned));
	}
	const saveButton = document.createElement("button");

	saveButton.id = `state_${state.id}_save`;
	saveButton.innerText = "Save";
	saveButton.setAttribute("onclick", "saveState(this)");
	nameInput.id = `state_${state.id}_name`
	abbreviationInput.id = `state_${state.id}_abbreviation`
	nameInput.name = `state_${state.id}_name`
	abbreviationInput.name = `state_${state.id}_abbreviation`
	nameInput.value = state.name;
	abbreviationInput.value = state.abbreviation;
	nameInput.placeholder = "State Name";
	abbreviationInput.placeholder = "State Abbreviation";
	regionSelect.id = `state_${state.id}_region`;

	nameInput.classList.add("borderless");
	nameInput.setAttribute("stateid", state.id);
	regionSelect.setAttribute("stateid", state.id);
	abbreviationInput.classList.add("borderless");

	row.appendChild(nameCell);
	row.appendChild(abbreviationCell);
	row.appendChild(regionCell);
	// row.appendChild(actionsCell);
	nameCell.appendChild(nameInput);
	abbreviationCell.appendChild(abbreviationInput);
	regionCell.appendChild(regionSelect);
	actionsCell.appendChild(saveButton);
	state.row = row;
	return row;
}

/**
 *
 * @param {HTMLInputElement} nameInput
 */
function updateRegionName(nameInput) {
	const regionId = nameInput.getAttribute("regionid");
	/**
	 * @type {Region}
	 */
	const region = regions.dict[regionId];
	region.name = nameInput.value;
	region.setChangedClass();
}

/**
 *
 * @param {HTMLInputElement} input
 */
function updateRegionIdentifier(input) {
	const regionId = input.getAttribute("regionid");
	/**
	 * @type {Region}
	 */
	const region = regions.dict[regionId];
	region.identifier = input.value;
	region.setChangedClass();
}

/**
 *
 * @param {HTMLSelectElement} select
 */
function setSalespersonId(select) {
	const regionId = select.getAttribute("regionid");
	const salespersonId = select.value;
	/**
	 * @type {Region}
	 */
	const region = regions.dict[regionId];
	region.salespersonId = salespersonId;
	region.setChangedClass();
}

/**
 *
 * @param {HTMLSelectElement} regionSelect
 */
function setStateRegionId(regionSelect) {
	const stateId = regionSelect.getAttribute("stateid");
	const regionId = regionSelect.value;
	/**
	 * @type {State}
	 */
	const state = states.dict[stateId];
	state.regionId = regionId;
	state.setChangedClass();
}

/**
 *
 * @param {HTMLButtonElement} saveButton
 */
async function saveRegions(saveButton) {
	saveButton.disabled = true;
	const saveData = {
		regions : []
	}
	for (const region of regions.arr) {
		if (region.changed) {
			saveData.regions.push(region.getSaveData());
		}
	}
	console.log(saveData);
	if (saveData.regions.length > 0) {
		const stringified = JSON.stringify(saveData).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");;
		console.log(stringified);
		setLoadingModalVisible(true);
		const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=salesRegions_saveRegions`, {
			method: 'POST',
			headers: {
				'Accept' : 'application/json',
				'Content-Type': 'application/json'
			},
			body: stringified
		})
		.then(res => res.text());
		console.log(responseText);
		init();
		setLoadingModalVisible(false);
	}
	saveButton.disabled = false;
}

/**
 *
 * @param {HTMLButtonElement} saveButton
 */
async function saveStates(saveButton) {
	saveButton.disabled = true;
	const saveData = {
		states : []
	}
	for (const state of states.arr) {
		if (state.changed) {
			saveData.states.push(state.getSaveData());
		}
	}
	console.log(saveData);
	if (saveData.states.length > 0) {
		const stringified = JSON.stringify(saveData).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");;
		console.log(stringified);
		setLoadingModalVisible(true);
		const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=salesRegions_saveStates`, {
			method: 'POST',
			headers: {
				'Accept' : 'application/json',
				'Content-Type': 'application/json'
			},
			body: stringified
		})
		.then(res => res.text());
		console.log(responseText);
		init();
		setLoadingModalVisible(false);
	}
	saveButton.disabled = false;
}

async function saveRegion(regionId) {
	/**
	 * @type {Region}
	 */
	const region = regions.dict[regionId];

}

/**
 *
 * @param {HTMLButtonElement} newRegionButton
 */
function newRegion(newRegionButton) {
	newRegionButton.disabled = true;
	const newRegionData = {
		"id" : "new",
		"name" : "",
		"identifier" : ""
	}
	const newRegion = new Region(newRegionData);
	regions.dict["new"] = newRegion;
	regions.arr.push(newRegion);
	generateRegionInputs();
}
