CONTROLLER_BASE_URL = "admin_controller.php";

const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';
const DEFAULT_SORT = 'date';

const COLUMN_DEFINITIONS = {
	poNumber : new ColumnDefinition('PO Number', 'poNumber', {isSortable: true, isFilterable: false}),
	date : new ColumnDefinition('Invoice Date', 'date', {isSortable: true}),
	vendorName: new ColumnDefinition('Vendor Name', 'vendorName', {isSortable: true}),
	sku : new ColumnDefinition('SKU', 'sku'),
	quantity : new ColumnDefinition('Quantity', 'quantity'),
	description : new ColumnDefinition('Description', 'description'),
	notes : new ColumnDefinition('Notes', 'notes'),
	subtotal : new ColumnDefinition('Subtotal', 'subtotal', {isSortable: true})
};

const columns = {
	/**
	 * @type {ColumnDefinition[]}
	 */
	viewAll : [
		COLUMN_DEFINITIONS.poNumber,
		COLUMN_DEFINITIONS.date,
		COLUMN_DEFINITIONS.vendorName,
		COLUMN_DEFINITIONS.sku,
		COLUMN_DEFINITIONS.quantity,
		COLUMN_DEFINITIONS.description,
		COLUMN_DEFINITIONS.notes,
		COLUMN_DEFINITIONS.subtotal,
	],
	/**
	 * @type {ColumnDefinition[]}
	 */
	default : [
		COLUMN_DEFINITIONS.poNumber,
		COLUMN_DEFINITIONS.date,
		COLUMN_DEFINITIONS.sku,
		COLUMN_DEFINITIONS.quantity,
		COLUMN_DEFINITIONS.description,
		COLUMN_DEFINITIONS.notes,
	]
}

const PAGE_ELEMENTS = {
	/**
	 * @type {HTMLTableRowElement}
	 */
	headerRow : document.getElementById('inventory_header_row'),
	/**
	 * @type {HTMLTableSectionElement}
	 */
	tableBody : document.getElementById('inventory_body'),
	/**
	 * @type {HTMLSpanElement}
	 */
	totalNumOrdersSpan : document.getElementById('total_orders_header_number'),
	/**
	 * @type {HTMLSpanElement}
	 */
	totalOrderValueSpan : document.getElementById('total_value_header_number'),
	/**
	 * @type {HTMLSpanElement}
	 */
	totalOrdersContainer : document.getElementById('total_orders_span'),
	/**
	 * @type {HTMLCollectionOf<HTMLImageElement>}
	 */
	sortIcons : document.getElementsByClassName('sort_icon')
}

const RECORD_DICT = {};
/**
 * @type {PurchaseOrder[]}
 */
const RECORD_ARRAY = [];

let hasPermissions = false;
let totalNumOrders = 0;
let totalOrderValue = 0;

const filter = {};
const sort = {
	by: DEFAULT_SORT,
	order: 1,
};

init();

async function init() {
	emptyArray();
	emptyDict();
	hasPermissions = await fetchPermissions();
	const records = await getRecordsFromDb();
	for (const record of records) {
		const order = new PurchaseOrder(record, hasPermissions);
		RECORD_DICT[order.getZohoId()] = order;
		RECORD_ARRAY.push(order);
	}
	setLoadingModalVisible(false);
	renderHeaders();
	renderBody();


	function emptyArray() {
		RECORD_ARRAY.length = 0;
	}
	function emptyDict() {
		for(const name in RECORD_DICT) {
			delete RECORD_DICT[name];
		}
	}

	async function fetchPermissions() {
		const response = await fetch(`${CONTROLLER_BASE_URL}?m=purchaseOrders_getPermissions`)
			.then(res => { return res });
		try {
			const data = response.json();
			return data;
		}
		catch (err) {
			console.error(err);
			const data = response.text();
			return data;
		}
	}
}

async function getRecordsFromDb() {
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=purchaseOrders_getRecords`)
		.then(res => { return res });
	try {
		const data = response.json();
		return data;
	}
	catch (err) {
		console.error(err);
		const data = response.text();
		return data;
	}
}


function renderHeaders() {
	if (RECORD_ARRAY.length > 0) {
		renderHeaderRow();
	}

	function renderHeaderRow() {
		/**
		 * @type {HTMLTableRowElement}
		 */
		const headerRow = PAGE_ELEMENTS.headerRow;

		headerRow.innerHTML = '';
		if (hasPermissions) {
			for (const column of columns.viewAll) {
				// headerRow.innerHTML += column.buildHeader();
				headerRow.appendChild(column.getHeaderCellElement());
			}
		}
		else {
			for (const column of columns.default) {
				// headerRow.innerHTML += column.buildHeader();
				headerRow.appendChild(column.getHeaderCellElement());
			}
		}
		headerRow.appendChild(generateActionHeader());


		function generateActionHeader() {
			const headerCell = document.createElement('th');
			const button = document.createElement('button');
			headerCell.appendChild(button);
			button.innerText = "Save All Changes";
			button.setAttribute('onclick', 'saveAllChanges(this)');
			return headerCell;
		}
	}
}

function renderBody() {
	const tableBody = PAGE_ELEMENTS.tableBody;
	PAGE_ELEMENTS.totalOrdersContainer.hidden = true;
	tableBody.innerHTML = '';
	let numRendered = 0;
	totalOrderValue = 0;
	for (const order of RECORD_ARRAY) {
		if (order.getIsValid()) {
			totalOrderValue += order.getSubtotal();
			const newRows = order.generateRows(hasPermissions);
			const evenOrOdd = (numRendered % 2 == 0) ? 'even' : 'odd';
			for (const row of newRows) {
				row.classList.add(evenOrOdd);
				tableBody.appendChild(row);
			}
			numRendered++;
		}
	}
	totalNumOrders = numRendered;
	if (hasPermissions) {
		PAGE_ELEMENTS.totalNumOrdersSpan.innerText = totalNumOrders.toLocaleString();
		PAGE_ELEMENTS.totalOrderValueSpan.innerText = totalOrderValue.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		PAGE_ELEMENTS.totalOrdersContainer.hidden = false;
	}
}

/**
 *
 * @param {HTMLTextAreaElement} noteInput
 */
function updateNotes(noteInput) {
	const quoteZohoId = noteInput.getAttribute('quoteZohoId');
	/**
	 * @type {PurchaseOrder}
	 */
	const purchaseOrder = RECORD_DICT[quoteZohoId];
	const newNotes = noteInput.value.trim();
	purchaseOrder.setNotes(newNotes);
	purchaseOrder.setIsChanged(true);
}

async function zohoRefreshPurchaseOrders() {
	setLoadingModalVisible(true);
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=purchaseOrders_get_inbound`)
		.then(response => response.text())
		.then(data => { return data }
	);
	init();
}

async function saveAllChanges(_input) {
	for (const purchaseOrder of RECORD_ARRAY) {
		if (purchaseOrder.getIsChanged()) {
			setLoadingModalVisible(true);
			saveChanges(purchaseOrder.getZohoId());
			setLoadingModalVisible(false);
		}
	}
}

/**
 *
 * @param {string} zohoId
 */
async function saveChanges(zohoId) {

	/**
	 * @type {PurchaseOrder}
	 */
	const purchaseOrder = RECORD_DICT[zohoId];
	if (purchaseOrder.getIsChanged()) {
		setLoadingModalVisible(true);

		const dataToSave = purchaseOrder.getSaveableFieldsAsObject();
		console.log(dataToSave);
		const stringified = JSON.stringify(dataToSave).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");
		console.log(stringified);
		const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=purchaseOrders_save_changes`, {
			method: 'POST',
			headers: {
				'Accept' : 'application/json',
				'Content-Type': 'application/json'
			},
			body: stringified
		});
		const content = await rawResponse.text();

		console.log(content);
		setLoadingModalVisible(false);
	}
}


/* Filter/sort functions below */

/**
 *
 * @param {HTMLInputElement} input
 */
function filterInput(input) {
	filter[input.getAttribute('filter')] = input.value;
	const sameNameInputs = document.getElementsByClassName(input.getAttribute('filter') + '_filter');
	for(const elem of sameNameInputs) {
		elem.value = input.value;
	}
	renderBody(true);
}

function clearFilter() {
	for (const entry in filter) {
		delete filter[entry];
	}
}

function getValidEntries() {
	/**
	 * @type {PurchaseOrder[]}
	 */
	const validEntries = [];
	validEntries.length = 0;
	for (const quote of RECORD_ARRAY) {
		for (const lineItem of quote.getLineItems()) {
			lineItem.setIsVisible(true);
		}
		let valid = true;
		for (const entry in filter) {
			const currentFilter = filter[entry];
			if (filter[entry] == '') {
				delete filter[entry];
			}
			// else if (entry == 'products') {
			// 	valid = validateProducts(quote, currentFilter);
			// }
			else if (quote[entry] && currentFilter && !quote[entry].toString().toLowerCase().includes(currentFilter.toLowerCase())) {
				valid = false;
			}
		}
		quote.setIsValid(valid);
	}

	/**
	 * Filter valid products
	 * @param {PurchaseOrder} record Record to check
	 * @param {string} filterValue Value to filter on
	 */
	function validateProducts(record, filterValue) {
		let valid = false;
		for (const product of record.lineItems) {
			if (product.filter.toLowerCase().includes(filterValue.toLowerCase())) {
				product.setIsVisible(true);
				valid = true;
			}
			else {
				product.setIsVisible(false);
			}
		}
		return valid;
	}
}

/* Sort functions below */

function setSort(newSort) {
	if (newSort == sort.by) {
		sort.order *= -1;
	} else {
		sort.by = newSort;
	}
	sortRecords();
	renderBody(false);

	function sortRecords() {
		RECORD_ARRAY.sort((a,b) => sortFunction(a,b));
		for (const icon of PAGE_ELEMENTS.sortIcons) {
			if (icon.parentElement.getAttribute('sort_order') != sort.by) {
				icon.src = SORT_NONE_URL;
			} else {
				if (sort.order == 1) {
					icon.src = SORT_ASC_URL;
				} else if (sort.order == -1) {
					icon.src = SORT_DESC_URL;
				}
			}
		}
		function sortFunction(a,b) {
			if(isNaN(a[sort.by]) || sort.by == 'date') {
				if(a[sort.by] < b[sort.by]) {
					return -1 * sort.order;
				}
				if(a[sort.by] > b[sort.by]) {
					return 1 * sort.order;
				}
			} else {
				if(parseInt(a[sort.by]) < parseInt(b[sort.by])) {
					return -1 * sort.order;
				}
				if(parseInt(a[sort.by]) > parseInt(b[sort.by])) {
					return 1 * sort.order;
				}
			}
		}
	}
}
