class PurchaseOrder {
	constructor(data, hasPermissions=false) {
		this.id = data['id'];
		/**
		 * @type {string}
		 */
		this.poNumber = data['poNumber'];
		/**
		 * @type {string}
		 */
		this.subject = data['subject'];
		if (hasPermissions) {
			this.subtotal = parseFloat(data['subtotal']) || 0.0;
		}
		else {
			this.subtotal = 0.0;
		}
		/**
		 * @type {string}
		 */
		this.zohoId = data['zohoId'];
		/**
		 * @type {string}
		 */
		this.notes = data['notes'];
		/**
		 * @type {string}
		 */
		const dateStr = data['date'];
		this.date = new Date(dateStr.replace(' ', 'T'));
		if (hasPermissions) {
			this.vendor = new Vendor(data['vendor']);
		}
		else {
			this.vendor = new Vendor({id: data['vendor']['id'], zohoId: 'noPermissions', name: 'noPermissions'});
		}
		this.lineItems = [];
		for (const lineItemData of data['lineItems']) {
			const newLineItem = new LineItem(lineItemData, hasPermissions);
			this.lineItems.push(newLineItem);
		}

		this.vendorName = this.getVendor().getName();

		this.isChanged = false;
		this.isValid = true;
	}

	getId() {
		return this.id;
	}

	getZohoId() {
		return this.zohoId;
	}

	getPoNumber() {
		return this.poNumber;
	}

	getSubject() {
		return this.subject;
	}

	getSubtotal() {
		return this.subtotal;
	}

	getLineItems() {
		return this.lineItems;
	}

	getDate() {
		return this.date;
	}

	getVendor() {
		return this.vendor;
	}

	getNotes() {
		return this.notes;
	}

	setNotes(notes) {
		this.notes = notes;
	}

	getIsChanged() {
		return this.isChanged;
	}

	/**
	 *
	 * @param {boolean} isChanged
	 */
	setIsChanged(isChanged) {
		this.isChanged = isChanged;
	}

	getIsValid() {
		return this.isValid;
	}

	/**
	 *
	 * @param {boolean} isValid
	 */
	setIsValid(isValid) {
		this.isValid = isValid;
	}

	getPoNumberCell(rowspan, renderLink=false) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		if (renderLink) {
			const link = document.createElement('a');
			cell.appendChild(link);
			link.target = "_blank";
			link.innerText = this.getPoNumber();
			link.href = `https://crm.zoho.com/crm/org26161413/tab/PurchaseOrders/${this.getZohoId()}`
		}
		else {
			cell.innerText = this.getPoNumber();
		}

		return cell;
	}

	getInvoiceDateCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		const dateString = this.getDate().toLocaleDateString();
		cell.innerText = dateString;
		return cell;
	}

	getVendorCell(rowspan) {
		return this.getVendor().getVendorCell(rowspan);
	}

	getSkuCell(lineItemIndex, renderLink) {
		const lineItem = this.getLineItems()[lineItemIndex];
		if (renderLink) {
			if (lineItem.getCode()) {
				return lineItem.getCodeCell(renderLink)
			}
			else {
				return lineItem.getNameCell(renderLink);
			}
		}
		else {
			if (lineItem.getCode()) {
				return lineItem.getCodeCell()
			}
			else {
				return lineItem.getNameCell();
			}
		}
	}

	getDescriptionCell(lineItemIndex) {
		const lineItem = this.getLineItems()[lineItemIndex];
		if (lineItem.getDescription()) {
			return lineItem.getDescriptionCell()
		}
		else {
			return lineItem.getNameCell();
		}
	}

	getQuantityCell(lineItemIndex) {
		const lineItem = this.getLineItems()[lineItemIndex];
		return lineItem.getQuantityCell();
	}

	getNotesCell(rowspan=1) {
		const cell = document.createElement('td');
		const textArea = document.createElement('textarea');
		textArea.rows = 5;
		cell.appendChild(textArea);
		cell.rowSpan = rowspan;
		textArea.innerText = this.getNotes();
		textArea.value = this.getNotes();
		textArea.setAttribute('quoteZohoId', this.getZohoId());
		textArea.setAttribute('oninput', 'updateNotes(this)');
		return cell;
	}

	getSubtotalCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getSubtotal().toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0})
		return cell;
	}

	getSaveChangesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const saveButton = document.createElement('button');
		cell.appendChild(saveButton);
		saveButton.innerText = "Save Changes";
		saveButton.setAttribute('onclick', `saveChanges('${this.getZohoId()}')`);
		return cell;
	}

	generateRows(hasPermissions=false) {
		const rows = [];
		rows.push(this.generateMainRow(hasPermissions));
		const visibleLineItems = this.getVisibleLineItems();
		for(let i = 1; i < visibleLineItems.length; i++) {
			rows.push(this.generateProductRow(i, hasPermissions));
		}
		return rows;
	}

	generateMainRow(hasPermissions=false) {
		const rowspan = Math.max(this.getVisibleLineItems().length, 1);
		const firstVisibleProductIndex = this.getFirstVisibleLineItem();

		const row = document.createElement('tr');
		row.appendChild(this.getPoNumberCell(rowspan, hasPermissions));
		row.appendChild(this.getInvoiceDateCell(rowspan));
		if (hasPermissions) {
			row.appendChild(this.getVendorCell(rowspan));
		}

		row.appendChild(this.getSkuCell(firstVisibleProductIndex, hasPermissions));
		row.appendChild(this.getQuantityCell(firstVisibleProductIndex));
		row.appendChild(this.getDescriptionCell(firstVisibleProductIndex));

		row.appendChild(this.getNotesCell(rowspan));
		if (hasPermissions) {
			row.appendChild(this.getSubtotalCell(rowspan));
		}

		row.appendChild(this.getSaveChangesCell(rowspan));

		return row;
	}

	generateProductRow(index, hasPermissions=false) {
		const row = document.createElement('tr');
		row.appendChild(this.getSkuCell(index, hasPermissions));
		row.appendChild(this.getQuantityCell(index));
		row.appendChild(this.getDescriptionCell(index));
		return row;
	}

	getFirstVisibleLineItem() {
		const lineItems = this.getLineItems();
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (lineItem.getIsVisible()) {
				return i;
			}
		}
	}

	getVisibleLineItems() {
		const lineItems = this.getLineItems();
		const visibleIndexes = [];
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (lineItem.getIsVisible()) {
				visibleIndexes.push(i);
			}
		}
		return visibleIndexes;
	}

	getSaveableFieldsAsObject() {
		const toSave = {
			dbId : this.getId(),
			zohoId : this.getZohoId(),
			fields: {
				notes : this.getNotes()
			}
		};
		return toSave;
	}
}
