class EbayOrder {
	constructor(orderData) {
		const shipData = orderData['fulfillmentStartInstructions'][0]['shippingStep']['shipTo'];
		/**
		 * @type {string}
		 */
		this.order_id = orderData['orderId'];
		this.price = parseFloat(orderData['pricingSummary']['priceSubtotal']['value']);
		/**
		 * @type {EbayLineItem[]}
		 */
		this.orderlines = [];
		for (const lineItem of orderData['lineItems']) {
			this.orderlines.push(new EbayLineItem(lineItem));
		}
		this.date_creation = orderData['creationDate'];
		/**
		 * @type {string}
		 */
		this.recipient_name = shipData['fullName'];
		/**
		 * @type {string}
		 */
		this.legacy_order_id = orderData['legacyOrderId'];
		this.shipping_address = {
			/**
			 * @type {string}
			 */
			street : shipData['contactAddress']['addressLine1'],
			/**
			 * @type {string}
			 */
			street2 : shipData['contactAddress']['addressLine2'],
			/**
			 * @type {string}
			 */
			city : shipData['contactAddress']['city'],
			/**
			 * @type {string}
			 */
			state : shipData['contactAddress']['stateOrProvince'],
			/**
			 * @type {string}
			 */
			zip : shipData['contactAddress']['postalCode']
		};
		/**
		 * @type {string}
		 */
		this.email = shipData['email'];
		/**
		 * @type {string}
		 */
		this.phone = shipData['primaryPhone']['phoneNumber'];
		/**
		 * @type {string}
		 */
		this.shipper = orderData['fulfillmentStartInstructions'][0]['shippingStep']['shippingServiceCode'];
		/**
		 * @type {string}
		 */
		this.status_name = orderData['orderFulfillmentStatus'];
		this.channel = {
			identifier : orderData['channel_identifier'],
			name : orderData['channel_name']
		}
		this.errorMsg = orderData['error_msg'] || null;
		this.labelsPrinted = orderData['labels_printed'] || null;
		this.url = `https://www.ebay.com/sh/ord/details?orderid=${orderData['orderId']}`;
	}

	getStatus() {
		return (this.status_name == 'NOT_STARTED') ? 'Awaiting Shipment' : this.status_name;
	}

	getNormalizedShipper() {
		const upsShippingMethodDict = {
			'UPSNextDayAir' : 'UPS Next Day Air',
			'UPS2ndDayAir' : 'UPS 2nd Day Air',
			'UPSGround' : 'UPS Ground'
		};
		if (this.shipper.startsWith('UPS')) {
			return (upsShippingMethodDict[this.shipper]) ? upsShippingMethodDict[this.shipper] : this.shipper;
		}
		else {
			return this.shipper;
		}
	}

	getTableRow() {
		const date = new Date(Date.parse(this.date_creation)).toLocaleString();
		let returnString =
			`<tr ${(this.errorMsg) ? `class="ship_error"` : ''}>
				<td class='halign-center' rowspan='${this.orderlines.length}'><input channel='${this.channel.identifier}' order_id='${this.order_id}' class='${this.channel.identifier}_select_checkbox' id='checkbox_${this.order_id}' type='checkbox' onchange='updateSelected(this)' /></td>
				<td rowspan='${this.orderlines.length}'><a href='${this.url}' target='_blank'>${this.order_id}</a></td>
				<td rowspan='${this.orderlines.length}'>${this.recipient_name}</td>
				<td rowspan='${this.orderlines.length}'>${date}</td>
				<td>${this.orderlines[0].getTitleCellContents()}</td>
				<td class='halign-center'>${this.orderlines[0]['quantity']}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>$${this.price.toLocaleString()}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>${this.getNormalizedShipper()}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'><p>${this.getStatus()}</p>${(this.errorMsg) ? `<p>${this.errorMsg}</p>` : ''}${(this.labelsPrinted && this.labelsPrinted != '0') ? `<p>Labels Printed</p>` : ''}</td>
				<td rowspan='${this.orderlines.length}' class="halign-center">
					${this.getActionButtons()}
				</td>
			</tr>
			`;
		for (let i=1; i < this.orderlines.length; i++) {
			returnString += `
				<tr>
					<td>${this.orderlines[i].getTitleCellContents()}</td>
					<td class='halign-center'>$${this.orderlines[i].quantity}</td>
				</tr>
			`;
		}
		return returnString;
	}

	getActionButtons() {
		let actionButtons = `
			<p style="margin-top:6px;">
				<a href="packingslips/${this.order_id}_ebay_packingslip.pdf" target="_blank" class="action_button">View Packing Slip</a>
			</p>
			<p style="margin-top:6px;">
				<a href="${this.getWorldshipUrl()}" target="_blank" class="action_button">Create Shipping Label</a>
			</p>
			`;

		if (this.labelsPrinted) {
			actionButtons += `
				<p style="margin-top:6px;">
					<button type='button' id='reprint_label_button_${this.order_id}' class='action_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="reprintLabel(this)">Reprint Labels</button>
				</p>
			`;
		}
		return actionButtons;
	}

	getWorldshipUrl() {
		const shipper = (this.shipper.startsWith("UPS")) ? this.shipper.substr(3).trim() : this.shipper;
		const ref1 = (this.channel.name == 'eBay Other') ? 'eBay' : this.channel.name;
		const urlElements = {
			company : encodeURI(this.recipient_name),
			address1 : encodeURI(sanitizeString(this.shipping_address.street)),
			address2 : encodeURI(sanitizeString(this.shipping_address.street2)),
			city : encodeURI(sanitizeString(this.shipping_address.city)),
			state : encodeURI(sanitizeString(this.shipping_address.state)),
			zip : encodeURI(sanitizeString(this.shipping_address.zip)),
			phone : encodeURI(sanitizeString(this.phone)),
			email : encodeURI(sanitizeString(this.email)),
			ref1 : encodeURI(sanitizeString(ref1)),
			ref2 : encodeURI(sanitizeString(this.order_id)),
			listing_id : this.orderlines[0].legacy_id,
			channel : encodeURI(sanitizeString(this.channel.identifier)),
			shipper : encodeURI(shipper),
			origin : 'open_orders'
		}
		let url = `${CONTROLLER_BASE_URL}?m=worldship`;
		for (const element in urlElements) {
			if (urlElements[element] && urlElements[element] != 'null' && urlElements[element] != 'undefined') {
				url += `&${element}=${urlElements[element]}`
			}
		}
		return url;

		/**
		 * Returns URL-safe string
		 * @param {string} str String to sanitize
		 */
		function sanitizeString(str) {
			if (str) {
				str = str.replace('#', 'TTS_NUM');
				str = str.replace('&', 'TTS_AMP');
				str = str.replace('?', 'TTS_QST');
			}
			return str;
		}
	}

	getPackingSlipUrl() {
		const legacyOrderIdElements = this.legacy_order_id.split('-');
		const transactionId = legacyOrderIdElements[1];
		const itemId = legacyOrderIdElements[0];
		return `https://k2b-print.ebay.com/ws/eBayISAPI.dll?BulkSalesRecordPrint&smPrint=Continue+&src=PrintHub&MfcISAPICommand=BulkSalesRecordPrint&printTemplate=2&urlstack=5508%7C%7C&transactionid=${transactionId}&itemid=${itemId}&&userid=sourceplusglobal`;
	}
}
