class BackMarketOrder {
	constructor(orderData) {
		const shipData = orderData['shipping_address'];
		/**
		 * @type {Number}
		 */
		const orderId = orderData['order_id'];
		/**
		 * @type {string}
		 */
		this.order_id = orderId.toString();
		this.price = parseFloat(orderData['price']);
		/**
		 * @type {BackMarketLineItem[]}
		 */
		this.orderlines = [];
		for (const lineItem of orderData['orderlines']) {
			this.orderlines.push(new BackMarketLineItem(lineItem));
		}
		this.date_creation = orderData['date_creation'];
		/**
		 * @type {string}
		 */
		this.recipient_name = `${shipData['first_name']} ${shipData['last_name']}`;
		this.shipping_address = {
			/**
			 * @type {string}
			 */
			street : shipData['street'],
			/**
			 * @type {string}
			 */
			street2 : shipData['street2'],
			/**
			 * @type {string}
			 */
			city : shipData['city'],
			/**
			 * @type {string}
			 */
			state : shipData['state_or_province'],
			/**
			 * @type {string}
			 */
			zip : shipData['postal_code']
		};
		/**
		 * @type {string}
		 */
		this.email = shipData['email'];
		/**
		 * @type {string}
		 */
		this.phone = shipData['phone'];
		/**
		 * @type {string}
		 */
		this.shipper = orderData['shipper'];
		/**
		 * @type {string}
		 */
		this.status_name = orderData['status_name'];
		this.delivery_note = orderData['delivery_note'];
		this.state = orderData['state'];
		this.channel = {
			identifier : orderData['channel_identifier'],
			name : orderData['channel_name']
		}
		this.errorMsg = orderData['error_msg'] || null;
		this.labelsPrinted = parseInt(orderData['labels_printed']) || null;
		this.url = `https://www.backmarket.com/bo_merchant/orders`;
	}

	getTableRow() {
		const date = new Date(Date.parse(this.date_creation)).toLocaleString();
		let returnString =
			`<tr ${(this.errorMsg) ? `class="ship_error"` : ''}>
				<td class='halign-center' rowspan='${this.orderlines.length}'><input channel='${this.channel.identifier}' order_id='${this.order_id}' class='${this.channel.identifier}_select_checkbox' id='checkbox_${this.order_id}' type='checkbox' onchange='updateSelected(this)' /></td>
				<td rowspan='${this.orderlines.length}'><a href='${this.url}' target='_blank'>${this.order_id}</a></td>
				<td rowspan='${this.orderlines.length}'>${this.recipient_name}</td>
				<td rowspan='${this.orderlines.length}'>${date}</td>
				<td>${this.orderlines[0].getTitleCellContents()}</td>
				<td class='halign-center'>${this.orderlines[0]['quantity']}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>$${this.price.toLocaleString()}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>${this.shipper}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'><p>${this.status_name}</p>${(this.errorMsg) ? `<p>${this.errorMsg}</p>` : ''}${(this.labelsPrinted && this.labelsPrinted != '0') ? `<p>Labels Printed</p>` : ''}</td>
				<td rowspan='${this.orderlines.length}' class="halign-center">
					${this.getActionButtons()}
				</td>
			</tr>
			`;
		for (let i=1; i < this.orderlines.length; i++) {
			returnString += `
				<tr>
					<td>${this.orderlines[i].getTitleCellContents()}</td>
					<td class='halign-center'>$${this.orderlines[i].quantity}</td>
				</tr>
			`;
		}
		return returnString;
	}

	getActionButtons() {
		let actionButtons = '';
		if (this.state == 0) {
			actionButtons += `
				<p style="margin-top:6px;">
					<button type='button' id='action_button_1_${this.order_id}' class='action_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="validateOrder(this)">Confirm Payment Validation</button>
				</p>
				<p style="margin-top:6px;">
					<button type='button' id='cancel_button_${this.order_id}' class='action_button cancel_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="cancelOrder(this)">Cancel Order</button>
				</p>
				`;
		}
		else if (this.state == 1) {
			actionButtons += `
				<p style="margin-top:6px;">
					<button type='button' id='action_button_1_${this.order_id}' class='action_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="validateOrder(this)">Validate Order</button>
				</p>
				<p style="margin-top:6px;">
					<button type='button' id='cancel_button_${this.order_id}' class='action_button cancel_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="cancelOrder(this)">Cancel Order</button>
				</p>
				`;
		}
		else if (this.state == 3) {
			actionButtons += `
				<p style="margin-top:6px;">
					<a href="packingslips/${this.order_id}_backmarket_packingslip.pdf" target="_blank" class="action_button">View Packing Slip</a>
				</p>
				<p style="margin-top:6px;">
					<a href="${this.getWorldshipUrl()}" target="_blank" class="action_button">Create Shipping Label</a>
				</p>
				<p style="margin-top:6px;">
					<button type='button' id='cancel_button_${this.order_id}' class='action_button cancel_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="cancelOrder(this)">Cancel Order</button>
				</p>
				`;
		}
		if (this.labelsPrinted) {
			actionButtons += `
				<p style="margin-top:6px;">
					<button type='button' id='reprint_label_button_${this.order_id}' class='action_button' order_id='${this.order_id}' channel='${this.channel.identifier}' onclick="reprintLabel(this)">Reprint Labels</button>
				</p>
			`;
		}
		return actionButtons;
	}

	getWorldshipUrl() {
		const shipper = (this.shipper.startsWith("UPS")) ? this.shipper.substr(3).trim() : this.shipper;
		const urlElements = {
			company : encodeURI(this.recipient_name),
			address1 : encodeURI(sanitizeString(this.shipping_address.street)),
			address2 : encodeURI(sanitizeString(this.shipping_address.street2)),
			city : encodeURI(sanitizeString(this.shipping_address.city)),
			state : encodeURI(sanitizeString(this.shipping_address.state)),
			zip : encodeURI(sanitizeString(this.shipping_address.zip)),
			phone : encodeURI(sanitizeString(this.phone)),
			email : encodeURI(sanitizeString(this.email)),
			ref1 : encodeURI(sanitizeString(this.channel.name)),
			ref2 : encodeURI(sanitizeString(this.order_id)),
			listing_id : this.orderlines[0].legacy_id,
			channel : encodeURI(sanitizeString(this.channel.identifier)),
			shipper : encodeURI(shipper),
			origin : 'open_orders'
		}
		let url = `${CONTROLLER_BASE_URL}?m=worldship`;
		for (const element in urlElements) {
			if (urlElements[element] && urlElements[element] != 'null') {
				url += `&${element}=${urlElements[element]}`
			}
		}
		return url;

		/**
		 * Returns URL-safe string
		 * @param {string} str String to sanitize
		 */
		function sanitizeString(str) {
			if (str) {
				str = str.replace('#', 'TTS_NUM');
				str = str.replace('&', 'TTS_AMP');
				str = str.replace('?', 'TTS_QST');
			}
			return str;
		}
	}
}
