class AmazonOrder {
	constructor(orderData) {
		/**
		 * @type {string}
		 */
		this.order_id = orderData['order_id'];
		this.price = parseFloat(orderData['subtotal']);
		/**
		 * @type {NeweggLineItem[]}
		 */
		this.orderlines = [];
		for (const lineItem of orderData['orderItems']) {
			this.orderlines.push(new AmazonLineItem(lineItem));
		}
		this.date_creation = this.parseDate(orderData['date_placed']);
		/**
		 * @type {string}
		 */
		this.recipient_name = orderData['recipient_name'];
		this.shipping_address = {
			/**
			 * @type {string}
			 */
			street : orderData['address_line_1'],
			/**
			 * @type {string}
			 */
			street2 : orderData['address_line_2'],
			/**
			 * @type {string}
			 */
			city : orderData['city'],
			/**
			 * @type {string}
			 */
			state : orderData['state'],
			/**
			 * @type {string}
			 */
			zip : orderData['postalCode']
		};
		/**
		 * @type {string}
		 */
		this.email = orderData['email'];
		/**
		 * @type {string}
		 */
		this.phone = orderData['phone'];
		/**
		 * @type {string}
		 */
		this.shipper = orderData['service'];
		/**
		 * @type {string}
		 */
		this.status_name = orderData['status'];
		this.channel = {
			identifier : 'amazon',
			name : 'Amazon'
		}
		this.errorMsg = orderData['error'] || null;
		this.labelsPrinted = orderData['labels_printed'] || null;
		this.url = `https://sellercentral.amazon.com/orders-v3/order/${this.order_id}`;
	}

	getStatus() {
		return (this.status_name == 'Unshipped') ? 'Awaiting Shipment' : this.status_name;
	}

	getNormalizedShipper() {
		const upsShippingMethodDict = {
			'UPSNextDayAir' : 'UPS Next Day Air',
			'UPS2ndDayAir' : 'UPS 2nd Day Air',
			'UPSGround' : 'UPS Ground'
		};
		if (this.shipper.startsWith('UPS')) {
			return (upsShippingMethodDict[this.shipper]) ? upsShippingMethodDict[this.shipper] : this.shipper;
		}
		else {
			return this.shipper;
		}
	}

	getTableRow() {
		const date = new Date(Date.parse(this.date_creation)).toLocaleString();
		let returnString =
			`<tr ${(this.errorMsg) ? `class="ship_error"` : ''}>
				<td class='halign-center' rowspan='${this.orderlines.length}'><input channel='${this.channel.identifier}' order_id='${this.order_id}' class='${this.channel.identifier}_select_checkbox' id='checkbox_${this.order_id}' type='checkbox' onchange='updateSelected(this)' /></td>
				<td rowspan='${this.orderlines.length}'><a href='${this.url}' target='_blank'>${this.order_id}</a></td>
				<td rowspan='${this.orderlines.length}'>${(this.recipient_name) ? this.recipient_name : ''}</td>
				<td rowspan='${this.orderlines.length}'>${date}</td>
				<td>${this.orderlines[0].getTitleCellContents()}</td>
				<td class='halign-center'>${this.orderlines[0]['quantity']}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>$${this.price.toLocaleString()}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'>${(this.shipper) ? this.getNormalizedShipper() : ''}</td>
				<td class='halign-center' rowspan='${this.orderlines.length}'><p>${this.getStatus()}</p>${(this.errorMsg) ? `<p>${this.errorMsg}</p>` : ''}${(this.labelsPrinted && this.labelsPrinted != '0') ? `<p>Labels Printed</p>` : ''}</td>
				<td rowspan='${this.orderlines.length}' class="halign-center">
					${this.getActionButtons()}
				</td>
			</tr>
			`;
		for (let i=1; i < this.orderlines.length; i++) {
			returnString += `
				<tr>
					<td>${this.orderlines[i].getTitleCellContents()}</td>
					<td class='halign-center'>$${this.orderlines[i].quantity}</td>
				</tr>
			`;
		}
		return returnString;
	}

	getActionButtons() {
		return ''; // TODO placeholder
		return `
			<p style="margin-top:6px;">
				<a href="packingslips/${this.order_id}_ebay_packingslip.pdf" target="_blank" class="action_button">View Packing Slip</a>
			</p>
			<p style="margin-top:6px;">
				<a href="${this.getWorldshipUrl()}" target="_blank" class="action_button">Create Shipping Label</a>
			</p>
			`;
	}

	getWorldshipUrl() {
		const shipper = (this.shipper.startsWith("UPS")) ? this.shipper.substr(3).trim() : this.shipper;
		const ref1 = (this.channel.name == 'eBay Other') ? 'eBay' : this.channel.name;
		const urlElements = {
			company : encodeURI(this.recipient_name),
			address1 : encodeURI(sanitizeString(this.shipping_address.street)),
			address2 : encodeURI(sanitizeString(this.shipping_address.street2)),
			city : encodeURI(sanitizeString(this.shipping_address.city)),
			state : encodeURI(sanitizeString(this.shipping_address.state)),
			zip : encodeURI(sanitizeString(this.shipping_address.zip)),
			phone : encodeURI(sanitizeString(this.phone)),
			email : encodeURI(sanitizeString(this.email)),
			ref1 : encodeURI(sanitizeString(ref1)),
			ref2 : encodeURI(sanitizeString(this.order_id)),
			listing_id : this.orderlines[0].legacy_id,
			channel : encodeURI(sanitizeString(this.channel.identifier)),
			shipper : encodeURI(shipper),
			origin : 'open_orders'
		}
		let url = `${CONTROLLER_BASE_URL}?m=worldship`;
		for (const element in urlElements) {
			if (urlElements[element] && urlElements[element] != 'null' && urlElements[element] != 'undefined') {
				url += `&${element}=${urlElements[element]}`
			}
		}
		return url;

		/**
		 * Returns URL-safe string
		 * @param {string} str String to sanitize
		 */
		function sanitizeString(str) {
			if (str) {
				str = str.replace('#', 'TTS_NUM');
				str = str.replace('&', 'TTS_AMP');
				str = str.replace('?', 'TTS_QST');
			}
			return str;
		}
	}
	/**
	 *
	 * @param {string} dateString String to parse into a date
	 */
	parseDate(dateString) {
		const offset = new Date().getTimezoneOffset() / -60;
		const [yearMonthDate, hourMinuteSecond] = dateString.split(' ');
		const [year, month, date] = yearMonthDate.split('-');
		const [hour, minute, second] = hourMinuteSecond.split(':');
		try {
			const dateTime = new Date(year, month-1, date, parseInt(hour) + offset, minute, second);
			return dateTime;
		}
		catch (e) {
			console.error(e);
		}
	}
}
