const CONTROLLER_BASE_URL = "admin_controller.php";

const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';
const DEFAULT_SORT = 'quoteMtsNumber';

const OPEN_QUOTE_DICT = {};
const OPEN_QUOTE_ARRAYS = {
	/**
	 * @type {OpenQuote[]}
	 */
	pricingNeeded : [],
	/**
	 * @type {OpenQuote[]}
	 */
	pricingUpdated : [],
	/**
	 * @type {OpenQuote[]}
	 */
	quoteSent : [],
};
const PAGE_ELEMENTS = {
	pricingNeeded : {
		/**
		 * @type {HTMLTableHeaderCellElement}
		 */
		heading : document.getElementById('pricingNeeded_heading'),
		/**
		 * @type {HTMLTableRowElement}
		 */
		headerRow : document.getElementById('pricingNeeded_header_row'),
		/**
		 * @type {HTMLTableSectionElement}
		 */
		body : document.getElementById('pricingNeeded_body'),
		/**
		 * @type {HTMLUListElement}
		 */
		pages : document.getElementById('pricingNeeded_pages')
	},
	pricingUpdated : {
		/**
		 * @type {HTMLTableHeaderCellElement}
		 */
		heading : document.getElementById('pricingUpdated_heading'),
		/**
		 * @type {HTMLTableRowElement}
		 */
		headerRow : document.getElementById('pricingUpdated_header_row'),
		/**
		 * @type {HTMLTableSectionElement}
		 */
		body : document.getElementById('pricingUpdated_body'),
		/**
		 * @type {HTMLUListElement}
		 */
		pages : document.getElementById('pricingUpdated_pages')
	},
	quoteSent : {
		/**
		 * @type {HTMLTableHeaderCellElement}
		 */
		heading : document.getElementById('quoteSent_heading'),
		/**
		 * @type {HTMLTableRowElement}
		 */
		headerRow : document.getElementById('quoteSent_header_row'),
		/**
		 * @type {HTMLTableSectionElement}
		 */
		body : document.getElementById('quoteSent_body'),
		/**
		 * @type {HTMLUListElement}
		 */
		pages : document.getElementById('quoteSent_pages')
	}
}
/**
 * @type {HTMLCollectionOf<HTMLImageElement>}
 */
const sort_icons = document.getElementsByClassName('sort_icon');


const COLUMN_DEFINITIONS = {
	mtsNumber: new ColumnDefinition('MTS Number', 'quoteMtsNumber', {isSortable: true, isFilterable: true}),
	status: new ColumnDefinition('Status', 'quoteStatus'),
	accountName: new ColumnDefinition('Account Name', 'accountName', {isSortable: true, isFilterable: true}),
	state: new ColumnDefinition('State', 'state'),
	createdDate: new ColumnDefinition('Date Created', 'quoteDateCreated', {isSortable: true}),
	notes: new ColumnDefinition('Notes', 'notes'),
	priceDate: new ColumnDefinition('Price Date', 'quotePriceDate', {isSortable: true}),
	shipDate: new ColumnDefinition('Ship Date', 'quoteShipDate', {isSortable: true}),
	rfb: new ColumnDefinition('RFB', 'rfb'),
	sku: new ColumnDefinition('SKU', 'products', {isFilterable: true}),
	description: new ColumnDefinition('Description', 'productName'),
	quantity: new ColumnDefinition('Qty.', 'productQuantity'),
	high: new ColumnDefinition('High', 'high'),
	minimum: new ColumnDefinition('Minimum', 'minimum'),
	aGrade: new ColumnDefinition('A', 'aGrade'),
	bGrade: new ColumnDefinition('B', 'bGrade'),
	cGrade: new ColumnDefinition('C', 'cGrade'),
	dGrade: new ColumnDefinition('D', 'dGrade'),

	// salesperson: new ColumnDefinition('Sales', 'salesperson', {isSortable: true, isFilterable: true}),
	// mts_number: new ColumnDefinition('MTS', 'mts_number', {isSortable: true, isFilterable: true}),
	// account_name: new ColumnDefinition('School', 'account_name', {isSortable: true, isFilterable: true}),
	// invoice_date: new ColumnDefinition('Invoice Date', 'invoice_date', {isSortable: true}),
	// ship_date: new ColumnDefinition('Ship Date', 'ship_date', {isSortable: true}),
	// products: new ColumnDefinition('SKU', 'products', {isFilterable: true}),
	// quantity: new ColumnDefinition('Quantity', 'quantity'),
	// notes: new ColumnDefinition('Notes', 'notes'),
	// tracking_numbers: new ColumnDefinition('Tracking Numbers', 'tracking_numbers'),
	// subtotal: new ColumnDefinition('Subtotal', 'subtotal', {isSortable: true}),
	// updateRecord: new ColumnDefinition('Update', 'updateRecord', {classNames: 'updateElement'}),
	// stateAbbreviation: new ColumnDefinition('State', 'stateAbbreviation', {isFilterable: true, isSortable: true})
};

/**
 * @type {ColumnDefinition[]}
 */
const columns = {
	default : [
		COLUMN_DEFINITIONS.mtsNumber,
		COLUMN_DEFINITIONS.accountName,
		COLUMN_DEFINITIONS.state,
		COLUMN_DEFINITIONS.createdDate,
		COLUMN_DEFINITIONS.notes,
		// COLUMN_DEFINITIONS.priceDate,
		// COLUMN_DEFINITIONS.shipDate,
		COLUMN_DEFINITIONS.rfb,
		COLUMN_DEFINITIONS.high,
		COLUMN_DEFINITIONS.minimum,
		COLUMN_DEFINITIONS.sku,
		COLUMN_DEFINITIONS.description,
		COLUMN_DEFINITIONS.quantity,
		COLUMN_DEFINITIONS.aGrade,
		COLUMN_DEFINITIONS.bGrade,
		COLUMN_DEFINITIONS.cGrade,
		COLUMN_DEFINITIONS.dGrade,
		COLUMN_DEFINITIONS.status
	]
};


const pagination = {
	pricingNeeded : {
		entriesPerPage: 50,
		currentPage: 0,
		maxPages: 1
	},
	pricingUpdated : {
		entriesPerPage: 50,
		currentPage: 0,
		maxPages: 1
	},
	quoteSent : {
		entriesPerPage: 50,
		currentPage: 0,
		maxPages: 1
	}
};

const filter = {};
const sort = {
	by: DEFAULT_SORT,
	order: 1,
};

/**
 * @type {Status[]}
 */
const STATUS_ARRAY = [];

/**
 * @type {Response}
 */
let lastResponse;

init();

async function init() {
	const returnedData = await getOpenQuotes();
	if (returnedData['statuses']) {
		const statuses = returnedData['statuses'];
		STATUS_ARRAY.length = 0;
		for(const status of statuses) {
			STATUS_ARRAY.push(new Status(status));
		}
	}

	if (returnedData['openQuotes']) {
		const openQuotes = returnedData['openQuotes'];
		processOpenQuotes(openQuotes);
		sortQuoteArrays();
	}
	renderHeaders();
	renderPagination();
	renderBody();
	setLoadingModalVisible(false);
}

async function zohoRefreshBuybacks() {
	setLoadingModalVisible(true);
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=open_quotes_refresh`)
		.then(response => response.text())
		.then(data => { return data }
	);
	init();
}

async function getOpenQuotes() {
	const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=open_quotes_fetch`)
	.then(response => { return response.text() });
	try {
		const data = JSON.parse(responseText);
		return data;
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
		return null;
	}
}

function processOpenQuotes(openQuotes) {
	emptyDict();
	for (const quote of openQuotes) {
		const parsedQuote = new OpenQuote(quote);
		OPEN_QUOTE_DICT[parsedQuote.getQuoteZohoId()] = parsedQuote;
	}
	function emptyDict() {
		for (const element in OPEN_QUOTE_DICT) {
			delete OPEN_QUOTE_DICT[element];
		}
	}
}

function sortQuoteArrays() {
	emptyArrays();
	for (const zohoId in OPEN_QUOTE_DICT) {
		/**
		 * @type {OpenQuote}
		 */
		const quote = OPEN_QUOTE_DICT[zohoId];
		if (quote.getStatusId() == '1') { // pricing needed
			OPEN_QUOTE_ARRAYS.pricingNeeded.push(quote);
		}
		else if (quote.getStatusId() == '2') { // pricing updated
			OPEN_QUOTE_ARRAYS.pricingUpdated.push(quote);
		}
		else if (quote.getStatusId() == '3') { // quote sent
			OPEN_QUOTE_ARRAYS.quoteSent.push(quote);
		}
	}

	function emptyArrays() {
		for (const category in OPEN_QUOTE_ARRAYS) {
			OPEN_QUOTE_ARRAYS[category].length = 0;
		}
	}
}

function renderHeaders() {
	for (const quoteType in OPEN_QUOTE_ARRAYS) {
		if (OPEN_QUOTE_ARRAYS[quoteType].length > 0) {
			PAGE_ELEMENTS[quoteType].heading.hidden = false;
			renderHeaderRow(quoteType);
		}
		else {
			PAGE_ELEMENTS[quoteType].heading.hidden = true;
		}
	}

	/**
	 *
	 * @param {string} status
	 */
	function renderHeaderRow(status) {
		/**
		 * @type {HTMLTableRowElement}
		 */
		const headerRow = PAGE_ELEMENTS[status].headerRow;

		headerRow.innerHTML = '';
		for (const column of columns.default) {
			// headerRow.innerHTML += column.buildHeader();
			headerRow.appendChild(column.getHeaderCellElement());
		}
		headerRow.appendChild(generateActionHeader());


		function generateActionHeader() {
			const headerCell = document.createElement('th');
			const button = document.createElement('button');
			headerCell.appendChild(button);
			button.innerText = "Save All Changes";
			button.setAttribute('onclick', 'saveAllChanges(this)');
			return headerCell;
		}
	}
}

function renderPagination() {
	for (const category in OPEN_QUOTE_ARRAYS) {
		const maxPages = Math.ceil(getNumValidEntries(category) / pagination[category].entriesPerPage);
		if (maxPages) {
			pagination[category].maxPages = maxPages;
			/**
			 * @type {HTMLUListElement}
			 */
			const pageControls = PAGE_ELEMENTS[category].pages;
			pageControls.innerHTML = '';
			const prevPageButton = document.createElement('li');
			prevPageButton.setAttribute('onclick', `setCurrentPage(this)`);
			prevPageButton.setAttribute('pnum', 'prev');
			prevPageButton.setAttribute('category', category);
			prevPageButton.classList.add('select-page-button');
			prevPageButton.innerText = '«';
			pageControls.appendChild(prevPageButton);
			for (let i = 0; i < pagination[category].maxPages; i++) {
				const pageButton = document.createElement('li');
				pageControls.appendChild(pageButton);
				pageButton.setAttribute('onclick', `setCurrentPage(this)`);
				pageButton.setAttribute('pnum', `${i}`);
				pageButton.setAttribute('category', category);
				pageButton.id = `${category}_pagebutton_${i}`;
				pageButton.classList.add('select-page-button');
				if (i == pagination[category].currentPage) {
					pageButton.classList.add('active');
				}
				pageButton.innerText = `${i+1}`;
			}
			const nextPageButton = document.createElement('li');
			nextPageButton.setAttribute('onclick', `setCurrentPage(this)`);
			nextPageButton.setAttribute('pnum', 'next');
			nextPageButton.setAttribute('category', category);
			nextPageButton.classList.add('select-page-button');
			nextPageButton.innerText = '»';
			pageControls.appendChild(nextPageButton);
		}
	}

	function getNumValidEntries(category) {
		/**
		 * @type {OpenQuote[]}
		 */
		const arr = OPEN_QUOTE_ARRAYS[category];
		let numValid = 0;
		for (const quote of arr) {
			numValid += (quote.getIsValid()) ? 1 : 0;
		}
		return numValid;
	}
}

function renderBody(refreshValidEntries = true) {
	if (refreshValidEntries) {
		getValidEntries();
	}
	for(const category in OPEN_QUOTE_ARRAYS) {
		/**
		 * @type {HTMLTableSectionElement}
		 */
		const tableBody = PAGE_ELEMENTS[category].body;
		tableBody.innerHTML = '';
		/**
		 * @type {OpenQuote[]}
		 */
		const categoryQuoteArray = OPEN_QUOTE_ARRAYS[category];
		const startIndex = pagination[category].currentPage * pagination[category].entriesPerPage;
		let numRendered = 0;
		for (let i = startIndex; i < categoryQuoteArray.length && numRendered < pagination[category].entriesPerPage; i++) {
			const activeQuote = categoryQuoteArray[i];
			const valid = activeQuote.getIsValid();
			if (valid) {
				const quoteRows = activeQuote.createQuoteRow();
				for (const row of quoteRows) {
					const evenOrOdd = (numRendered % 2 == 0) ? 'even' : 'odd';
					row.classList.add(evenOrOdd);
					tableBody.appendChild(row);
				}
				numRendered++;
			}
		}
	}
	applyDatepicker();
}

/**
 *
 * @param {HTMLLIElement} pageControlButton
 */
function setCurrentPage(pageControlButton) {
	const category = pageControlButton.getAttribute('category');
	const pnum = pageControlButton.getAttribute('pnum');
	const nextPage = (pnum == 'prev') ? Math.max(pagination[category].currentPage - 1, 0) : (pnum == 'next') ? Math.min(pagination[category].currentPage + 1, pagination[category].maxPages - 1) : parseInt(pnum);
	/**
	 * @type {number}
	 */
	const currentActivePage = pagination[category].currentPage;
	if (currentActivePage != nextPage) {
		/**
		 * @type {HTMLLIElement}
		 */
		const currentActivePageButton = document.getElementById(`${category}_pagebutton_${currentActivePage}`);
		/**
		 * @type {HTMLLIElement}
		 */
		const nextActivePageButton = document.getElementById(`${category}_pagebutton_${nextPage}`);

		currentActivePageButton.classList.remove('active');
		nextActivePageButton.classList.add('active');
		pagination[category].currentPage = nextPage;
		renderBody(false);
	}
}

/**
 *
 * @param {string} quoteZohoId Zoho ID of the quote to expand/contract
 */
function toggleQuoteExpanded(quoteZohoId) {
	/**
	 * @type {OpenQuote}
	 */
	const openQuote = OPEN_QUOTE_DICT[quoteZohoId];
	openQuote.toggleExpanded();
}

/**
 *
 * @param {HTMLInputElement} priceInput
 */
function updatePrice(priceInput) {
	const quoteZohoId = priceInput.getAttribute('quoteZohoId');
	const grade = priceInput.getAttribute('grade');
	const index = parseInt(priceInput.getAttribute('index'));
	const newPrice = parseInt(priceInput.value);
	/**
	 * @type {OpenQuote}
	 */
	const activeQuote = OPEN_QUOTE_DICT[quoteZohoId];
	const activeLineItem = activeQuote.getLineItemByIndex(index);
	if (grade == 'A') {
		activeLineItem.setAGradePrice(newPrice);
		const highPriceCell = document.getElementById(`${quoteZohoId}_${index}_highPriceCell`);
		highPriceCell.innerText = activeLineItem.getHighPrice().toLocaleString('en-us', {style: 'currency', currency: 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
	}
	else if (grade == 'B') {
		activeLineItem.setBGradePrice(newPrice);
	}
	else if (grade == 'C') {
		activeLineItem.setCGradePrice(newPrice);
	}
	else if (grade == 'D') {
		activeLineItem.setDGradePrice(newPrice);
		const minimumPriceCell = document.getElementById(`${quoteZohoId}_${index}_minimumPriceCell`);
		minimumPriceCell.innerText = activeLineItem.getMinimumPrice().toLocaleString('en-us', {style: 'currency', currency: 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
	}
	activeQuote.setIsChanged(true);
}

/**
 *
 * @param {HTMLSelectElement} statusSelect
 */
function updateStatus(statusSelect) {
	const quoteZohoId = statusSelect.getAttribute('quoteZohoId');
	/**
	 * @type {OpenQuote}
	 */
	const activeQuote = OPEN_QUOTE_DICT[quoteZohoId];
	const newStatusId = statusSelect.value;
	activeQuote.setStatusId(newStatusId);
	activeQuote.setIsChanged(true);
}

/**
 *
 * @param {HTMLInputElement} dateInput
 */
function updateQuoteDate(dateInput) {
	const dateType = dateInput.getAttribute('dateType');
	if (dateType == 'shipDate') {
		updateShipDate(dateInput);
	}
	else if (dateType == 'priceDate') {
		updatePriceDate(dateInput);
	}
}

/**
 *
 * @param {HTMLInputElement} priceDateInput
 */
function updatePriceDate(priceDateInput) {
	const quoteZohoId = priceDateInput.getAttribute('quoteZohoId');
	/**
	 * @type {OpenQuote}
	 */
	const activeQuote = OPEN_QUOTE_DICT[quoteZohoId];
	const newPriceDate = priceDateInput.value;
	activeQuote.setPriceDate(newPriceDate);
	activeQuote.setIsChanged(true);
}

/**
 *
 * @param {HTMLInputElement} shipDateInput
 */
function updateShipDate(shipDateInput) {
	const quoteZohoId = shipDateInput.getAttribute('quoteZohoId');
	/**
	 * @type {OpenQuote}
	 */
	const activeQuote = OPEN_QUOTE_DICT[quoteZohoId];
	const newShipDate = shipDateInput.value;
	activeQuote.setShipDate(newShipDate);
	activeQuote.setIsChanged(true);
}

/**
 *
 * @param {HTMLTextAreaElement} noteInput
 */
function updateNotes(noteInput) {
	const quoteZohoId = noteInput.getAttribute('quoteZohoId');
	/**
	 * @type {OpenQuote}
	 */
	const activeQuote = OPEN_QUOTE_DICT[quoteZohoId];
	const newNotes = noteInput.value.trim();
	activeQuote.setNotes(newNotes);
	activeQuote.setIsChanged(true);
}

async function saveAllChanges(input) {
	for (const category in OPEN_QUOTE_ARRAYS) {
		/**
		 * @type {OpenQuote[]}
		 */
		const activeArray = OPEN_QUOTE_ARRAYS[category];
		for (const quote of activeArray) {
			if (quote.getIsChanged()) {
				saveChanges(quote.getQuoteZohoId(), false);
			}
		}
	}

	sortQuoteArrays();
	renderHeaders();
	renderPagination();
	renderBody();
	setLoadingModalVisible(false);
}

/**
 *
 * @param {string} quoteZohoId
 */
async function saveChanges(quoteZohoId, render=true) {
	setLoadingModalVisible(true);
	/**
	 * @type {OpenQuote}
	 */
	const quoteToSave = OPEN_QUOTE_DICT[quoteZohoId];
	const dataToSave = quoteToSave.getSaveableFieldsAsObject();
	const stringified = JSON.stringify(dataToSave).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");
	const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=open_quotes_save_changes`, {
		method: 'POST',
		headers: {
			'Accept' : 'application/json',
			'Content-Type': 'application/json'
		},
		body: stringified
	});
	const content = await rawResponse.text();

	console.log(content);

	if (render) {
		sortQuoteArrays();
		renderHeaders();
		renderPagination();
		renderBody();
		setLoadingModalVisible(false);
	}
}

function applyDatepicker() {
	const datefield = document.createElement('input');
	datefield.type = 'date';
	if (datefield.type != 'date') {
		try {
			$("input[isDateInput='yes']").datepicker({ dateFormat: 'yy-mm-dd' });
		}
		catch (e) {
			console.error(e);
		}
	}
	delete datefield;
}

/* Filter/sort functions below */

/**
 *
 * @param {HTMLInputElement} input
 */
function filterInput(input) {
	filter[input.getAttribute('filter')] = input.value;
	const sameNameInputs = document.getElementsByClassName(input.getAttribute('filter') + '_filter');
	for(const elem of sameNameInputs) {
		elem.value = input.value;
	}
	renderBody(true);
	renderPagination();
}

function clearFilter() {
	for (const entry in filter) {
		delete filter[entry];
	}
}

function getValidEntries() {
	totalValue = 0;
	const validEntries = {

	};
	validEntries.length = 0;
	pagination.currentPage = 0;
	for (const arrayType in OPEN_QUOTE_ARRAYS) {
		/**
		 * @type {OpenQuote[]}
		 */
		const currentArray = OPEN_QUOTE_ARRAYS[arrayType];
		for (const quote of currentArray) {
			for (const lineItem of quote.getLineItems()) {
				lineItem.setIsVisible(true);
			}
			let valid = true;
			for (const entry in filter) {
				const currentFilter = filter[entry];
				if (filter[entry] == '') {
					delete filter[entry];
				}
				else if (entry == 'products') {
					valid = validateProducts(quote, currentFilter);
				}
				else if (quote[entry] && currentFilter && !quote[entry].toString().toLowerCase().includes(currentFilter.toLowerCase())) {
					valid = false;
				}
			}
			quote.setIsValid(valid);
		}
	}


	// for (const record of SHIP_LOG_ARRAY) {
	// 	for (const product of record.products) {
	// 		product.visible = true;
	// 	}
	// 	let valid = true;
	// 	for(const entry in filter) {
	// 		if (filter[entry] == '') {
	// 			delete filter[entry];
	// 		}
	// 		else if (entry == 'products') {
	// 			valid = validateProducts(record, filter[entry]);
	// 		}
	// 		else if (record[entry] && filter[entry] && !record[entry].toString().toLowerCase().includes(filter[entry].toLowerCase())) {
	// 			valid = false;
	// 		}
	// 	}
	// 	if (valid) {
	// 		if (isNaN(record.getSubtotal())) {
	// 			console.log('nan detected');
	// 			console.log(record);
	// 		}
	// 		else {
	// 			totalValue += parseFloat(record.subtotal);
	// 		}
	// 		validEntries.push(record);
	// 	}
	// }
	// pagination.maxPages = Math.ceil(validEntries.length / pagination.entriesPerPage);

	/**
	 * Filter valid products
	 * @param {OpenQuote} record Record to check
	 * @param {string} filterValue Value to filter on
	 */
	function validateProducts(record, filterValue) {
		let valid = false;
		for (const product of record.lineItems) {
			if (product.filter.toLowerCase().includes(filterValue.toLowerCase())) {
				product.setIsVisible(true);
				valid = true;
			}
			else {
				product.setIsVisible(false);
			}
		}
		return valid;
	}
}

/* Sort functions below */

function setSort(newSort) {
	if (newSort == sort.by) {
		sort.order *= -1;
	} else {
		sort.by = newSort;
	}
	sortRecords();
	renderBody(false);

	function sortRecords() {
		for (const category in OPEN_QUOTE_ARRAYS) {
			/**
			 * @type {OpenQuote[]}
			 */
			const currentArray = OPEN_QUOTE_ARRAYS[category];
			currentArray.sort((a,b) => sortFunction(a,b));
		}
		for (const icon of sort_icons) {
			if (icon.parentElement.getAttribute('sort_order') != sort.by) {
				icon.src = SORT_NONE_URL;
			} else {
				if (sort.order == 1) {
					icon.src = SORT_ASC_URL;
				} else if (sort.order == -1) {
					icon.src = SORT_DESC_URL;
				}
			}
		}
		function sortFunction(a,b) {
			if(isNaN(a[sort.by]) || sort.by == 'quoteShipDate' || sort.by == 'quotePriceDate' || sort.by == 'quoteDateCreated') {
				if(a[sort.by] < b[sort.by]) {
					return -1 * sort.order;
				}
				if(a[sort.by] > b[sort.by]) {
					return 1 * sort.order;
				}
			} else {
				if(parseInt(a[sort.by]) < parseInt(b[sort.by])) {
					return -1 * sort.order;
				}
				if(parseInt(a[sort.by]) > parseInt(b[sort.by])) {
					return 1 * sort.order;
				}
			}
		}
	}
}
