class OpenQuote {

	constructor(data) {
		/**
		 * @type {string}
		 */
		const quoteDateCreatedString = data['quoteDateCreated'];
		/**
		 * @type {string}
		 */
		const quoteShipDateString = data['quoteShipDate'] || '';
		/**
		 * @type {string}
		 */
		const quotePriceDateString = data['quotePriceDate'] || '';
		/**
		 * @type {string}
		 */
		this.quoteZohoId = data['quoteZohoId'];
		/**
		 * @type {string}
		 */
		this.quoteSubject = data['quoteSubject'];
		/**
		 * @type {string}
		 */
		this.quoteMtsNumber = data['quoteMtsNumber'];
		this.quoteRfb = (data['quoteRfb'] == '1');
		/**
		 * @type {Date}
		 */
		this.quoteDateCreated = new Date(quoteDateCreatedString.replace(' ', 'T'));
		/**
		 * @type {Date}
		 */
		this.quoteShipDate = new Date(quoteShipDateString.replace(' ', 'T'));
		/**
		 * @type {Date}
		 */
		this.quotePriceDate = new Date(quotePriceDateString.replace(' ', 'T'));
		/**
		 * @type {string}
		 */
		this.accountName = data['accountName'];
		/**
		 * @type {string}
		 */
		this.accountZohoId = data['accountZohoId'];
		/**
		 * @type {string}
		 */
		this.salespersonName = data['salespersonName'];
		/**
		 * @type {string}
		 */
		this.salespersonZohoId = data['salespersonZohoId'];

		/**
		 * @type {LineItem[]}
		 */
		this.lineItems = [];

		this.status = "TODO"; // TODO
		this.statusId = data['quoteStatusId'];
		/**
		 * @type {string}
		 */
		this.state = data['stateAbbreviation'];
		this.shipDate = "TODO" // TODO

		/**
		 * @type {string}
		 */
		this.quoteNotes = data['quoteNotes'] || '';

		this.expanded = false;

		this.source = (this.quoteRfb) ? "RFB" : "TODO";

		this.isValid = true;

		this.changed = false;


		for (const lineItem of data['lineItems']) {
			this.lineItems.push(new LineItem(lineItem));
		}
	}

	createQuoteRow(rowspan=1) {
		const expanded = this.expanded;
		const rows = [];
		const visibleLineItemIndexes = this.getVisibleLineItems();
		rowspan = (expanded && visibleLineItemIndexes.length > 1) ? visibleLineItemIndexes.length+1 : Math.min(visibleLineItemIndexes.length, 2);
		rows.push(this.createMainRow(rowspan));
		if (visibleLineItemIndexes.length > 1) {
			if (expanded) {
				for (let i = 1; i < visibleLineItemIndexes.length; i++) {
					const index = visibleLineItemIndexes[i];
					rows.push(this.createLineItemRow(index));
				}
			}
			rows.push(this.createExpansionToggleRow());
		}
		return rows;
	}

	createMainRow(rowspan) {
		const row = document.createElement('tr');
		row.setAttribute('zohoId', this.getQuoteZohoId());

		row.appendChild(this.getQuoteMtsNumberCell(rowspan));
		row.appendChild(this.getAccountNameCell(rowspan));
		row.appendChild(this.getStateCell(rowspan));
		row.appendChild(this.getDateCreatedCell(rowspan));
		// row.appendChild(this.getPriceDateCell(rowspan));
		row.appendChild(this.getNotesCell(rowspan));
		// row.appendChild(this.getShipDateCell(rowspan));
		row.appendChild(this.getRfbCell(rowspan));



		const firstVisibleProductIndex = this.getFirstVisibleLineItem();
		row.appendChild(this.getHighCountCell(firstVisibleProductIndex));
		row.appendChild(this.getMinimumCountCell(firstVisibleProductIndex));
		row.appendChild(this.getProductCodeCell(firstVisibleProductIndex));
		row.appendChild(this.getProductNameCell(firstVisibleProductIndex));
		row.appendChild(this.getProductQuantityCell(firstVisibleProductIndex));
		row.appendChild(this.getAGradeCell(firstVisibleProductIndex));
		row.appendChild(this.getBGradeCell(firstVisibleProductIndex));
		row.appendChild(this.getCGradeCell(firstVisibleProductIndex));
		row.appendChild(this.getDGradeCell(firstVisibleProductIndex));
		row.appendChild(this.getStatusCell(rowspan));
		row.appendChild(this.getSaveChangesCell(rowspan));
		return row;
	}

	/**
	 *
	 * @param {number} index Line item index
	 */
	createLineItemRow(index) {
		const row = document.createElement('tr');
		row.appendChild(this.getHighCountCell(index));
		row.appendChild(this.getMinimumCountCell(index));
		row.appendChild(this.getProductCodeCell(index));
		row.appendChild(this.getProductNameCell(index));
		row.appendChild(this.getProductQuantityCell(index));
		row.appendChild(this.getAGradeCell(index));
		row.appendChild(this.getBGradeCell(index));
		row.appendChild(this.getCGradeCell(index));
		row.appendChild(this.getDGradeCell(index));
		return row;
	}

	createExpansionToggleRow() {
		const row = document.createElement('tr');
		const cell = document.createElement('td');
		row.appendChild(cell);
		const link = document.createElement('a');
		cell.appendChild(link);
		link.href = "javascript:void(0);";
		link.setAttribute('onclick', `toggleQuoteExpanded('${this.getQuoteZohoId()}')`)
		cell.classList.add('halign-center');
		cell.colSpan = 9;
		if (this.expanded) {
			link.innerText = "↥ View Fewer Line Items ↥";
		}
		else {
			link.innerText = "↧ View More Line Items ↧";
		}
		return row;
	}

	getMemberByFieldName(fieldName) {
		return this[fieldName];
	}

	getQuoteZohoId() {
		return this.quoteZohoId;
	}

	getQuoteMtsNumber() {
		return this.quoteMtsNumber;
	}

	getQuoteMtsNumberCell(rowspan=1) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		cell.rowSpan = rowspan;
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Quotes/${this.getQuoteZohoId()}`;
		link.target = "_blank";
		link.innerText = this.getQuoteMtsNumber();
		return cell;
	}

	getStatus() {
		return this.status;
	}

	getStatusId() {
		return this.statusId;
	}

	setStatusId(statusId) {
		this.statusId = statusId;
	}

	getStatusCell(rowspan = 1) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.appendChild(this.getStatusSelect());
		return cell;
	}

	getStatusSelect() {
		const select = document.createElement('select');
		select.classList.add('borderless');
		for (const status of STATUS_ARRAY) {
			const selected = (status.getId() == this.getStatusId());
			select.appendChild(status.getStatusOption(selected));
		}
		select.setAttribute('quoteZohoId', this.getQuoteZohoId());
		select.setAttribute('onchange', 'updateStatus(this)');
		return select;
	}

	getState() {
		return this.state;
	}

	getStateCell(rowspan = 1) {
		return this.getSimpleTextCell('state', rowspan, true);
	}

	getNotes() {
		return this.quoteNotes;
	}

	setNotes(newNotes) {
		this.quoteNotes = newNotes;
	}

	getNotesCell(rowspan=1) {
		const cell = document.createElement('td');
		const textArea = document.createElement('textarea');
		cell.appendChild(textArea);
		cell.rowSpan = rowspan;
		textArea.innerText = this.getNotes();
		textArea.value = this.getNotes();
		textArea.setAttribute('quoteZohoId', this.getQuoteZohoId());
		textArea.setAttribute('oninput', 'updateNotes(this)');
		return cell;
	}

	getDateCreated() {
		return this.quoteDateCreated;
	}

	getDateCreatedCell(rowspan=1) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getDateCreated().toLocaleDateString();
		return cell;
	}

	getPriceDate() {
		return this.quotePriceDate;
	}

	setPriceDate(value) {
		const newDate = new Date(`${value}T12:00:00`);
		this.quotePriceDate = newDate;
	}

	getPriceDateCell(rowspan=1) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.appendChild(this.getDateInput('priceDate'), this.getPriceDate());
		return cell;
	}

	getShipDate() {
		return this.quoteShipDate;
	}

	setShipDate(value) {
		const newDate = new Date(`${value}T12:00:00`);
		this.quoteShipDate = newDate;
	}

	getShipDateCell(rowspan=1) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.appendChild(this.getDateInput('shipDate'), this.getShipDate());
		return cell;
	}

	getRfbCell(rowspan=1) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const checkbox = document.createElement('input');
		cell.appendChild(checkbox);
		checkbox.type = 'checkbox';
		checkbox.checked = (this.quoteRfb);
		checkbox.classList.add('rfb');
		return cell;
	}

	getDateInput(dateType='', value=null) {
		const input = document.createElement('input');
		input.placeholder = new Date().toISOString().substring(0, 10);
		if (!value) {
			if (dateType == 'shipDate') {
				value = this.getShipDate();
			}
			else if (dateType == 'creationDate') {
				value = this.getDateCreated();
			}
			else if (dateType == 'priceDate') {
				value = this.getPriceDate();
			}
		}
		if (value instanceof Date && !isNaN(value.valueOf())) {
			console.log(value);
			input.value = value.toISOString().substring(0, 10);
			console.log(input.value);
		}
		input.id = `${this.getQuoteZohoId()}_${dateType}_dateInput`;
		input.type = "date";
		input.setAttribute('isDateInput', 'yes');
		input.setAttribute('dateType', dateType);
		input.setAttribute('onchange', 'updateQuoteDate(this)');
		input.setAttribute('quoteZohoId', this.getQuoteZohoId());
		// if (value) {
		// 	input.value = value;
		// }
		return input;
	}

	getSourceCell(rowspan=1) {
		return this.getSimpleTextCell('source', rowspan);
	}

	getAccountZohoId() {
		return this.accountZohoId;
	}

	getAccountName() {
		return this.accountName;
	}

	getAccountNameCell(rowspan=1) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		cell.rowSpan = rowspan;
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Accounts/${this.getAccountZohoId()}`;
		link.target = "_blank";
		link.innerText = this.getAccountName();
		return cell;
	}

	getSaveChangesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const saveButton = document.createElement('button');
		cell.appendChild(saveButton);
		saveButton.innerText = "Save Changes";
		saveButton.setAttribute('onclick', `saveChanges('${this.getQuoteZohoId()}')`);
		return cell;
	}

	getLineItems() {
		return this.lineItems;
	}

	getLineItemByIndex(index) {
		if (index >= this.getLineItems().length) {
			index = Math.max(this.getLineItems().length-1, 0);
		}
		return this.lineItems[index];
	}

	getProductCodeCell(index) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Quotes/${this.getQuoteZohoId()}#secDiv_Product_Details`;
		cell.appendChild(link);
		link.target = "_blank";
		link.innerText = this.getLineItemByIndex(index).getProductCode();
		return cell;
	}

	getProductNameCell(index) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Quotes/${this.getQuoteZohoId()}#secDiv_Product_Details`;
		cell.appendChild(link);
		link.target = "_blank";
		link.innerText = this.getLineItemByIndex(index).getProductName();
		return cell;
	}

	getProductQuantityCell(index) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		link.href = `https://crm.zoho.com/crm/org26161413/tab/Quotes/${this.getQuoteZohoId()}#secDiv_Product_Details`;
		cell.appendChild(link);
		link.target = "_blank";
		link.innerText = this.getLineItemByIndex(index).getProductQuantity().toLocaleString();
		return cell;
	}

	/**
	 *
	 * @param {number} index Product index
	 */
	getHighCountCell(index) {
		const cell = document.createElement('td');
		cell.classList.add('halign-center');
		cell.id = `${this.getQuoteZohoId()}_${index}_highPriceCell`;
		cell.innerText = this.getLineItemByIndex(index).getHighPrice().toLocaleString('en-us', {style: 'currency', currency: 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		return cell;
	}

	getMinimumCountCell(index) {
		const cell = document.createElement('td');
		cell.classList.add('halign-center');
		cell.id = `${this.getQuoteZohoId()}_${index}_minimumPriceCell`;
		cell.innerText = this.getLineItemByIndex(index).getMinimumPrice().toLocaleString('en-us', {style: 'currency', currency: 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		return cell;
	}

	/**
	 *
	 * @param {number} index
	 */
	getAGradeCell(index) {
		const cell = document.createElement('td');
		const input = document.createElement('input');
		cell.appendChild(input);
		cell.classList.add('halign-center');
		input.type = "number";
		input.classList.add('borderless');
		input.style.width = "50px";
		input.min = '0';
		input.value = this.getLineItemByIndex(index).getAGradePrice();
		input.setAttribute('quoteZohoId', this.getQuoteZohoId());
		input.setAttribute('index', index.toString());
		input.setAttribute('grade', 'A');
		input.setAttribute('oninput', 'updatePrice(this)');
		return cell;
	}

	/**
	 *
	 * @param {number} index
	 */
	getBGradeCell(index) {
		const cell = document.createElement('td');
		const input = document.createElement('input');
		cell.appendChild(input);
		cell.classList.add('halign-center');
		input.type = "number";
		input.classList.add('borderless');
		input.style.width = "50px";
		input.min = '0';
		input.value = this.getLineItemByIndex(index).getBGradePrice();
		input.setAttribute('quoteZohoId', this.getQuoteZohoId());
		input.setAttribute('index', index.toString());
		input.setAttribute('grade', 'B');
		input.setAttribute('oninput', 'updatePrice(this)');
		return cell;
	}

	/**
	 *
	 * @param {number} index
	 */
	getCGradeCell(index) {
		const cell = document.createElement('td');
		const input = document.createElement('input');
		cell.appendChild(input);
		cell.classList.add('halign-center');
		input.type = "number";
		input.classList.add('borderless');
		input.style.width = "50px";
		input.min = '0';
		input.value = this.getLineItemByIndex(index).getCGradePrice();
		input.setAttribute('quoteZohoId', this.getQuoteZohoId());
		input.setAttribute('index', index.toString());
		input.setAttribute('grade', 'C');
		input.setAttribute('oninput', 'updatePrice(this)');
		return cell;
	}

	/**
	 *
	 * @param {number} index
	 */
	getDGradeCell(index) {
		const cell = document.createElement('td');
		const input = document.createElement('input');
		cell.appendChild(input);
		cell.classList.add('halign-center');
		input.type = "number";
		input.classList.add('borderless');
		input.style.width = "50px";
		input.min = '0';
		input.value = this.getLineItemByIndex(index).getDGradePrice();
		input.setAttribute('quoteZohoId', this.getQuoteZohoId());
		input.setAttribute('index', index.toString());
		input.setAttribute('grade', 'D');
		input.setAttribute('oninput', 'updatePrice(this)');
		return cell;
	}

	getIsValid() {
		return this.isValid;
	}

	setIsValid(isValid) {
		if (typeof isValid == 'boolean') {
			this.isValid = isValid;
		}
	}

	getSimpleTextCell(fieldName, rowspan, centered=false) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = this.getMemberByFieldName(fieldName);
		if (centered) {
			cell.classList.add('halign-center')
		}
		return cell;
	}

	getPlaceHolderCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.innerText = "TODO"
		return cell;
	}

	getFirstVisibleLineItem() {
		const lineItems = this.getLineItems();
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (lineItem.getIsVisible()) {
				return i;
			}
		}
	}

	getVisibleLineItems() {
		const lineItems = this.getLineItems();
		const visibleIndexes = [];
		for (let i = 0; i < lineItems.length; i++) {
			const lineItem = lineItems[i];
			if (lineItem.getIsVisible()) {
				visibleIndexes.push(i);
			}
		}
		return visibleIndexes;
	}

	toggleExpanded() {
		this.expanded = !this.expanded;
		renderBody(false);
	}

	getSaveableFieldsAsObject() {
		// const priceDate = (isNaN(this.getPriceDate().valueOf())) ? null : this.getPriceDate().toISOString().replace('T', ' ').substring(0, 19);
		// const shipDate = (isNaN(this.getShipDate().valueOf())) ? null : this.getShipDate().toISOString().replace('T', ' ').substring(0, 19);

		const toSave = {
			zohoId: this.getQuoteZohoId(),
			fields: {
				// priceDate: priceDate,
				// shipDate: shipDate,
				statusId : this.getStatusId(),
				notes : this.getNotes()
			},
			lineItems : []
		};
		for (const lineItem of this.getLineItems()) {
			toSave.lineItems.push(lineItem.getSaveableFieldsAsObject());
		}
		return toSave;
	}

	getIsChanged() {
		return this.changed;
	}

	setIsChanged(changed) {
		if (typeof changed == 'boolean') {
			this.changed = changed;
		}
	}
}
