const CONTROLLER_BASE_URL = "admin_controller.php";

const activeRegion = getRegionParameter();

const channels = {};

/**
 * @type {Set<number>}
 */
const fetchedYears = new Set();

const ttsLegacyRegionIdentifiers = ['west', 'north', 'south'];
const ttsRegionIdentifiers = [];
const ecommChannelIdentifiers = ['amazon', 'backmarket', 'ebayAuction', 'ebayMarket', 'newegg', 'others'];
const compositeChannels = ['tts', 'sp'];
const wholesaleChannels = ['wholesale', 'localWholesale'];
const TODAY = new Date();
const currentYear = TODAY.getFullYear();
const currentMonthString = (TODAY.getMonth() + 1 < 10) ? '0' + (TODAY.getMonth()+1).toString() : (TODAY.getMonth()+1).toString();
const currentDayString = (TODAY.getDate() < 10) ? '0' + (TODAY.getDate()).toString() : (TODAY.getDate()).toString();
let otherYear = currentYear-1;
let projectedSales = 0;

/**
 * @type {State[]}
 */
const states = [];

const permissions = {
	'dashboard' : {
		name : "Dashboard",
		identifier : 'dashboard',
		allowed : false,
		href : `${CONTROLLER_BASE_URL}?m=metrics_dashboard`
	},
	'west' : {
		name : "West",
		identifier : 'west',
		allowed : false,
		href : `${CONTROLLER_BASE_URL}?m=region_metrics_dashboard&region=west`
	},
	'north' : {
		name : "Northeast",
		identifier : 'north',
		allowed : false,
		href : `${CONTROLLER_BASE_URL}?m=region_metrics_dashboard&region=north`
	},
	'south' : {
		name : "South",
		identifier : 'south',
		allowed : false,
		href : `${CONTROLLER_BASE_URL}?m=region_metrics_dashboard&region=south`
	},
	'midwest' : {
		name : "Midwest",
		identifier : 'midwest',
		allowed : false,
		href : `${CONTROLLER_BASE_URL}?m=region_metrics_dashboard&region=midwest`
	},
};



init();


async function init() {
	document.getElementById('previousYearButton').hidden = true;
	document.getElementById('nextYearButton').hidden = true;
	setLoadingModalVisible(true);
	const projectedSalesResponse = await fetch(`${CONTROLLER_BASE_URL}?m=metrics_get_region_projected_sales&region=${activeRegion}`)
	.then(response => response.text());
	try {
		const data = JSON.parse(projectedSalesResponse);
		if (data['projectedSales']) {
			projectedSales = parseFloat(data['projectedSales']);
		}
	}
	catch (err) {
		console.error(err);
		console.log(projectedSalesResponse);
	}

	const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=metrics_get_region_year_data&year=${currentYear}&region=${activeRegion}`)
	.then(response => response.text());

	try {
		const data = JSON.parse(responseText);
		if (!data) {
			window.location.replace(`${CONTROLLER_BASE_URL}`);
		}
		if (data['hasDashboardPermission'] || data['regionPermissions']) {
			processPermissions(data['hasDashboardPermission'], data['regionPermissions']);
		}
		if (data['states']) {
			processStates(data['states']);
		}
		if (data['months']) {
			processMonths(data['objects']);
		}
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
	}
	const yearComparisonData = await getYearComparisonData();
	// console.log(yearComparisonData);
	renderYearComparisonSpans(yearComparisonData);
	console.log('done!');
	setLoadingModalVisible(false);
}

function processPermissions(hasDashboardPermission, regionPermissionsData) {
	if (hasDashboardPermission) {
		permissions['dashboard']['allowed'] = true;
	}
	for (const permission of regionPermissionsData) {
		permissions[permission]['allowed'] = true;
	}
	renderPermittedRegionLinks();
}

function processStates(stateData) {
	for (const state in stateData) {
		console.log(`state: ${state}`, stateData[state]);
		const newStateObject = new State(stateData[state]);
		states.push(newStateObject);
	}
	renderTableBody(currentYear, 'current');
	renderTableBody(otherYear, 'other');
	renderTotals();
}

async function getYearComparisonData() {
	const yearComparisonData = await fetch(`${CONTROLLER_BASE_URL}?m=metrics_get_region_mtd_comparisons&region=${activeRegion}&year=${currentYear}&month=${currentMonthString}&day=${currentDayString}&otherYear=${otherYear}`)
	.then(response => response.json())
	.then(data => {
		return data;
	})
	.catch(err => console.error(err));
	return yearComparisonData;
}

function renderYearComparisonSpans(data) {

	// MTD calculations
	const otherYearMTDTtsTotal = parseFloat(data['otherYearMTDTtsTotal']);
	const thisYearMTDTtsTotal = parseFloat(data['thisYearMTDTtsTotal']);
	const otherYearMTDTotal = otherYearMTDTtsTotal;
	const thisYearMTDTotal = thisYearMTDTtsTotal;

	const ttsMTDPercentageDifference = (thisYearMTDTtsTotal / otherYearMTDTtsTotal) * 100;
	const totalMTDPercentageDifference = (thisYearMTDTotal / otherYearMTDTotal) * 100;

	const ttsMTDDisplayPercentage = (ttsMTDPercentageDifference >= 100) ? `+${Math.round(ttsMTDPercentageDifference - 100)}%` : `-${Math.round(100 - ttsMTDPercentageDifference)}%`;
	const totalMTDDisplayPercentage = (totalMTDPercentageDifference >= 100) ? `+${Math.round(totalMTDPercentageDifference - 100)}%` : `-${Math.round(100 - totalMTDPercentageDifference)}%`;

	// YTD calculations
	const otherYearYTDTtsTotal = parseFloat(data['otherYearYTDTtsTotal']);
	const thisYearYTDTtsTotal = parseFloat(data['thisYearYTDTtsTotal']);
	const otherYearYTDTotal = otherYearYTDTtsTotal;
	const thisYearYTDTotal = thisYearYTDTtsTotal;

	const ttsYTDPercentageDifference = (thisYearYTDTtsTotal / otherYearYTDTtsTotal) * 100;
	const totalYTDPercentageDifference = (thisYearYTDTotal / otherYearYTDTotal) * 100;

	const ttsYTDDisplayPercentage = (ttsYTDPercentageDifference >= 100) ? `+${Math.round(ttsYTDPercentageDifference - 100)}%` : `-${Math.round(100 - ttsYTDPercentageDifference)}%`;
	const totalYTDDisplayPercentage = (totalYTDPercentageDifference >= 100) ? `+${Math.round(totalYTDPercentageDifference - 100)}%` : `-${Math.round(100 - totalYTDPercentageDifference)}%`;

	/**
	 * @type {HTMLDivElement}
	 */
	const yearComparisonGroup = document.getElementById('yearComparisonHeader');
	/**
	 * @type {HTMLCollectionOf<HTMLSpanElement>}
	 */
	const otherYearDisplaySpans = document.getElementsByClassName('otherYearNumber');
	/**
	 * @type {HTMLCollectionOf<HTMLSpanElement>}
	 */
	const thisYearDisplaySpans = document.getElementsByClassName('thisYearNumber');
	/**
	 * @type {HTMLCollectionOf<HTMLTableCellElement>}
	 */
	const yearComparisonSpColumn = document.getElementsByClassName("yearComparisonSpColumn");
	/**
	 * @type {HTMLCollectionOf<HTMLTableCellElement>}
	 */
	const yearComparisonTtsColumn = document.getElementsByClassName("yearComparisonTtsColumn");

	// MTD elements
	/**
	 * @type {HTMLSpanElement}
	 */
	const otherYearTTSMTDToNow = document.getElementById('otherYearTTSMTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const thisYearTTSMTDToNow = document.getElementById('thisYearTTSMTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const otherYearTotalMTDToNow = document.getElementById('otherYearTotalMTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const thisYearTotalMTDToNow = document.getElementById('thisYearTotalMTDToNow');

	// YTD elements
	/**
	 * @type {HTMLSpanElement}
	 */
	const otherYearTTSYTDToNow = document.getElementById('otherYearTTSYTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const thisYearTTSYTDToNow = document.getElementById('thisYearTTSYTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const otherYearTotalYTDToNow = document.getElementById('otherYearTotalYTDToNow');
	/**
	 * @type {HTMLSpanElement}
	 */
	const thisYearTotalYTDToNow = document.getElementById('thisYearTotalYTDToNow');

	yearComparisonGroup.hidden = true;
	for (const element of otherYearDisplaySpans) {
		element.innerText = otherYear;
	}
	for (const element of thisYearDisplaySpans) {
		element.innerText = currentYear
	}
	otherYearTTSMTDToNow.innerText = otherYearMTDTtsTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'});

	thisYearTTSMTDToNow.innerText = `${thisYearMTDTtsTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'})} (${ttsMTDDisplayPercentage})`;

	otherYearTotalMTDToNow.innerText = otherYearMTDTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'});
	thisYearTotalMTDToNow.innerText = `${thisYearMTDTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'})} (${totalMTDDisplayPercentage})`;

	otherYearTTSYTDToNow.innerText = otherYearYTDTtsTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'});

	thisYearTTSYTDToNow.innerText = `${thisYearYTDTtsTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'})} (${ttsYTDDisplayPercentage})`;

	otherYearTotalYTDToNow.innerText = otherYearYTDTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'});
	thisYearTotalYTDToNow.innerText = `${thisYearYTDTotal.toLocaleString('en-US', {style: 'currency', currency: 'USD'})} (${totalYTDDisplayPercentage})`;


	for (const cell of yearComparisonSpColumn) {
		cell.hidden = true;
	}
	for (const cell of yearComparisonTtsColumn) {
		cell.hidden = true;
	}
	yearComparisonGroup.hidden = false;
}

/**
 * Fetches data from database for a given year and returns via ajax request
 * @param {number} year The year for which to get the data
 * @param {string} yearType Whether is current or other year
 */
function getYearData(year, yearType) {
	const url = `${CONTROLLER_BASE_URL}?m=metrics_get_year_data&year=${year}`;
	setLoadingModalVisible(true);
	makeAjaxRequest(url, '', 'POST', returnReply);

	function returnReply() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					try {
						const data = JSON.parse(http_request.response);
						for (const channelName in data) {
							if (!channels[channelName]) {
								channels[channelName] = new Channel(data[channelName]);
							}
							else {
								/**
								 * @type {Channel}
								 */
								const existingChannelData = channels[channelName];
								existingChannelData.addData(data[channelName]);
							}
						}
						fetchedYears.push(year);
						renderTableBody(year, yearType);
						if (yearType == 'current') {
							otherYear = year-1;
							getYearData(otherYear, 'other');
							renderTotals();
						}
					}
					catch (err) {
						console.error(err);
					}
					setLoadingModalVisible(false);
					break;
				default:
					alert(`Error with AJAX request. Request status: ${http_request.status}\nRequest response: ${http_request.response}`);
					setLoadingModalVisible(false);
			}
			return true;
		}
	}

	function generateTotals() {

	}
}

/**
 *
 * @param {number} year Year to render. To be converted to string
 * @param {string} yearType Whether it is the current or the other year
 */
function renderTableBody(year, yearType) {
	renderYearTable(year.toString(), yearType);
	/**
	 *
	 * @param {string} year
	 * @param {string} yearType Current or other year
	 */
	function renderYearTable(year, yearType) {
		let even = true;
		const table = document.getElementById(`${yearType}YearTable`);
		const title = document.getElementById(`${yearType}YearTitle`);
		title.innerText = year;
		table.hidden = true;
		const tableBody = document.getElementById(`${yearType}YearTableBody`);
		tableBody.innerHTML = '';
		const totalSalesRow = generateTotalSalesRow(year);
		totalSalesRow.classList.add((even) ? 'even' : 'odd');
		even = !even;
		tableBody.appendChild(totalSalesRow);
		for (const state of states) {
			const newRow = state.generateStateRow(year);
			newRow.classList.add((even) ? 'even' : 'odd');
			even = !even;
			tableBody.appendChild(newRow);
		}
		table.hidden = false;
	}
	/**
	 *
	 * @param {string} year
	 */
	function generateTotalSalesRow(year) {
		const row = document.createElement('tr');
		const rowHead = document.createElement('th');
		row.appendChild(rowHead);
		row.classList.add('padding-y-10');
		rowHead.innerText = 'Total Sales';
		for (let i = 0; i < 12; i++) {
			const monthCell = document.createElement('td');
			row.appendChild(monthCell);
			monthCell.innerText = getMonthTotal(year, i).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		for (let i = 0; i < 4; i++) {
			const quarter = i+1;
			const quarterCell = document.createElement('td');
			row.appendChild(quarterCell);
			quarterCell.innerText = getQuarterTotal(year, quarter).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		const yearTotalCell = document.createElement('td');
		row.appendChild(yearTotalCell);
		yearTotalCell.innerText = getYearTotal(year).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});



		return row;

		/**
		 *
		 * @param {string} year
		 * @param {number} monthIndex
		 */
		function getMonthTotal(year, monthIndex) {
			let total = 0.0;
			for (const state of states) {
				try {
					total += state.getMonthTotal(year, monthIndex);

				}
				catch(err) {
					console.error(err);
					console.log(state);
				}
			}
			return total;
		}

		/**
		 *
		 * @param {string} year Year for which to get data
		 * @param {number} quarter Which quarter
		 */
		function getQuarterTotal(year, quarter) {
			let total = 0.0;
			for (const state of states) {
				total += state.getQuarterTotal(year, quarter);
			}
			return total;
		}

		function getYearTotal(year) {
			let total = 0.0;
			for (const state of states) {
				total += state.getYearTotal(year);
			}
			return total;
		}
	}

}

function renderTotals() {
	const today = new Date();
	const currentMonthIndex = today.getMonth();
	const totalYearValue = getYearTotalValue(currentYear);
	const totalMonthValue = getMonthTotal(currentYear, currentMonthIndex);
	const totalSalesCurrentYearSpan = document.getElementById('totalSalesCurrentYear');
	const totalSalesCurrentMonthSpan = document.getElementById('totalSalesCurrentMonth');
	const projectedSalesDisplay = document.getElementById('projectedSalesCurrentMonth');
	totalSalesCurrentYearSpan.innerText = totalYearValue.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
	totalSalesCurrentMonthSpan.innerText = totalMonthValue.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
	const projectedSalesTotal = projectedSales + totalMonthValue;
	projectedSalesDisplay.innerText = projectedSalesTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});

	/**
	 *
	 * @param {string} year
	 */
	function getYearTotalValue(year) {
		let total = 0.0;
		for (const state of states) {
			total += state.getYearTotal(year);
		}
		return total;
	}

	/**
	 *
	 * @param {string} year
	 * @param {number} monthIndex
	 */
	function getMonthTotal(year, monthIndex) {
		let total = 0.0;
		for (const state of states) {
			total += state.getMonthTotal(year, monthIndex);
		}
		return total;
	}
}

function renderProjectedSales() {
	renderTotals();
}

function renderPermittedRegionLinks() {
	/**
	 * @type {HTMLHeadingElement}
	 */
	const regionLinkHolder = document.getElementById('regionLinkHolder');
	regionLinkHolder.hidden = true;

	let i = 0;
	for (const permissionType in permissions) {
		const regionName = permissions[permissionType].name;
		const regionIdentifier = permissions[permissionType].identifier;
		const isAllowed = permissions[permissionType].allowed;
		const regionHref = permissions[permissionType].href;


		if (isAllowed) {
			if (i) {
				const dividerSpan = document.createElement('span');
				dividerSpan.innerText = ' | ';
				regionLinkHolder.appendChild(dividerSpan);
			}
			if (regionIdentifier == activeRegion) {
				const regionNonLink = document.createElement('span');
				regionNonLink.innerText = regionName;
				regionLinkHolder.appendChild(regionNonLink);
			}
			else {
				const regionLink = document.createElement('a');
				regionLink.href = regionHref;
				regionLink.innerText = regionName;
				regionLinkHolder.appendChild(regionLink);
			}
			i++;
		}
		regionLinkHolder.hidden = false;
	}
}

function getPreviousYearData() {
	const nextYearButton = document.getElementById('nextYearButton');
	nextYearButton.disabled = false;
	otherYear -= 1;
	if (fetchedYears.includes(otherYear)) {
		renderTableBody(otherYear, 'other');
	}
	else {
		getYearData(otherYear, 'other');
	}
	fetch(`${CONTROLLER_BASE_URL}?m=metrics_get_mtd_comparisons&year=${currentYear}&month=${currentMonthString}&day=${currentDayString}&otherYear=${otherYear}`)
	.then(response => response.json())
	.then(data => {
		renderYearComparisonSpans(data);
	});
}

function getNextYearData() {
	const nextYearButton = document.getElementById('nextYearButton');
	nextYearButton.disabled = false;
	otherYear += 1;
	if (otherYear >= currentYear - 1) {
		nextYearButton.disabled = true;
	}
	if (fetchedYears.includes(otherYear)) {
		renderTableBody(otherYear, 'other');
	}
	else {
		getYearData(otherYear, 'other');
	}
	fetch(`${CONTROLLER_BASE_URL}?m=metrics_get_mtd_comparisons&year=${currentYear}&month=${currentMonthString}&day=${currentDayString}&otherYear=${otherYear}`)
	.then(response => response.json())
	.then(data => {
		renderYearComparisonSpans(data);
	});
}

function processLegacyData(data) {
	console.log(data);
	const monthNames = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
	const monthNumbers = ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12'];
	for (const year in data) {
		const yearData = data[year];
		for (const channelName in yearData) {
			if (channelName != 'TotalSales') {
				const channelData = yearData[channelName];
				const newChannelYearData = {
					year : year,
					months : []
				};

				for (let i = 0; i < channelData.length; i++) {
					const monthSubtotal = channelData[i];
					const constructorData = {
						month : {
							name : monthNames[i],
							number : monthNumbers[i]
						},
						subtotal : monthSubtotal
					};
					newChannelYearData.months.push(constructorData);
				}
				try {
					/**
					 * @type {Channel}
					 */
					const existingChannelData = channels[channelName];
					existingChannelData.addData(newChannelYearData);
				}
				catch (err) {
					console.error(err);
					console.log(channelName);
				}
			}
		}
		fetchedYears.push(parseInt(year));
	}
}

/**
 *
 * @param {string} yearType 'current' or 'other'
 */
function generateCsv(yearType) {
	const yearToExport = (yearType == 'other') ? otherYear.toString() : currentYear.toString();
	const rows = [generateCsvHeaderRow()];
	rows.push(generateTotalSalesRow());
	for (const state of states) {
		rows.push(state.generateCsvRow(yearToExport));
	}

	let csvContent = "data:text/csv;charset=utf-8," + rows.map(e => e.join(",")).join("\n");
	const encodedUri = encodeURI(csvContent);
	const link = document.createElement("a");
	link.setAttribute("href", encodedUri);
	link.setAttribute("download", `${yearToExport}_metrics_export_${getTimestamp(new Date())}.csv`);
	document.body.appendChild(link); // Required for FF

	link.click();
	delete link;

	function generateCsvHeaderRow() {
		const headerRow = [
			`"${yearToExport}"`,
			'"Jan"',
			'"Feb"',
			'"Mar"',
			'"Apr"',
			'"May"',
			'"Jun"',
			'"Jul"',
			'"Aug"',
			'"Sep"',
			'"Oct"',
			'"Nov"',
			'"Dec"',
			'"Q1"',
			'"Q2"',
			'"Q3"',
			'"Q4"',
			'"Yearly Total"'
		];
		return headerRow;
	}


	function generateTotalSalesRow(s) {
		const newRow = ['Total Sales'];
		for (let i = 0; i < 12; i++) {
			let monthTotal = 0.0;
			for (const state of states) {
				monthTotal += state.getMonthTotal(yearToExport, i);
			}
			newRow.push(`"${monthTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0})}"`);
		}
		for (let i = 0; i < 4; i++) {
			const quarter = i+1;
			let quarterTotal = 0.0;
			for (const state of states) {
				quarterTotal += state.getQuarterTotal(yearToExport, quarter);
			}
			newRow.push(`"${quarterTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0})}"`);
		}
		let yearTotal = 0.0;
		for (const state of states) {
			yearTotal += state.getYearTotal(yearToExport);
		}
		newRow.push(`"${yearTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0})}"`);

		return newRow;
	}

	/**
	 *
	 * @param {Date} time
	 */
	function getTimestamp(time) {
		let timestamp = time.getFullYear().toString();
		if (time.getMonth()+1 < 10) {
			timestamp += '0';
		}
		timestamp += (time.getMonth()+1).toString();
		if (time.getDate() < 10) {
			timestamp += '0';
		}
		timestamp += time.getDate().toString();
		if (time.getHours() < 10) {
			timestamp += '0';
		}
		timestamp += time.getHours().toString();
		if (time.getMinutes() < 10) {
			timestamp += '0';
		}
		timestamp += time.getMinutes().toString();
		if (time.getSeconds() < 10) {
			timestamp += '0';
		}
		timestamp += time.getSeconds().toString();

		return timestamp
	}
}

function getRegionParameter() {
	const searchTerm = '&region';
	const params = window.location.search;
	const startIndex = params.indexOf(searchTerm) + searchTerm.length + 1;
	const regionString = params.substring(startIndex);
	return regionString;
}
