class State {
	constructor(data) {
		/**
		 * @type {string}
		 */
		this.name = data.name;
		/**
		 * @type {string}
		 */
		this.abbreviation = data.abbreviation;
		/**
		 * @type {string}
		 */
		this.type = data.type;

		this.years = {};

		for (const year in data.years) {
			const yearData = data.years[year];
			this.years[year] = new YearSummary(yearData);
		}
	}

	/**
	 *
	 * @param {string} year Year for which to generate a row
	 */
	generateStateRow(year) {
		const row = document.createElement('tr');
		const rowHead = document.createElement('th');
		row.appendChild(rowHead);
		row.classList.add('padding-y-10');
		rowHead.innerText = this.name;
		for (let i = 0; i < 12; i++) {
			const monthCell = document.createElement('td');
			row.appendChild(monthCell);
			monthCell.innerText = this.getMonthTotal(year, i).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		for (let i = 0; i < 4; i++) {
			const quarter = i+1;
			const quarterCell = document.createElement('td');
			row.appendChild(quarterCell);
			quarterCell.innerText = this.getQuarterTotal(year, quarter).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		const yearTotalCell = document.createElement('td');
		row.appendChild(yearTotalCell);
		yearTotalCell.innerText = this.getYearTotal(year).toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		return row;

	}

	/**
	 *
	 * @param {string} year Year to export
	 */
	generateCsvRow(year) {
		const row = [this.getName()];
		for(let i = 0; i < 12; i++) {
			const monthTotal = this.getMonthTotal(year, i);
			row.push('"' + monthTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0}) + '"');
		}
		for (let i = 0; i < 4; i++) {
			const quarterTotal = this.getQuarterTotal(year, i+1);
			row.push('"' + quarterTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0}) + '"');
		}
		const yearTotal = this.getYearTotal(year);
		row.push('"' + yearTotal.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0}) + '"');
		return row;
	}

	addData(data) {
		if (data.years) {
			for (const year in data.years) {
				const yearData = data.years[year];
				this.years[year] = new YearSummary(yearData);
			}
		}
		else if (data.year) {
			this.years[data.year] = new YearSummary(data);
		}
	}

	getYearTotal(year) {
		try {
			/**
			 * @type {YearSummary}
			 */
			const yearData = this.years[year];
			return yearData.getYearTotal();
		}
		catch(err) {
			console.error(err);
			return 0.0;
		}
	}

	/**
	 *
	 * @param {string} year
	 * @param {number} quarter
	 */
	getQuarterTotal(year, quarter) {
		try {
			/**
			 * @type {YearSummary}
			 */
			const yearData = this.years[year];
			return yearData.getQuarterTotal(quarter);
		}
		catch(err) {
			console.error(err);
			return 0.0;
		}
	}

	/**
	 *
	 * @param {string} year Year to fetch
	 * @param {number} monthIndex Index of the month to fetch
	 */
	getMonthTotal(year, monthIndex) {
		try {
			/**
			 * @type {YearSummary}
			 */
			const yearData = this.years[year];
			return yearData.getMonthTotal(monthIndex);
		}
		catch(err) {
			console.error(err);
			return 0.0;
		}
	}

	getName() {
		return this.name;
	}
}
