/**
 * Defines column parameters.
 * @author Michael Kemper
 * @since 2020-06-22
 */
class ColumnDefinition {
	/**
	 *
	 * @param {string} displayName Set column display name.
	 * @param {string} identifier Set column identifier.
	 * @param options Set column options.
	 */
	constructor(displayName, identifier, options = {isNumeric: false, isSortable: false, isFilterable: false, prefix: '', linksTo: '', linkBase: '', spansMultipleRows: true, classNames: ''}) {
		this.displayName = displayName;
		this.identifier = identifier;
		this.isNumeric = options.isNumeric || false;
		this.isSortable = options.isSortable || false;
		this.isFilterable = options.isFilterable || false;
		this.prefix = options.prefix || '';
		this.linksTo = options.linksTo || '';
		this.linkBase = options.linkBase || '';
		if (options.spansMultipleRows === false) {
			this.spansMultipleRows = false;
		}
		else {
			this.spansMultipleRows = true;
		}
		this.classNames = options.classNames || '';
	}

	buildHeader() {
		const sortString = (this.isSortable) ? `sort_order="${this.identifier}"` : '';
		const classes = `header_cell ${this.classNames}`;

		return `
			<th class="${classes}" ${sortString}>
				${this.getSortSpan()}
				${this.getFilterInput()}
			</th>
		`;
	}

	getHeaderCellElement() {
		const headerCell = document.createElement('th');
		headerCell.classList.add('header_cell');
		if (this.classNames) {
			for (const className of this.classNames.split(' ')) {
				headerCell.classList.add(className);
			}
		}
		if (this.isSortable) {
			headerCell.setAttribute('sort_order', this.identifier);
			headerCell.appendChild(this.getSortSpanElement());
		}
		else {
			headerCell.innerText = this.displayName;
		}
		if (this.isFilterable) {
			headerCell.appendChild(this.getFilterInputElement());
		}
		return headerCell;
	}

	getFilterInput() {
		return (this.isFilterable) ? `<input type="text" class="filter_input ${this.identifier}_filter width-80pct" is_numeric="${this.isNumeric}" filter="${this.identifier}" placeholder="Filter..." onkeyup="filterInput(this)" />` : '';
	}

	getFilterInputElement() {
		const inputElement = document.createElement('input');
		inputElement.type = 'text';
		inputElement.classList.add('filter_input');
		inputElement.classList.add('width-80pct');
		inputElement.classList.add(`${this.identifier}_filter`);
		inputElement.setAttribute('is_numeric', this.isNumeric);
		inputElement.setAttribute('filter', this.identifier);
		inputElement.placeholder = "Filter...";
		inputElement.setAttribute('onkeyup', 'filterInput(this)');
		return inputElement;
	}

	getSortSpan() {
		return (this.isSortable) ? `<span class="sort_icon_span cursor-pointer" sort_order="${this.identifier}" is_numeric="${this.isNumeric}" onclick="setSort(this.getAttribute('sort_order'))">${this.displayName} <img class="sort_icon" src="../svg/sort.svg"></span>` : `${this.displayName}`;
	}

	getSortSpanElement() {
		const span = document.createElement('span');
		span.classList.add('sort_icon_span');
		span.classList.add('cursor-pointer');
		span.setAttribute('sort_order', this.identifier);
		span.setAttribute('is_numeric', this.isNumeric);
		span.setAttribute('onclick', "setSort(this.getAttribute('sort_order'))");
		span.innerText = this.displayName;

		const img = document.createElement('img');
		img.src = "../svg/sort.svg";
		img.classList.add('sort_icon');
		span.appendChild(img);
		return span;
	}

	getIdentifier() {
		return this.identifier;
	}
}
