class Account {
	constructor(data) {
		/**
		 * @type {string}
		 */
		this.id = (data['id']) ? data['id'] : (data['dbId']) ? data['dbId'] : '';
		/**
		 * @type {string}
		 */
		this.zohoId = (data['zohoId']) ? data['zohoId'] : (data['zoho_id']) ? data['zoho_id'] : '';

		/**
		 * @type {string}
		 */
		this.name = data['name'];
		/**
		 * @type {string}
		 */
		this.accountType = data['accountType'];
		/**
		 * @type {string}
		 */
		this.phoneNumber = data['phoneNumber'];
		/**
		 * @type {string}
		 */
		this.website = data['website'];

		this.totalSpent = parseFloat(data['totalSpent']);

		this.enrollment = parseInt(data['enrollment']);

		this.billingAddress = {
			/**
			 * @type {string}
			 */
			street : data['billingStreet'],
			/**
			 * @type {string}
			 */
			city : data['billingCity'],
			/**
			 * @type {string|State}
			 */
			state : data['billingState'] || STATE_DICT[data['billingStateId']] || '',
			/**
			 * @type {string}
			 */
			code : data['billingCode']
		};

		this.shippingAddress = {
			/**
			 * @type {string}
			 */
			street : data['shippingStreet'],
			/**
			 * @type {string}
			 */
			city : data['shippingCity'],
			/**
			 * @type {string|State}
			 */
			state : data['shippingState'] || STATE_DICT[data['shippingStateId']] || '',
			/**
			 * @type {string}
			 */
			code : data['shippingCode']
		};

		this.newCustomer = data['newCustomer'];

		/**
		 * @type {HTMLTableRowElement}
		 */
		this.row = null;
	}

	/**
	 * Gets database ID of account
	 * @returns Database ID of account
	 */
	getId() {
		return this.id;
	}
	/**
	 * Sets database ID
	 * @param {string} id
	 */
	setId(id) {
		this.id = id;
	}

	/**
	 * Gets zoho ID of account
	 * @returns Zoho ID of account
	 */
	getZohoId() {
		return this.zohoId;
	}
	/**
	 * Sets zoho ID
	 * @param {string} zohoId
	 */
	setZohoId(zohoId) {
		this.zohoId = zohoId;
	}

	getZohoLinkUrl() {
		const baseUrl = "https://crm.zoho.com/crm/org26161413/tab/Accounts/";
		return baseUrl.concat(this.getZohoId());
	}

	/**
	 * Gets name of account
	 * @returns Name of account
	 */
	getName() {
		return this.name;
	}
	/**
	 * Sets name
	 * @param {string} name
	 */
	setName(name) {
		this.name = name;
	}

	getPhoneNumber() {
		return this.phoneNumber;
	}
	setPhoneNumber(phoneNumber) {
		this.phoneNumber = phoneNumber;
	}

	getWebsite() {
		return this.website;
	}
	setWebsite(website) {
		this.website = website;
	}

	/**
	 * Gets accountType of account
	 * @returns Type of account
	 */
	getAccountType() {
		return this.accountType;
	}
	/**
	 * Sets accountType
	 * @param {string} accountType
	 */
	setAccountType(accountType) {
		this.accountType = accountType;
	}

	getTotalSpent() {
		return this.totalSpent;
	}

	getEnrollment() {
		return (isNaN(this.enrollment)) ? 0 : this.enrollment;
	}

	getAddressStreet() {
		if (this.getBillingStreet()) {
			return this.getBillingStreet();
		}
		else if (this.getShippingStreet()) {
			return this.getShippingStreet();
		}
	}

	getAddressCity() {
		if (this.getBillingCity()) {
			return this.getBillingCity();
		}
		else if (this.getShippingCity()) {
			return this.getShippingCity();
		}
	}

	getAddressState() {
		if (this.getBillingState()) {
			const state = this.getBillingState();
			if (state instanceof State) {
				return state.getName();
			}
			else {
				return state;
			}
		}
		else if (this.getShippingState()) {
			const state = this.getShippingState();
			if (state instanceof State) {
				return state.getName();
			}
			else {
				return state;
			}
		}
	}

	getBillingStreet() {
		return this.billingAddress.street;
	}

	getBillingCity() {
		return this.billingAddress.city;
	}

	getBillingCode() {
		return this.billingAddress.code;
	}

	getBillingState() {
		return this.billingAddress.state;
	}

	getShippingStreet() {
		return this.shippingAddress.street;
	}

	getShippingCity() {
		return this.shippingAddress.city;
	}

	getShippingCode() {
		return this.shippingAddress.code;
	}

	getShippingState() {
		return this.shippingAddress.state;
	}

	getRow() {
		return this.row;
	}
	setRow(row) {
		this.row = row;
	}

	/* Cell builders start */

	/*
		zohoAccountLink : new ColumnDefinition('Zoho Account', 'zohoAccountLink'),
		name : new ColumnDefinition('Name', 'name'),
		phoneNumber : new ColumnDefinition('Phone Number', 'phoneNumber'),
		website : new ColumnDefinition('Website', 'website'),
		accountType : new ColumnDefinition('Account Type', 'accountType'),
		enrollment : new ColumnDefinition('Enrollment', 'enrollment'),
		totalSpent : new ColumnDefinition('Total Spent', 'totalSpent'),
		billingStreet : new ColumnDefinition('Billing Street', 'billingStreet'),
		billingCity : new ColumnDefinition('Billing City', 'billingCity'),
		billingState : new ColumnDefinition('Billing State', 'billingState'),
		billingCode : new ColumnDefinition('Billing Code', 'billingCode'),
		shippingStreet : new ColumnDefinition('Shipping Street', 'shippingStreet'),
		shippingCity : new ColumnDefinition('Shipping City', 'shippingCity'),
		shippingState : new ColumnDefinition('Shipping State', 'shippingState'),
		shippingCode : new ColumnDefinition('Shipping Code', 'shippingCode')
	*/

	getCellByIdentifier(identifier, classes=[]) {
		const textCellIdentifiers = [
			'name',
			'phoneNumber',
			'accountType',
			'enrollment',
			'totalSpent',
			'billingStreet',
			'billingCity',
			'billingState',
			'billingCode',
			'shippingStreet',
			'shippingCity',
			'shippingState',
			'shippingCode',
		];
		if (textCellIdentifiers.includes(identifier)) {
			return this.getTextCell(this.getValueByIdentifier(identifier), classes);
		}
		else if (identifier == "zohoAccountLink") {
			return this.getLinkCell('Zoho Account', this.getZohoLinkUrl(), classes);
		}
		else if (identifier == 'website') {
			if (this.getWebsite()) {
				return this.getLinkCell(this.getWebsite(), `https://${this.getWebsite()}`, classes)
			}
			else {
				return this.getTextCell('', classes);
			}
		}
	}

	/**
	 * Builds cell with email address
	 * @param {string[]} classes List of classes for the cell
	 * @returns Cell with email address
	 */
	getEmailAddressCell(classes=[]) {
		return this.getTextCell(this.getEmailAddress(), classes);
	}

	getNameCell(classes=[]) {
		return this.getTextCell(this.getName(), classes);
	}

	/* Generic cell builders begin */

	/**
	 * Builds a data cell with text
	 * @param {string} textContent Text to put in the cell
	 * @param {string[]} classes List of classes for the cell
	 * @returns Cell node
	 */
	getTextCell(textContent, classes=[]) {
		const cell = document.createElement('td');
		cell.innerText = textContent;
		if (classes.length > 0) {
			cell.classList.add(classes);
		}
		return cell;
	}

	getPlaceholderCell(classes=[]) {
		return this.getTextCell('placeholder', classes);
	}

	getErrorCell(classes=[]) {
		return this.getTextCell('ERROR', classes);
	}

	/**
	 * Builds a cell with a URL link
	 * @param {string} linkText Text for the link
	 * @param {string} href URL to link to
	 * @param {string[]} classes List of cell classes
	 * @returns Cell containing link
	 */
	getLinkCell(linkText, href, classes=[]) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		link.href = href;
		link.innerText = linkText;
		link.target = "_blank";
		cell.appendChild(link);
		if (classes.length > 0) {
			cell.classList.add(classes);
		}
		return cell;

	}
	/* Cell builders end */

	/**
	 *
	 * @param {ColumnDefinition[]} columns
	 */
	buildRow(columns) {
		const row = document.createElement('tr');
		// row.setAttribute('onclick', `openModal(${this.getId()})`);

		// const classList = ['cursor-pointer'];
		const classList = [];
		for (const column of columns) {
			const identifier = column.getIdentifier();

			try {
				row.appendChild(this.getCellByIdentifier(identifier, classList));
			}
			catch(err) {
				console.error(err);
				row.appendChild(this.getErrorCell());
			}
		}
		this.setRow(row);
		return row;
	}

	/*
		zohoAccountLink : new ColumnDefinition('Zoho Account', 'zohoAccountLink'),
		name : new ColumnDefinition('Name', 'name'),
		phoneNumber : new ColumnDefinition('Phone Number', 'phoneNumber'),
		website : new ColumnDefinition('Website', 'website'),
		accountType : new ColumnDefinition('Account Type', 'accountType'),
		enrollment : new ColumnDefinition('Enrollment', 'enrollment'),
		totalSpent : new ColumnDefinition('Total Spent', 'totalSpent'),
		billingStreet : new ColumnDefinition('Billing Street', 'billingStreet'),
		billingCity : new ColumnDefinition('Billing City', 'billingCity'),
		billingState : new ColumnDefinition('Billing State', 'billingState'),
		billingCode : new ColumnDefinition('Billing Code', 'billingCode'),
		shippingStreet : new ColumnDefinition('Shipping Street', 'shippingStreet'),
		shippingCity : new ColumnDefinition('Shipping City', 'shippingCity'),
		shippingState : new ColumnDefinition('Shipping State', 'shippingState'),
		shippingCode : new ColumnDefinition('Shipping Code', 'shippingCode')
	*/

	/**
	 *
	 * @param {string} identifier
	 */
	getValueByIdentifier(identifier) {
		if (identifier == 'name') {
			return this.getName();
		}
		else if (identifier == 'phoneNumber') {
			return this.getPhoneNumber();
		}
		else if (identifier == 'website') {
			return this.getWebsite();
		}
		else if (identifier == 'accountType') {
			return this.getAccountType();
		}
		else if (identifier == 'enrollment') {
			return this.getEnrollment().toLocaleString();
		}
		else if (identifier == 'totalSpent') {
			return this.getTotalSpent().toLocaleString('en-US', {style: 'currency', currency: 'USD'});
		}
		else if (identifier == 'billingStreet') {
			return this.getBillingStreet();
		}
		else if (identifier == 'billingCity') {
			return this.getBillingCity();
		}
		else if (identifier == 'billingState') {
			if (this.getBillingState() instanceof State) {
				return this.getBillingState().getName();
			}
			else {
				return this.getBillingState();
			}
		}
		else if (identifier == 'billingCode') {
			return this.getBillingCode();
		}
		else if (identifier == 'shippingStreet') {
			return this.getShippingStreet();
		}
		else if (identifier == 'shippingCity') {
			return this.getShippingCity();
		}
		else if (identifier == 'shippingState') {
			if (this.getShippingState() instanceof State) {
				return this.getShippingState().getName();
			}
			else {
				return this.getShippingState();
			}
		}
		else if (identifier == 'shippingCode') {
			return this.getShippingCode();
		}
	}

	/*

	billingCity: "Jackson"
	billingCode: "20101 State Highway 88"
	billingState: "CA"
	billingStreet: "20101 State Highway 88"
	hasUpdates: null
	newCustomer: false
	shippingCity: null
	shippingCode: null
	shippingState: null
	shippingStreet: null
	*/
}
