const CONTROLLER_BASE_URL = "admin_controller.php";
const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';



const COLUMN_DEFINES = {
	email : new ColumnDefinition('Email', 'emailAddress', {isSortable: true}),
	name : new ColumnDefinition('Name', 'name', {isSortable: true}),
	organization : new ColumnDefinition('Organization', 'organization', {isSortable: true}),
	title : new ColumnDefinition('Title', 'title', {isSortable: true}),
	leadSource : new ColumnDefinition('Lead Source', 'leadSource', {isSortable: true}),
	accountType : new ColumnDefinition('Account Type', 'accountType', {isSortable: true}),
	enrollment : new ColumnDefinition('Enrollment', 'enrollment', {isSortable: true}),
	totalSpent : new ColumnDefinition('Total Spent', 'totalSpent', {isSortable: true}),
	city : new ColumnDefinition('City', 'city', {isSortable: true}),
	state : new ColumnDefinition('State', 'state', {isSortable: true}),
	region : new ColumnDefinition('Region', 'region', {isSortable: true}),
	onZohoLeads : new ColumnDefinition('Zoho Leads', 'onZohoLeads', {isSortable: true}),
	onZohoContacts : new ColumnDefinition('Zoho Contacts', 'onZohoContacts', {isSortable: true}),
	onConstantContact : new ColumnDefinition('Constant Contact', 'onConstantContact', {isSortable: true}),
	emailOptOut : new ColumnDefinition('Email Opt Out', 'emailOptOut', {isSortable: true}),
};

const columns = [
	COLUMN_DEFINES.email,
	COLUMN_DEFINES.name,
	COLUMN_DEFINES.organization,
	COLUMN_DEFINES.title,
	COLUMN_DEFINES.emailOptOut,
];

const additionalColumns = [];

const channelPresenceColumns = [
	COLUMN_DEFINES.onZohoLeads,
	COLUMN_DEFINES.onZohoContacts,
	COLUMN_DEFINES.onConstantContact,
];

const filter = {};
const sort = {
	order: 1,
	by: ''
}

let totalMatches = 0;

/**
 * @type {HTMLCollectionOf<HTMLImageElement>}
 */
 const SORT_ICONS = document.getElementsByClassName('sort_icon');

/**
 * @type {HTMLDivElement}
 */
const EMAIL_LIST_TABLE_CONTAINER = document.getElementById('emailListTableContainer');

/**
 * @type {HTMLFormElement}
 */
const SEARCH_INTERFACE = document.getElementById('searchInterface');

const SEARCH_FORM_INPUTS = {
	/**
	 * @type {HTMLSelectElement}
	 */
	region : document.getElementById('search_region'),
	/**
	 * @type {HTMLSelectElement}
	 */
	state : document.getElementById('search_state'),
	/**
	 * @type {HTMLSelectElement}
	 */
	accountType : document.getElementById('search_accountType'),
};

const TABLE_ELEMENTS = {
	/**
	 * @type {HTMLDivElement}
	 */
	container : document.getElementById('contactListTableContainer'),
	/**
	 * @type {HTMLTableElement}
	 */
	table : document.getElementById('contactListTable'),
	/**
	 * @type {HTMLTableSectionElement}
	 */
	thead : document.getElementById('contactListTableHead'),
	/**
	 * @type {HTMLTableSectionElement}
	 */
	tbody : document.getElementById('contactListTableBody')
}

const CONTACT_DICT = {};
/**
 * @type {ConsolidatedContact[]}
 */
const CONTACT_LIST = [];


const REGION_DICT = {};
/**
 * @type {Region[]}
 */
const REGION_ARRAY = [];

const STATE_DICT = {};
/**
 * @type {State[]}
 */
const STATE_ARRAY = [];

init();

async function init() {
	await performSetup();
	// await prepareRegions();
	generateStateOptions('0');

	SEARCH_INTERFACE.hidden = false;
	setLoadingModalVisible(false);
	// const rawContactList = await fetchEmailList();
	// console.log(rawContactList);
	// if (rawContactList) {
	// 	const processedContacts = processContactList(rawContactList);
	// 	console.log(processedContacts);
	// }
}

async function performSetup() {
	const setupData = await getSetupData();
	if (setupData) {
		const regionData = setupData['regions'];
		parseRegionData(regionData);
		const accountTypes = setupData['accountTypes'];
		parseAccountTypes(accountTypes);
	}

	async function getSetupData() {
		const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=emailList_getSetupData`)
			.then (response => response.text());


		try {
			return JSON.parse(responseText);
		}
		catch (err) {
			console.error(err);
			console.log(responseText);
			alert('Failed to parse response from server. Check console for stack trace.');
			return null;
		}
	}

	function parseAccountTypes(accountTypes) {
		for (const accountType of accountTypes) {
			const option = document.createElement('option');
			option.innerText = accountType;
			SEARCH_FORM_INPUTS.accountType.appendChild(option);
		}
	}
}

function parseRegionData(regionDataArray) {
	for (const regionData of regionDataArray) {
		const region = new Region(regionData);
		REGION_DICT[region.getId()] = region;
		REGION_ARRAY.push(region);
	}
	for (const region of REGION_ARRAY) {
		for (const state of region.getStates()) {
			STATE_DICT[state.getId()] = state;
			STATE_ARRAY.push(state);
		}
	}
	generateRegionOptions();

	function generateRegionOptions() {
		SEARCH_FORM_INPUTS.region.innerHTML = '';
		const selectAllOption = document.createElement('option');
		selectAllOption.value = '0';
		selectAllOption.innerText = "All";
		selectAllOption.selected = true;
		SEARCH_FORM_INPUTS.region.appendChild(selectAllOption);
		for (const region of REGION_ARRAY) {
			SEARCH_FORM_INPUTS.region.appendChild(region.getRegionOption());
		}
	}
}


/**
 *
 * @param {string} regionId
 */
function generateStateOptions(regionId=null) {
	if (regionId === null) {
		regionId = SEARCH_FORM_INPUTS.region.value;
	}
	/**
	 * @type {State[]}
	 */
	const states = [];
	if (regionId == '0') {
		for(const region of REGION_ARRAY) {
			for (const state of region.getStates()) {
				states.push(state);
			}
		}
	}
	else {
		/**
		 * @type {Region}
		 */
		const region = REGION_DICT[regionId];
		for (const state of region.getStates()) {
			states.push(state);
		}
	}
	states.sort((a,b) => parseInt(a.getId()) - parseInt(b.getId()));

	SEARCH_FORM_INPUTS.state.innerHTML = '';
	const selectAllOption = document.createElement('option');
	selectAllOption.value = '0';
	selectAllOption.innerText = "All";
	selectAllOption.selected = true;
	SEARCH_FORM_INPUTS.state.appendChild(selectAllOption);
	SEARCH_FORM_INPUTS.state.value = selectAllOption.value;
	for (const state of states) {
		SEARCH_FORM_INPUTS.state.appendChild(state.getStateOption());
	}
}

/**
 *
 * @param {HTMLSelectElement} regionSelect
 */
function setSearchRegion(regionSelect) {
	const regionId = regionSelect.value;
	generateStateOptions(regionId);
}

async function fetchEmailList() {
	const responseText = await fetch(`${CONTROLLER_BASE_URL}?m=emailList_get`)
		.then(response => response.text());

	try {
		return JSON.parse(responseText);
	}
	catch(err) {
		console.error(err);
		console.log(responseText);
		alert('Failed to parse response from server. Check console for stack trace.');
		return null;
	}
}

function processContactList(rawContactList) {
	/**
	 * @type {ConsolidatedContact[]}
	 */
	const contactList = [];
	for (const rawContact of rawContactList) {
		const processedContact = new ConsolidatedContact(rawContact);
		contactList.push(processedContact);
	}
	return contactList;
}

/**
 *
 * @param {Event} event
 */
async function submitSearch(event) {
	event.preventDefault();
	setLoadingModalVisible(true);
	await performSearch();
	setLoadingModalVisible(false);
}

async function performSearch() {
	setLoadingModalVisible(true);
	const payload = buildSearchPayload();

	const params = Object.entries(payload).map(([key, val]) => `${encodeURIComponent(key)}=${encodeURIComponent(val)}`).join('&');
	const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=emailList_get&${params}`,
		{
			method: 'GET',
			headers: {
				'Accept': 'application/json',
				'Content-Type': 'application/json'
			},
			// body: JSON.stringify(payload)
		}
	)
	.catch(error => {
		console.error(error);
		alert("Failed to fetch");
		setLoadingModalVisible(false);
	});
	const resultText = await rawResponse.text();
	try {
		const resultJson = JSON.parse(resultText);
		console.log(resultJson);
		const matches = resultJson['totalCount'];
		renderTotalMatchesValue(matches);
		populateContactDictAndArray(resultJson['processedContacts']);
		createHeadRow();
		createContactRows();
	}
	catch(err) {
		console.error(err);
		console.log(resultText);
		alert('Failed to parse response from server. Check console for stack trace.');
	}
	setLoadingModalVisible(false);
}

/**
 *
 * @param {Event} event
 */
async function exportResults(event) {
	event.preventDefault();
	setLoadingModalVisible(true);
	const payload = buildSearchPayload();

	const params = Object.entries(payload).map(([key, val]) => `${encodeURIComponent(key)}=${encodeURIComponent(val)}`).join('&');
	const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=emailList_export&${params}`)
	.catch(error => {
		console.error(error);
		setLoadingModalVisible(false);
	});
	const resultText = await rawResponse.text();
	try {
		// console.log(resultText);
		const fileName = resultText.substring(resultText.lastIndexOf('/') + 1);
		const link = document.createElement('a');
		link.href = resultText;
		link.target = "_blank";
		link.download = fileName;
		link.click();

		delete link;
	}
	catch(err) {
		console.error(err);
		console.log(resultText);
		alert('Failed to parse response from server. Check console for stack trace.');
	}


	setLoadingModalVisible(false);
}


function buildSearchPayload() {
	additionalColumns.length = 0;

	/**
	 * @type {HTMLCollectionOf<HTMLInputElement|HTMLSelectElement>}
	 */
	const inputs = document.getElementsByClassName('search_input');

	const payload = {};

	for (const input of inputs) {
		if (sort.by && sort.by != '') {
			payload['sortBy'] = sort.by;
			payload['sortOrder'] = (sort.order == 1) ? 'asc' : 'desc';
		}
		if (input.value) {
			const inputName = input.name;
			const inputValue = input.value;
			payload[inputName] = inputValue;

			if (input.value != '0' && COLUMN_DEFINES[inputName] && !columns.includes(COLUMN_DEFINES[inputName]) && !additionalColumns.includes(COLUMN_DEFINES[inputName])) {
				if (inputName == "region") {
					additionalColumns.push(COLUMN_DEFINES['state'])
				}
				else {
					additionalColumns.push(COLUMN_DEFINES[inputName]);
				}
			}
		}
	}
	// console.log(payload);

	return payload;
}

function renderTotalMatchesValue(matches=0) {
	/**
	 * @type {HTMLDivElement}
	 */
	const resultsCountContainer = document.getElementById('resultsCountContainer');
	/**
	 * @type {HTMLSpanElement}
	 */
	const totalMatchesSpan = document.getElementById('totalMatchesSpan');

	totalMatches = matches;

	totalMatchesSpan.innerText = matches.toLocaleString();
	resultsCountContainer.hidden = false;
}

function populateContactDictAndArray(contactDataFromServer) {
	clearContactDictAndArray();
	for (const contactData of contactDataFromServer) {
		const contact = new ConsolidatedContact(contactData);
		CONTACT_DICT[contact.getId()] = contact;
		CONTACT_LIST.push(contact);
	}
}

function clearContactDictAndArray() {
	for (const id in CONTACT_DICT) {
		delete CONTACT_DICT[id];
	}
	CONTACT_LIST.length = 0;
}

function createHeadRow() {
	TABLE_ELEMENTS.thead.innerHTML = '';

	const headRow = document.createElement('tr');

	for (const column of columns) {
		headRow.appendChild(column.getHeaderCellElement());
	}
	for (const column of additionalColumns) {
		headRow.appendChild(column.getHeaderCellElement());
	}
	for (const column of channelPresenceColumns) {
		headRow.appendChild(column.getHeaderCellElement());
	}

	TABLE_ELEMENTS.thead.appendChild(headRow);
}

function createContactRows() {
	TABLE_ELEMENTS.tbody.innerHTML = '';
	const activeColumns = [];
	for (const column of columns) {
		activeColumns.push(column);
	}
	for (const column of additionalColumns) {
		activeColumns.push(column);
	}
	for (const column of channelPresenceColumns) {
		activeColumns.push(column);
	}
	for (const contact of CONTACT_LIST) {
		const row = contact.getContactRow(activeColumns);
		TABLE_ELEMENTS.tbody.appendChild(row);
	}
}

/**
 *
 * @param {string} sortOrder
 */
async function setSort(sortOrder) {
	if (sort.by == sortOrder) {
		sort.order *= -1;
	}
	else {
		sort.by = sortOrder;
	}
	if (totalMatches <= 1000) {
		// If total matches less than or equal to 1000, perform javascript sort
		sortContacts();
		createContactRows();
	}
	else {
		// If more, query to get new data set
		await performSearch();
		sortContacts();
		createContactRows();
	}
}

function sortContacts() {
	CONTACT_LIST.sort((a, b) => sortFunction(a,b));
	for (const icon of SORT_ICONS) {
		if (icon.parentElement.getAttribute('sort_order') != sort.by) {
			icon.src = SORT_NONE_URL;
		} else {
			if (sort.order == 1) {
				icon.src = SORT_ASC_URL;
			} else if (sort.order == -1) {
				icon.src = SORT_DESC_URL;
			}
		}
	}

	/**
	 * Sort function
	 * @param {ConsolidatedContact} a First compared contact
	 * @param {ConsolidatedContact} b Second compared contact
	 * @returns int determining sort order
	 */
	function sortFunction(a,b) {
		const aVal = a.getValueByIdentifier(sort.by);
		const bVal = b.getValueByIdentifier(sort.by);
		if (typeof aVal == 'boolean') {
			const aValInt = (aVal) ? 1 : 0;
			const bValInt = (bVal) ? 1 : 0;
			if(aValInt < bValInt) {
				const result = -1 * sort.order;
				return result;
			}
			if(aValInt > bValInt) {
				const result = 1 * sort.order;
				return result;
			}
		}
		if(isNaN(aVal)) {
			if(aVal < bVal) {
				const result = -1 * sort.order;
				return result;
			}
			if(aVal > bVal) {
				const result = 1 * sort.order;
				return result;
			}
		} else {
			if(parseInt(aVal) < parseInt(bVal)) {
				const result = -1 * sort.order;
				return result;
			}
			if(parseInt(aVal) > parseInt(bVal)) {
				const result = 1 * sort.order;
				return result;
			}
		}
	}
}
