/**
 * @type {HTMLFormElement}
 */
const returnsForm = document.getElementById('returnsForm');
/**
 * @type {HTMLButtonElement}
 */
const submitButton = document.getElementById('btnSubmit');

/**
 * @type {HTMLTableRowElement}
 */
const inputHeaderRow = document.getElementById('inputHeaderRow');
/**
 * @type {HTMLTableDataCellElement}
 */
const enteredSPNumDisplay = document.getElementById('displaySPNums');

/**
 * @type {HTMLInputElement}
 */
const scanInput = document.getElementById('scanInput');
/**
 * @type {HTMLInputElement}
 */
const orderNumInput = document.getElementById('orderNumInput');


/**
 * @type {HTMLTableRowElement}
 */
const serialInputRow = document.getElementById('serialInputRow');
/**
 * @type {HTMLTextAreaElement}
 */
const serialTextAreaInput = document.getElementById('serialTextAreaInput');

/**
 * @type {HTMLTableRowElement}
 */
const singleSPInputRow = document.getElementById('singleSPInputRow');
/**
 * @type {HTMLInputElement}
 */
const singleSPInput = document.getElementById('singleSPInput');

/**
 * @type {HTMLTableRowElement}
 */
const spSelectRow = document.getElementById('spSelectRow');
/**
 * @type {HTMLSelectElement}
 */
const spNumbersInput = document.getElementById('spNumbersInput');

const CONTROLLER_BASE_URL = 'admin_controller.php';
let orderNum = '';
let enteredSPs = 0;
let trackingSet = false;

init();

function init() {
	toggleElementDisabled(submitButton, true);
	addScanInputListeners(scanInput);
	addOrderInputListeners(orderNumInput);
	addTextAreaListeners(serialTextAreaInput);
	focusElement(scanInput);
}



// scanInput.addEventListener('change', (event) => {
// 	toggleElementHidden(serialInputRow);
// 	toggleElementHidden(spSelectRow);
// })


/**
 *
 * @param {HTMLInputElement} scanInput The scan input object on which to attach the event listeners.
 */
function addScanInputListeners(scanInput) {
	scanInput.addEventListener('change', (_event) => {
		val = scanInput.value;
		if (trackingSet) {
			resetForm();
		} else {
			trackingSet = true;
		}
		scanInput.value = val;
		const inputType = getInputType(scanInput.value);
		if (inputType == 'ups') {
			getOrderNumFromTracking(scanInput.value);
		} else if (inputType == 'order') {
			orderNum = scanInput.value;
			showOrderInputs();
			getSpNumsFromOrderNum(orderNum);
		}
	});
}

/**
 *
 * @param {HTMLInputElement} orderNumInput Order input
 */
function addOrderInputListeners(orderNumInput) {
	orderNumInput.addEventListener('change', (_event) => {
		orderNum = orderNumInput.value;
		getSpNumsFromOrderNum(orderNum);
	});
}

/**
 *
 * @param {HTMLTextAreaElement} serialTextAreaInput
 */
function addTextAreaListeners(serialTextAreaInput) {
	const spPattern = /^(SP\d{6})$/;
	serialTextAreaInput.addEventListener('change', (_event) => {
		enteredNum = serialTextAreaInput.value.trim().toUpperCase();
		if (spPattern.test(enteredNum)) {
			confirmSpNumExists(enteredNum);
		} else if (enteredNum.length > 6) {
			getSpNumFromSerial(enteredNum);
		}
	});
	serialTextAreaInput.addEventListener('keyup', (event) => {
		if (event.key == 'Enter') {
			enteredNum = serialTextAreaInput.value.trim().toUpperCase();
			if (spPattern.test(enteredNum)) {
				confirmSpNumExists(enteredNum);
			} else if (enteredNum.length > 6) {
				getSpNumFromSerial(enteredNum);
			}
		}
	});
}

function getOrderNumFromTracking(trackingNum) {
	let postStr = 'tracking_no=' + trackingNum;
	let url = CONTROLLER_BASE_URL + '?m=get_order_num_from_tracking';
	makeAjaxRequest(url, postStr, 'POST', returnOrderNum)
}

function returnOrderNum() {
	if (http_request.readyState == 4) {
		switch(http_request.status) {
			case 200:
				if (http_request.responseText == 'false') {
					alert('Tracking number has no order number associated with it. Please double-check tracking number, or manually enter order number.');
					showOrderInputs();
					toggleElementReadonly(orderNumInput, false);
					orderNumInput.focus();
					return false;
				}
				const response = http_request.response;
				const data = JSON.parse(response);
				orderNum = data['orderNumber'];
				showOrderInputs();
				getSpNumsFromOrderNum(orderNum);
				// alert('AJAX success! ' + orderNum);
				// orderNum = http_request['orderNumber'];
				break;
			default:
				alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
		}
	}
}

function getSpNumsFromOrderNum(orderNum) {
	let postStr = 'order_no=' + orderNum;
	let url = CONTROLLER_BASE_URL + '?m=get_sp_nums_from_order_num';
	makeAjaxRequest(url, postStr, 'POST', returnSpNums);
}
// 1Z0000000000000000 -> 479642 for single match
// 1Z0000000000000001 -> anynumber for no match
// 1Z0000000000000002 -> 500100 for multimatch
function returnSpNums() {
	if (http_request.readyState == 4) {
		switch(http_request.status) {
			case 200:
				if (http_request.responseText == 'false') {
					revealElement(serialInputRow);
					toggleElementReadonly(serialInputRow, false);
					revealElement(inputHeaderRow);
					toggleElementDisabled(singleSPInput, true);
					toggleElementDisabled(spNumbersInput, true);
					serialTextAreaInput.focus();
					return false;
				}
				const response = http_request.response;
				const data = JSON.parse(response);
				if (data.length === 1) {
					toggleElementDisabled(submitButton, false);
					singleSPInput.value = data[0]['oNum'];
					toggleElementReadonly(singleSPInput, true);
					toggleElementDisabled(spNumbersInput, true);
					toggleElementDisabled(serialTextAreaInput, true);
					revealElement(singleSPInputRow);
					submitForm();
				} else if (data.length > 1) {
					spNumbersInput.size = data.length;
					for (const entry of data) {
						const option = document.createElement('option');
						option.text = entry['oNum'] + ' - ' + entry['serialNumber'];
						option.value = entry['oNum'];
						spNumbersInput.appendChild(option);
					}
					revealElement(spSelectRow);
					toggleElementReadonly(spNumbersInput, false);
					toggleElementDisabled(submitButton, false);
					toggleElementDisabled(singleSPInput, true);
					toggleElementDisabled(serialTextAreaInput, true);
				}
				break;
			default:
				alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
		}
	}
}

function getSpNumFromSerial(serialNum) {
	let postStr = 'serial_no=' + serialNum;
	let url = CONTROLLER_BASE_URL + '?m=get_sp_num_from_serial';
	makeAjaxRequest(url, postStr, 'POST', returnSpNum)
}

function confirmSpNumExists(spNum) {
	let postStr = 'sp_no=' + spNum;
	let url = CONTROLLER_BASE_URL + '?m=confirm_sp_exists';
	makeAjaxRequest(url, postStr, 'POST', returnSpNum);
}

function returnSpNum() {
	if (http_request.readyState == 4) {
		switch(http_request.status) {
			case 200:
				if (http_request.responseText == 'false') {
					alert('SP number not found. Check serial number and try again.');
				} else {
					const response = http_request.response;
					const data = JSON.parse(response);
					spNum = data['oNum'];
					const spNumSpan = document.createElement('span');
					spNumSpan.classList.add('spNum');
					spNumSpan.id = 'spNum_' + enteredSPs;
					spNumSpan.innerText = spNum;
					const delSPNumSpan = document.createElement('span');
					delSPNumSpan.innerText = ' [delete]';
					const hdnSPInput = document.createElement('input');
					hdnSPInput.type = 'hidden';
					hdnSPInput.name = 'spNumbers[]';
					hdnSPInput.value = spNum;
					const p = document.createElement('p');
					p.style.margin = 0;
					p.id = 'spNumContainer_' + enteredSPs;
					delSPNumSpan.addEventListener('click', (_event) => {
						p.remove();
					});
					enteredSPNumDisplay.appendChild(p);
					p.appendChild(spNumSpan);
					p.appendChild(delSPNumSpan);
					p.appendChild(hdnSPInput);
					serialTextAreaInput.value = '';
					enteredSPs++;
					toggleElementDisabled(submitButton, false);
				}
				break;
			default:
				alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
		}
	}
}

function showOrderInputs() {
	orderNumInput.value = orderNum;
	toggleElementHidden(orderNumInput, false);
	toggleElementHidden(document.getElementById('orderNumColumnHead'), false);
}


/**
 *
 * @param {HTMLElement} element Element to be revealed
 */
function revealElement(element) {
	toggleElementHidden(element, false);
}

/**
 *
 * @param {string} inputStr The number scanned in the scanner input field.
 * @returns {string} 'ups' if a UPS tracking number, else 'order'.
 */
function getInputType(inputStr) {
	const upsPattern = /^(1Z.{16})$/;
	if (upsPattern.test(inputStr)) {
		return 'ups';
	} else {
		return 'order';
	}
}


/**
 *
 * @param {HTMLInputElement} element Input to be focused
 */
function focusElement(element) {
	element.focus();
}

/**
 *
 * @param {HTMLElement} element Element to be hidden or shown
 * @param {boolean} val Value to set hidden to. Defaults to opposite of current value. true indicates readonly, false indicates enabled.
 */
function toggleElementHidden(element, val=!element.hidden) {
	element.hidden = val;
}

/**
 *
 * @param {HTMLInputElement} element element to be toggled
 * @param {boolean} val Value to set disabled to. Defaults to opposite of current value. true indicates disabled, false indicates enabled.
 */
function toggleElementDisabled(element, val=!element.disabled) {
	element.disabled = val;
}

/**
 *
 * @param {HTMLInputElement} element element to be toggled
 * @param {boolean} val Value to set readonly to. Defaults to opposite of current value. true indicates readonly, false indicates enabled.
 */
function toggleElementReadonly(element, val=!element.readOnly) {
	element.readOnly = val;
}

function resetForm() {
	returnsForm.reset();
	for (const node of spNumbersInput.childNodes) {
		node.remove();
	}
	for (const node of enteredSPNumDisplay.childNodes) {
		node.remove();
	}
	toggleElementHidden(inputHeaderRow, true);
	toggleElementHidden(serialInputRow, true);
	toggleElementHidden(spSelectRow, true);
	toggleElementHidden(singleSPInputRow, true);
	toggleElementHidden(inputHeaderRow, true);
	toggleElementHidden(inputHeaderRow, true);
}

function submitForm() {
	returnsForm.submit();
}
