const CONTROLLER_BASE_URL = 'admin_controller.php';

const orderInputs = document.getElementsByClassName('order_number_input');
const spInputs = document.getElementsByClassName('sp_number_input');
const trackingNumberInputs = document.getElementsByClassName('tracking_number_input');
/**
 * @type {HTMLCollectionOf<HTMLInputElement>}
 */
const shippingDateInputs = document.getElementsByClassName('shipping_date_input');
const shippingMethodSelects = document.getElementsByClassName('shipping_method_select');
const checkoutForm = document.getElementById('checkoutForm');
const openZohoInvoices = [];
const otherValidZohoInvoices = [];

const expectedItems = {};
const enteredItems = {};


initListeners();

async function initListeners() {
	if (checkoutForm.addEventListener) {
		checkoutForm.addEventListener("submit", function(evt) {
			if (document.activeElement.classList.contains('order_number_input') || !validate()) {
				evt.preventDefault();
			}
		}, true);
	}
	else {
		checkoutForm.attachEvent('onsubmit', function(evt){
			if (document.activeElement.classList.contains('order_number_input') || !validate()) {
				evt.preventDefault();
			}
		});
	}
	await getOpenZohoInvoices();
	initOrderInputListeners(orderInputs);
	initMultiInputListeners(spInputs, 'sp');
	initMultiInputListeners(trackingNumberInputs, 'tracking');
	orderInputs[0].focus();
}


/**
 *
 * @param {HTMLTextAreaElement[]} inputs Collection of text area inputs
 * @param {string} inputType String of type of inputs
 */
function initMultiInputListeners(inputs, inputType) {
	const otherType = (inputType == 'sp') ? 'tracking' : 'sp';
	for (const input of inputs) {
		input.addEventListener('keydown', (event) => {
			if (event.key === 'Tab') {
				textAreaListenerFunction(input);
			}
		});
		input.addEventListener('keyup', (event) => {
			if (event.key === 'Enter') {
				textAreaListenerFunction(input);
			}
		});
		input.addEventListener('focus', (_event) => {
			if (input.value.length > 0) {
				input.value = input.value.split('\n').filter(entry => entry.length > 0).join('\n') + '\n';
			}
		});
		input.addEventListener('blur', (_event) => {
			input.value = input.value.split('\n').filter(entry => entry.length > 0).join('\n');
		});
	}

	/**
	 *
	 * @param {HTMLTextAreaElement} input
	 */
	function textAreaListenerFunction(input) {
		const entries = input.value.trim().split('\n');
		const lastEntry = getLastEntry(entries);
		const lastEntryType = evaluateEntryType(lastEntry);
		const row = parseInt(input.getAttribute('inputrow'));
		const inputs = {
			currentRow : {
				/**
				 * @type {HTMLInputElement}
				 */
				orderNumber : document.getElementById(`order_number_${row}`),
				/**
				 * @type {HTMLTextAreaElement}
				 */
				spNumbers : document.getElementById(`sp_numbers_${row}`),
				/**
				 * @type {HTMLTextAreaElement}
				 */
				trackingNumbers : document.getElementById(`tracking_numbers_${row}`)
			},
			nextRow : {
				/**
				 * @type {HTMLInputElement}
				 */
				orderNumber : document.getElementById(`order_number_${row + 1}`),
				/**
				 * @type {HTMLTextAreaElement}
				 */
				spNumbers : document.getElementById(`sp_numbers_${row + 1}`),
				/**
				 * @type {HTMLTextAreaElement}
				 */
				trackingNumbers : document.getElementById(`tracking_numbers_${row + 1}`)
			}
		};
		if (lastEntryType) {
			entries.pop();
			input.value = [...new Set(entries)].join('\n');
			if (lastEntryType == 'sp') {
				inputs.currentRow.spNumbers.value = [...new Set(inputs.currentRow.spNumbers.value.trim().split('\n').filter(entry => entry.trim().length > 0).concat(lastEntry))].join('\n').concat('\n');
				setRows(inputs.currentRow.spNumbers);
				updateEnteredItems(inputs.currentRow.spNumbers);
				inputs.currentRow.spNumbers.focus();
			}
			else if (lastEntryType == 'serial') {
				getSpNumFromSerial(lastEntry, inputs.currentRow.spNumbers);
			}
			else if (lastEntryType == 'tracking') {
				inputs.currentRow.trackingNumbers.value = [...new Set(inputs.currentRow.trackingNumbers.value.trim().split('\n').filter(entry => entry.trim().length > 0).concat(lastEntry))].join('\n').concat('\n');
				if (inputs.currentRow.trackingNumbers.value.trim().split('\n').length == 1) {
					getOrderFromTrackingNum(inputs.currentRow.trackingNumbers);
				}
				setRows(inputs.currentRow.trackingNumbers);
				inputs.currentRow.trackingNumbers.focus();
			}
			else if (lastEntryType == 'order') {
				const orderInput = getNextEmptyOrderInput(row);
				if (orderInput) {
					if (isMtsOrderNumber(lastEntry.trim())) {
						handleMtsOrders(lastEntry.trim(), orderInput);
					}
					orderInput.value = lastEntry.trim();
				}
			}
		}
		updateRunningCounter(inputs.currentRow.spNumbers);
	}
}

/**
 *
 * @param {HTMLTextAreaElement} input
 */
function updateEnteredItems(input) {
	const values = input.value.trim().split('\n').join(',');
	const url = `${CONTROLLER_BASE_URL}?m=ecomm_checkout_get_asset_number_counts`;
	const postStr = `oNums=${values}`;
	console.log(postStr);
	makeAjaxRequest(url, postStr, 'POST', returnAssetCounts);

	function returnAssetCounts() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					try {
						const data = JSON.parse(http_request.response);
						const row = input.getAttribute('inputrow');
						/**
						 * @type {HTMLParagraphElement}
						 */
						const enteredItemsContainer = document.getElementById(`entered_items_container_${row}`)
						console.log(data);
						if (data && typeof data == 'object') {
							const items = (data['items']) ? data['items'] : [];
							if (items) {
								enteredItems[row] = {};
								for (const item of items) {
									enteredItems[row][item['asset']] = {
										'sku' : item['asset'],
										'quantity' : parseInt(item['asset_count'])
									};
								}
								enteredItemsContainer.innerHTML = '<h3 style="margin:0;">Entered Items:</h3>';
								for (const item in enteredItems[row]) {
									enteredItemsContainer.innerHTML += `<h4 style="margin:0;">${enteredItems[row][item]['sku']} - ${enteredItems[row][item]['quantity']}</h4>`;
								}
								enteredItemsContainer.hidden = false;
							}
						}
					}
					catch (e) {
						console.error(e);
					}
					setLoadingModalVisible(false);
					break;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
					setLoadingModalVisible(false);
					return false;
			}
		}
	}
}

function getNextEmptyOrderInput(row) {
	for (let i = row + 1; i < 25; i++) {
		/**
		 * @type {HTMLInputElement}
		 */
		const orderInput = document.getElementById(`order_number_${i}`);
		if (!orderInput.value || orderInput.value.trim() == '') {
			return orderInput;
		}
	}
	return false;
}

/**
 *
 * @param {string} orderNum MTS order number to be handled
 * @param {HTMLInputElement} orderInput Order input on which logic is performed
 */
function handleMtsOrders(orderNum, orderInput) {
	const row = parseInt(getCurrentRow(orderInput.id));
	const tracking = document.getElementById('tracking_numbers_' + row);
	if (!openZohoInvoices.includes(orderNum) && !otherValidZohoInvoices.includes(orderNum)) {
		getZohoInvoiceByMTSNumber(orderNum, orderInput);
	} else {
		getTrackingNumFromOrder(orderNum.trim(), tracking);
	}
	document.getElementById('shipping_method_' +  row).disabled = false;
	document.getElementById('sp_numbers_' + row).focus();
}

/**
 *
 * @param {HTMLInputElement[]} orderInputs
 */
function initOrderInputListeners(orderInputs) {
	for (const orderInput of orderInputs) {
		orderInput.addEventListener('keydown', (event) => {
			if (event['key'] === 'Tab') {
				event.preventDefault();
				const row = getCurrentRow(orderInput.id);
				const order = orderInput.value;
				if (isMtsOrderNumber(order)) {
					handleMtsOrders(order, orderInput);
					document.getElementById('sp_numbers_' + row).focus();
				}
				else if (isUpsNumber(order)) {
					const trackingField = document.getElementById(`tracking_numbers_${row}`);
					orderInput.value = '';
					trackingField.value = order;
					getOrderFromTrackingNum(trackingField);
				} else {
					document.getElementById('sp_numbers_' + row).focus();
				}
			}
		});
		orderInput.addEventListener('keyup', (event) => {
			if (event['key'] === 'Enter') {
				const row = getCurrentRow(orderInput.id);
				const order = orderInput.value;
				if (isMtsOrderNumber(order)) {
					handleMtsOrders(order, orderInput);
				}
				else if (isUpsNumber(order)) {
					const trackingField = document.getElementById(`tracking_numbers_${row}`);
					orderInput.value = '';
					trackingField.value = order;
					getOrderFromTrackingNum(trackingField);
				} else {
					getExpectedData(orderInput);
					document.getElementById('sp_numbers_' + row).focus();
				}
			}
		});
	}
}


/**
 *
 * @param {string} str Last entry to be parsed
 * @returns {string} Type of entry - tracking, sp, or order
 */
function evaluateEntryType(str) {
	const upsPattern = /^(1Z.{16})$/;
	const fedexPattern = /^(\d{12}|.{15})$/;
	const uspsPattern = /^(.{22})$/;
	const spPattern = /^(SP\d{6})$/;
	const appleSerialPattern = /^(.{12})$/;
	const mtsOrderPattern = /^(MTS\d{5,6})$/;
	const ebayPattern = /^(\d{12}-\d{7})$/;
	const ebayPattern2 = /^(\d{12}-\d{13})$/;
	const backmarketPattern = /^(\d{6})$/;
	const amazonPattern = /^(\d{3}-\d{7}-\d{7})$/;
	const neweggPattern = /^(\d{9})$/;
	const chanxPattern = /^(\d{8})$/;
	const generalPOPattern = /^(PO.+)$/;
	if (!str) {
		return null;
	} else if (upsPattern.test(str.trim()) || fedexPattern.test(str.trim()) || uspsPattern.test(str.trim())) {
		return 'tracking';
	} else if (spPattern.test(str.trim())) {
		return 'sp';
	} else if (appleSerialPattern.test(str.trim())) {
		return 'serial';
	} else if (ebayPattern.test(str.trim()) ||
				ebayPattern2.test(str.trim()) ||
				backmarketPattern.test(str.trim()) ||
				amazonPattern.test(str.trim()) ||
				neweggPattern.test(str.trim()) ||
				chanxPattern.test(str.trim()) ||
				mtsOrderPattern.test(str.trim()) ||
				generalPOPattern.test(str.trim())
				) {
		return 'order';
	} else {
		return 'serial';
	}
}

function isTrackingNumber(str) {
	return (evaluateEntryType(str) == 'tracking');
}

function isUpsNumber(str) {
	const pattern = /^(1Z.{16})$/;
	return pattern.test(str.trim());
}

function isSpNumber(str) {
	const pattern = /^(SP\d{6})$/;
	return pattern.test(str.trim());
}

function isMtsOrderNumber(str) {
	const pattern = /^(MTS\d{5,6})$/;
	return pattern.test(str.trim());
}

/**
 *
 * @param {string} str The element ID to check
 * @returns {string} The current row, as a string
 */
function getCurrentRow(str) {
	return str.substring(str.lastIndexOf('_') + 1);
}

/**
 *
 * @param {string[]} arr Array of strings to get the last entry from
 * @returns {string} Last entry element
 */
function getLastEntry(arr) {
	return arr[arr.length - 1];
}

/**
 *
 * @param {HTMLTextAreaElement} element
 * @param {number} numRows
 */
function addRows(element, numRows = 1) {
	element.rows += numRows;
}

/**
 *
 * @param {HTMLTextAreaElement} element
 * @param {number} numRows
 */
function setRows(element, numRows=0) {
	if (numRows == 0) {
		numRows = Math.max(getNumEntries(element) + 1, 2);
	}
	element.rows = numRows;
}

/**
 *
 * @param {string} str The SP number to test
 * @returns {boolean} true if a valid SP number, else false
 */
function isValidSPNumber(str) {
	const spPattern = /^(SP\d{6})$/;
	return (spPattern.test(str) || str == '');
}


function validate() {
	let countMessage = '';
	// let expectedItemMessage = '';
	let errorMessage = '';
	let foundOrderInput = false;
	for (const orderInput of orderInputs) {
		if (orderInput.value && orderInput.value.length > 0) {
			foundOrderInput = true;
		}
		const currentRow = getCurrentRow(orderInput.id);
		const count = getCount(currentRow);
		const currentSpInput = document.getElementById('sp_numbers_' + currentRow);
		const currentShipMethodInput = document.getElementById('shipping_method_' + currentRow);
		const spEntries = (currentSpInput.value) ? currentSpInput.value.trim().split('\n') : [];
		if (orderInput.value && orderInput.value.length > 0 && isMtsOrderNumber(orderInput.value) && !(openZohoInvoices.includes(orderInput.value) || otherValidZohoInvoices.includes(orderInput.value))) {
			errorMessage += `Invalid MTS order entered: ${orderInput.value} on row ${currentRow}\n`;
		}
		if (orderInput.value && orderInput.value.length > 0 && (!currentSpInput.value || currentSpInput.value.length < 1)) {
			// errorMessage += 'No SP numbers entered for order ' + orderInput.value + ' on row ' + currentRow + '\n';
		}
		for (const spEntry of spEntries) {
			if (!isValidSPNumber(spEntry)) {
				errorMessage += 'Invalid SP number found for order ' + orderInput.value + ' on row ' + currentRow + '\n';
			}
		}
		if (isMtsOrderNumber(orderInput.value) && (!currentShipMethodInput.value || currentShipMethodInput.value == 'Select')) {
			errorMessage += 'Ship method not selected for row ' + currentRow + '\n';
		}
		if (orderInput.value) countMessage += `${orderInput.value} has ${count} ${(count == '1') ? 'device' : 'devices'}.\n`;
		if (expectedItems[currentRow]) {
			let expectedItemMessage = `${orderInput.value} expects the following item(s):\n`;
			for (const item in expectedItems[currentRow]) {
				expectedItemMessage += `SKU: ${expectedItems[currentRow][item]['sku']} - Qty: ${expectedItems[currentRow][item]['quantity']}\n`;
			}
			countMessage += expectedItemMessage;
			if (enteredItems[currentRow]) {
				let enteredItemMessage = `Entered the following item(s):\n`;
				for (const item in enteredItems[currentRow]) {
					enteredItemMessage += `SKU: ${enteredItems[currentRow][item]['sku']} - Qty: ${enteredItems[currentRow][item]['quantity']}\n`;
				}
				countMessage += enteredItemMessage;
			}
		}

	}
	if (!foundOrderInput) {
		errorMessage += 'No order numbers entered.\n';
	}
	if (errorMessage.length > 0) {
		alert(errorMessage);
		return false;
	} else {
		countMessage += '\nConfirm that these counts are correct.\n\nSubmit?';
		if (confirm(countMessage)) {
			setLoadingModalVisible(true);
			return true;
		} else {
			return false
		}
	}
}

/**
 * Gets the tracking number(s) associated with a given order number
 * @param {string} orderNum Order num to use to query DB
 * @param {HTMLTextAreaElement} trackingNumberInputField The tracking number field into which to place any returned values
 */
function getTrackingNumFromOrder(orderNum, trackingNumberInputField) {
	setLoadingModalVisible(true);
	const postStr = 'order_no=' + orderNum;
	const url = CONTROLLER_BASE_URL + '?m=get_tracking_num_from_order';
	makeAjaxRequest(url, postStr, 'POST', returnTrackingNum)

	function returnTrackingNum() {
		if (http_request.readyState == 4) {
			switch(http_request.status) {
				case 200:
					setLoadingModalVisible(false);
					if (http_request.responseText == 'false') {
						return false;
					}
					const response = http_request.response;
					const data = JSON.parse(response);
					const trackingNums = [];
					for (const datum of data) {
						trackingNums.push(datum['trackingNumber']);
					}
					if (isUpsNumber(trackingNums[0])) {
						const select = document.getElementById('shipping_method_' + getCurrentRow(trackingNumberInputField.id));
						select.value = "UPS";
					}
					trackingNumberInputField.value = trackingNums.join('\n');
					setRows(trackingNumberInputField);
					return true;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
					return false;
			}
		}
	}
}

/**
 * Gets order numbers associated with a given tracking number.
 * @param {HTMLTextAreaElement} trackingNumInputField Field for tracking number input
 */
function getOrderFromTrackingNum(trackingNumInputField) {
	const trackingNum = trackingNumInputField.value.trim().split('\n')[0].trim();
	if (trackingNum && isUpsNumber(trackingNum)) {
		setLoadingModalVisible(true);
		const postStr = `tracking_num=${trackingNum}`;
		const url = `${CONTROLLER_BASE_URL}?m=checkout_get_order_num_from_tracking`;
		makeAjaxRequest(url, postStr, 'POST', returnOrderInfo);
	}

	function returnOrderInfo() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					try {
						const data = JSON.parse(http_request.response);
						const row = trackingNumInputField.getAttribute('inputrow');
						/**
						 * @type {HTMLInputElement}
						 */
						const orderInput = document.getElementById(`order_number_${row}`);
						/**
						 * @type {HTMLTextAreaElement}
						 */
						const spInput = document.getElementById(`sp_numbers_${row}`)
						if (data && typeof data == 'object') {
							const orderNum = data['orderNum'];
							const trackingNums = [];
							if (data['trackingNums']) {
								for (const num of data['trackingNums']) {
									trackingNums.push(num['tracking_number']);
								}
								trackingNumInputField.value = trackingNums.join('\n') + '\n';
								if (isUpsNumber(trackingNums[0])) {
									const select = document.getElementById(`shipping_method_${row}`);
									select.value = "UPS";
								}
							}
							if (data['expectedItems']) {
								/**
								 * @type {HTMLParagraphElement}
								 */
								const expectedItemsContainer = document.getElementById(`expected_items_container_${row}`);
								/**
								 * @type {HTMLParagraphElement}
								 */
								const enteredItemsContainer = document.getElementById(`entered_items_container_${row}`);
								/**
								 * @type {HTMLParagraphElement}
								 */
								const counterContainer = document.getElementById(`counter_container_${row}`);
								expectedItems[row] = {};
								for (const item of data['expectedItems']) {
									expectedItems[row][item['sku']] = {
										'sku' : item['sku'],
										'quantity' : parseInt(item['quantity']),
										'isInternal' : parseInt(item['is_internal_sku'])
									};
								}
								expectedItemsContainer.innerHTML = '<h3 style="margin:0;">Expected Items:</h3>';
								for (const item in expectedItems[row]) {
									expectedItemsContainer.innerHTML += `<h4 style="margin:0;">${expectedItems[row][item]['sku']} - ${expectedItems[row][item]['quantity']}</h4>`;
									if (expectedItems[row][item]['description']) {
										expectedItemsContainer.innerHTML += `<h4 style="margin:0;">${expectedItems[row][item]['description']}</h4>`;
									}
								}
								expectedItemsContainer.hidden = false;
								enteredItemsContainer.hidden = false;
								counterContainer.hidden = true;
							}
							orderInput.value = orderNum;
							spInput.focus();
						}
					}
					catch (e) {
						console.error(e);
					}
					setLoadingModalVisible(false);
					return true;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
					setLoadingModalVisible(false);
					return false;
			}
		}
	}
}

/**
 * Gets a SP number from DB based on entered serial number. Returns most recent SP number associated with that serial.
 * @param {string} serialNum Serial number to check
 * @param {HTMLTextAreaElement} spNumberFieldInput SP number to put the resulting number into
 */
function getSpNumFromSerial(serialNum, spNumberFieldInput) {
	setLoadingModalVisible(true);
	const postStr = 'serial_no=' + serialNum;
	const url = CONTROLLER_BASE_URL + '?m=get_sp_num_from_serial';
	makeAjaxRequest(url, postStr, 'POST', returnSpNum)
	function returnSpNum() {
		if (http_request.readyState == 4) {
			switch(http_request.status) {
				case 200:
					if (http_request.responseText == 'false') {
						alert('SP number not found. Check serial number and try again.');
					} else {
						spNumberFieldInput.focus();
						const response = http_request.response;
						const data = JSON.parse(response);
						const spNum = data['oNum'];
						spNumberFieldInput.value = spNumberFieldInput.value.split('\n').filter(entry => entry.length > 0).concat(spNum).join('\n') + '\n';
						setRows(spNumberFieldInput);
						spNumberFieldInput.focus();
					}
					setLoadingModalVisible(false);
					updateEnteredItems(spNumberFieldInput);
					break;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
			}
		}
	}
}

/**
 *
 * @param {HTMLInputElement} orderInput
 */
function getExpectedData(orderInput) {
	setLoadingModalVisible(true);
	const postStr = 'orderNumber=' + orderInput.value;
	const url = `${CONTROLLER_BASE_URL}?m=get_ecomm_open_order_expected_data`;
	makeAjaxRequest(url, postStr, 'POST', returnData);
	function returnData() {
		if (http_request.readyState == 4) {
			switch(http_request.status) {
				case 200:
					console.log(http_request.response);
					const row = orderInput.getAttribute('inputrow');
					try {
						const data = JSON.parse(http_request.response);
						const items = data['items'];
						/**
						 * @type {string[]}
						 */
						const trackingNums = data['trackingNumbers'];
						if (trackingNums) {
							/**
							 * @type {HTMLTextAreaElement}
							 */
							const trackingNumberField = document.getElementById(`tracking_numbers_${row}`);
							trackingNumberField.value = trackingNums.join('\n');
						}
						console.log(items);
						console.log(trackingNums);
						/**
						 * @type {HTMLParagraphElement}
						 */
						const expectedItemsContainer = document.getElementById(`expected_items_container_${row}`);
						/**
						 * @type {HTMLParagraphElement}
						 */
						const enteredItemsContainer = document.getElementById(`entered_items_container_${row}`);
						/**
						 * @type {HTMLParagraphElement}
						 */
						const counterContainer = document.getElementById(`counter_container_${row}`);
						expectedItems[row] = {};
						for (const item of items) {
							expectedItems[row][item['sku']] = {
								'sku' : item['sku'],
								'quantity' : parseInt(item['quantity']),
								'isInternal' : parseInt(item['is_internal_sku'])
							};
						}
						expectedItemsContainer.innerHTML = '<h3 style="margin:0;">Expected Items:</h3>';
						for (const item in expectedItems[row]) {
							expectedItemsContainer.innerHTML += `<h4 style="margin:0;">${expectedItems[row][item]['sku']} - ${expectedItems[row][item]['quantity']}</h4>`;
							if (expectedItems[row][item]['description']) {
								expectedItemsContainer.innerHTML += `<h4 style="margin:0;">${expectedItems[row][item]['description']}</h4>`;
							}
						}
						expectedItemsContainer.hidden = false;
						enteredItemsContainer.hidden = false;
						counterContainer.hidden = true;
						document.getElementById('sp_numbers_' + row).focus();
					}
					catch (e) {
						console.error(e);
					}
					setLoadingModalVisible(false);
					break;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
			}
		}
	}
}

async function getOpenZohoInvoices() {
	setLoadingModalVisible(true);
	const postStr = "zoho_search=open";
	const url = CONTROLLER_BASE_URL + '?m=get_zoho_invoices'
	return makeAjaxRequest(url, postStr, 'POST', returnOpenZohoInvoices);
	function returnOpenZohoInvoices() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					setLoadingModalVisible(false);
					if (http_request.responseText == '') {
						alert ('Error with AJAX request. Check network connectivity and try again. This may be due to Zoho API throttling.');
						return false;
					}
					if (http_request.responseText == 'false' || http_request.responseText == 'No ContentNO CONTENT204null') {
						alert('Invoice not found. Check invoice number and try again.');
						return false;
					} else {
						const invoices = JSON.parse(http_request.response);
						for (const invoice of invoices) {
							openZohoInvoices.push(invoice['MTS_Quote_Number']);
						}
						return true;
					}
					break;
				default:
					setLoadingModalVisible(false);
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
					return false;
			}
		}
	}
}

/**
 *
 * @param {string} mtsNumber MTS number to search for
 * @param {HTMLInputElement} input Order input
 */
function getZohoInvoiceByMTSNumber(mtsNumber, input) {
	setLoadingModalVisible(true);
	const postStr = `zoho_search=((MTS_Quote_Number:equals:${mtsNumber}))`;
	const url = CONTROLLER_BASE_URL + '?m=get_zoho_invoices';
	makeAjaxRequest(url, postStr, 'POST', returnZohoInvoice);
	function returnZohoInvoice() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					setLoadingModalVisible(false);
					if (http_request.responseText == '') {
						alert ('Error with AJAX request. Check network connectivity and try again. This may be due to Zoho API throttling.');
						return false;
					}
					if (http_request.responseText == 'false' || http_request.responseText == 'No ContentNO CONTENT204null') {
						alert('Invoice not found. Check invoice number and try again.');
						return false;
					} else {
						const invoices = JSON.parse(http_request.response);
						for (const invoice of invoices) {
							if (invoice['Status'] == "Shipped") {
								alert(`${invoice['MTS_Quote_Number']} has already been shipped. Make sure you want to resubmit this shipped invoice.`);
							}
							otherValidZohoInvoices.push(invoice['MTS_Quote_Number']);
						}
						handleMtsOrders(invoices[0]['MTS_Quote_Number'], input);
						setLoadingModalVisible(false);
					}
					break;
				default:
					alert('Error with AJAX request. Check network connectivity and try again. HTTP request status: ' + http_request.status);
			}
		}
	}
}

/**
 *
 * @param {HTMLInputElement} obj
 */
function setAllDates(obj) {
	for(const inpt of shippingDateInputs) {
		inpt.value = obj.value;
	}
}

/**
 *
 * @param {HTMLTextAreaElement} input
 */
function getNumEntries(input) {
	return input.value.trim().split('\n').length;
}

/**
 *
 * @param {HTMLTextAreaElement} input
 */
function updateRunningCounter(input) {
	const counter = document.getElementById(`count_devices_${getCurrentRow(input.id)}`);
	counter.innerHTML = input.value.trim().split('\n').filter(entry => entry.trim().length > 0).length;
}

function getCount(row) {
	return document.getElementById(`count_devices_${row}`).innerHTML;
}
