const getModuleAndView = () => {
	let page = window.location.href.substring(window.location.href.indexOf('m=')+2);
	let view = '';
	if (page.indexOf(`&v=`) != -1) {
		view = page.substring(page.indexOf(`&v=`) + 3);
		page = page.substring(0, page.indexOf(`&v=`));
	}
	if (page == 'inventory_new') {
		page = 'inventory';
	}
	return [page, view];
}
const [page, view] = getModuleAndView();
const CONTROLLER_BASE_URL = 'admin_controller.php';
const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';
const DEFAULT_SORT = 'machine_description';
const ICON_PATH_BASE = '/img/icons';
const ICON_PATHS = {
	amazon : `${ICON_PATH_BASE}/amazon.png`,
	backmarket : `${ICON_PATH_BASE}/backmarket.png`,
	ebay : `${ICON_PATH_BASE}/ebay.png`,
	newegg : `${ICON_PATH_BASE}/newegg.png`,
}

const PAGE_ELEMENTS = {
	tables : {
		in_stock : {
			/**
			 * @type {HTMLTableElement}
			 */
			ipad: document.getElementById('in_stock_ipad_table'),
			/**
			 * @type {HTMLTableElement}
			 */
			macbook: document.getElementById('in_stock_macbook_table'),
			/**
			 * @type {HTMLTableElement}
			 */
			imac: document.getElementById('in_stock_imac_table'),
			/**
			 * @type {HTMLTableElement}
			 */
			chromebook: document.getElementById('in_stock_chromebook_table'),
			/**
			 * @type {HTMLTableElement}
			 */
			other: document.getElementById('in_stock_other_table'),
		}
	},
	tableBodies : {
		in_stock : {
			/**
			 * @type {HTMLTableSectionElement}
			 */
			ipad : document.getElementById('instock_ipads'),
			/**
			 * @type {HTMLTableSectionElement}
			 */
			imac : document.getElementById('instock_imacs'),
			/**
			 * @type {HTMLTableSectionElement}
			 */
			macbook : document.getElementById('instock_macbooks'),
			/**
			 * @type {HTMLTableSectionElement}
			 */
			chromebook : document.getElementById('instock_chromebooks'),
			/**
			 * @type {HTMLTableSectionElement}
			 */
			other : document.getElementById('instock_others'),
		},
	},
	headRows : {
		in_stock : {
			/**
			 * @type {HTMLTableRowElement}
			 */
			ipad: document.getElementById('in_stock_ipad_head_row'),
			/**
			 * @type {HTMLTableRowElement}
			 */
			macbook: document.getElementById('in_stock_macbook_head_row'),
			/**
			 * @type {HTMLTableRowElement}
			 */
			imac: document.getElementById('in_stock_imac_head_row'),
			/**
			 * @type {HTMLTableRowElement}
			 */
			chromebook: document.getElementById('in_stock_chromebook_head_row'),
			/**
			 * @type {HTMLTableRowElement}
			 */
			other: document.getElementById('in_stock_other_head_row'),
		},
	},
	titles : {
		in_stock: {
			/**
			 * @type {HTMLHeadingElement}
			 */
			ipad: document.getElementById('instock_ipad_title'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			macbook: document.getElementById('instock_macbook_title'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			imac: document.getElementById('instock_imac_title'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			chromebook: document.getElementById('instock_chromebook_title'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			other: document.getElementById('instock_other_title'),
		},
	},
	headers : {
		in_stock: {
			/**
			 * @type {HTMLHeadingElement}
			 */
			ipad: document.getElementById('instock_ipad_head'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			macbook: document.getElementById('instock_macbook_head'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			imac: document.getElementById('instock_imac_head'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			chromebook: document.getElementById('instock_chromebook_head'),
			/**
			 * @type {HTMLHeadingElement}
			 */
			other: document.getElementById('instock_other_head'),
		},
	},
	/**
	 * @type {HTMLCollectionOf<HTMLImageElement>}
	 */
	sortIcons : document.getElementsByClassName('sort_icon')
}

const ASSET_DICT = {};

const ASSET_ARRAYS = {
	in_stock: {
		/**
		 * @type {Asset[]}
		 */
		// all: [],
		/**
		 * @type {Asset[]}
		 */
		ipad: [],
		/**
		 * @type {Asset[]}
		 */
		imac: [],
		/**
		 * @type {Asset[]}
		 */
		macbook: [],
		/**
		 * @type {Asset[]}
		 */
		chromebook: [],
		/**
		 * @type {Asset[]}
		 */
		other: []
	},
};

const COLUMN_DEFINITIONS = {
	asset: new ColumnDefinition('SKU', 'asset', {isSortable: true, isFilterable: true}),
	machine_description: new ColumnDefinition('Description', 'machine_description', {isSortable: true, isFilterable: true}),
	ram: new ColumnDefinition('RAM', 'ramVal', {isNumeric: true, isFilterable: true, isSortable: true}),
	storage: new ColumnDefinition('Storage', 'storageVal', {isNumeric: true, isFilterable: true, isSortable: true}),
	processor: new ColumnDefinition('Processor', 'processor', {isSortable: true, isFilterable: true}),
	processor_speed: new ColumnDefinition('Processor Speed', 'processor_speed', {isSortable: true, isFilterable: true}),
	item_count: new ColumnDefinition('Total Available', 'item_count', {isNumeric: true, isFilterable: true, isSortable: true}),
	first_stock_count: new ColumnDefinition('First Stock', 'first_stock_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	second_stock_count: new ColumnDefinition('Second Stock', 'second_stock_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	first_stock_pricing: new ColumnDefinition('Pricing (First Stock)', 'first_stock_pricing', {isNumeric: true, isSortable: true, isFilterable: true, prefix: '$'}),
	second_stock_pricing: new ColumnDefinition('Pricing (Second Stock)', 'second_stock_pricing', {isNumeric: true, isSortable: true, isFilterable: true, prefix: '$'}),
	first_stock_ten_pack_pricing: new ColumnDefinition('10-Pack Pricing (First Stock)', 'first_stock_ten_pack_pricing', {isNumeric: true, isSortable: false, isFilterable: true, prefix: '$'}),
	second_stock_ten_pack_pricing: new ColumnDefinition('10-Pack Pricing (Second Stock)', 'second_stock_ten_pack_pricing', {isNumeric: true, isSortable: false, isFilterable: true, prefix: '$'}),
	first_stock_ten_pack_discount: new ColumnDefinition('10-Pack Discount (First Stock)', 'first_stock_ten_pack_discount', {isNumeric: true, isSortable: false, isFilterable: true, prefix: '$'}),
	second_stock_ten_pack_discount: new ColumnDefinition('10-Pack Discount (Second Stock)', 'second_stock_ten_pack_discount', {isNumeric: true, isSortable: false, isFilterable: true, prefix: '$'}),
	latest_os: new ColumnDefinition('Latest OS', 'latest_os'),
	autoUpdateExpiration: new ColumnDefinition('AUE', 'autoUpdateExpiration'),
	upgradeable: new ColumnDefinition('Upgradeable', 'upgradeable'),
	notes: new ColumnDefinition('Notes', 'notes'),
	color: new ColumnDefinition('Color', 'color'),
	activation_lock_count: new ColumnDefinition('Activation Locked', 'activation_lock_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	mdm_lock_count: new ColumnDefinition('MDM Locked', 'mdm_lock_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	poNumber: new ColumnDefinition('PO Number', 'poNumber', {isSortable: true, isFilterable: true}),
	item_count_less_demand: new ColumnDefinition('Total Available', 'total_item_count_less_demand', {isNumeric: true, isFilterable: true, isSortable: true}),
	first_stock_count_less_demand: new ColumnDefinition('First Stock', 'first_stock_count_less_demand', {isSortable: true, isFilterable: true, isNumeric: true}),
	second_stock_count_less_demand: new ColumnDefinition('Second Stock', 'second_stock_count_less_demand', {isSortable: true, isFilterable: true, isNumeric: true}),
	average_cost: new ColumnDefinition('Average Cost', 'average_cost', {isSortable: true, isFilterable: true, isNumeric: true, prefix: '$'}),
	renewed_count: new ColumnDefinition('Renewed', 'renewed_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	certified_count: new ColumnDefinition('Certified Refurbished', 'certified_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	scratchdent_count: new ColumnDefinition('Scratch & Dent', 'scratchdent_count', {isSortable: true, isFilterable: true, isNumeric: true}),
	renewed_price: new ColumnDefinition('Pricing (Renewed)', 'renewed_price', {isSortable: true, isFilterable: true, isNumeric: true, prefix: '$'}),
	certified_price: new ColumnDefinition('Pricing (Certified)', 'certified_price', {isSortable: true, isFilterable: true, isNumeric: true, prefix: '$'}),
	scratchdent_price: new ColumnDefinition('Pricing (S & D)', 'scratchdent_price', {isSortable: true, isFilterable: true, isNumeric: true, prefix: '$'}),
	device_condition: new ColumnDefinition('Condition', 'device_condition', {isSortable: true, isFilterable: true}),
	touchscreen: new ColumnDefinition('Touchscreen', 'touchscreen', {isSortable: true, isFilterable: true}),
	dropship_quantity: new ColumnDefinition('Dropship Quantity', 'dropship_quantity', {isSortable : true, isFilterable : true, isNumeric : true}),
	dropship_cost: new ColumnDefinition('Dropship Cost', 'dropship_cost', { isSortable: true, isFilterable : true, isNumeric : true })
}

const COLUMNS = {
	in_stock : {
		generic: [
			COLUMN_DEFINITIONS.asset,
			COLUMN_DEFINITIONS.machine_description,
			COLUMN_DEFINITIONS.ram,
			COLUMN_DEFINITIONS.storage,
			COLUMN_DEFINITIONS.processor,
			COLUMN_DEFINITIONS.processor_speed,
			COLUMN_DEFINITIONS.dropship_quantity,
			COLUMN_DEFINITIONS.dropship_cost,
			COLUMN_DEFINITIONS.first_stock_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_discount,
			COLUMN_DEFINITIONS.latest_os,
			COLUMN_DEFINITIONS.notes
		],
		ipad: [
			COLUMN_DEFINITIONS.asset,
			COLUMN_DEFINITIONS.machine_description,
			COLUMN_DEFINITIONS.storage,
			COLUMN_DEFINITIONS.dropship_quantity,
			COLUMN_DEFINITIONS.dropship_cost,
			COLUMN_DEFINITIONS.first_stock_pricing,
			// COLUMN_DEFINITIONS.second_stock_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_pricing,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_discount,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_discount,
			COLUMN_DEFINITIONS.processor,
			COLUMN_DEFINITIONS.latest_os,
			COLUMN_DEFINITIONS.color
		],
		imac: [
			COLUMN_DEFINITIONS.asset,
			COLUMN_DEFINITIONS.machine_description,
			COLUMN_DEFINITIONS.ram,
			COLUMN_DEFINITIONS.storage,
			COLUMN_DEFINITIONS.processor,
			COLUMN_DEFINITIONS.processor_speed,
			COLUMN_DEFINITIONS.dropship_quantity,
			COLUMN_DEFINITIONS.dropship_cost,
			COLUMN_DEFINITIONS.first_stock_pricing,
			// COLUMN_DEFINITIONS.second_stock_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_pricing,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_discount,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_discount,
			COLUMN_DEFINITIONS.latest_os,
		],
		macbook: [
			COLUMN_DEFINITIONS.asset,
			COLUMN_DEFINITIONS.machine_description,
			COLUMN_DEFINITIONS.ram,
			COLUMN_DEFINITIONS.storage,
			COLUMN_DEFINITIONS.processor,
			COLUMN_DEFINITIONS.processor_speed,
			COLUMN_DEFINITIONS.dropship_quantity,
			COLUMN_DEFINITIONS.dropship_cost,
			COLUMN_DEFINITIONS.first_stock_pricing,
			// COLUMN_DEFINITIONS.second_stock_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_pricing,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_discount,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_discount,
			COLUMN_DEFINITIONS.latest_os,
		],
		chromebook: [
			COLUMN_DEFINITIONS.asset,
			COLUMN_DEFINITIONS.machine_description,
			COLUMN_DEFINITIONS.ram,
			COLUMN_DEFINITIONS.storage,
			COLUMN_DEFINITIONS.processor,
			COLUMN_DEFINITIONS.device_condition,
			COLUMN_DEFINITIONS.touchscreen,
			COLUMN_DEFINITIONS.dropship_quantity,
			COLUMN_DEFINITIONS.dropship_cost,
			COLUMN_DEFINITIONS.first_stock_pricing,
			// COLUMN_DEFINITIONS.second_stock_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_pricing,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_pricing,
			COLUMN_DEFINITIONS.first_stock_ten_pack_discount,
			// COLUMN_DEFINITIONS.second_stock_ten_pack_discount,
			COLUMN_DEFINITIONS.autoUpdateExpiration,
		],
	},
}

const filter = {};
const sort = {
	order: 1,
	by: DEFAULT_SORT
}

let isFirstLoad = true;
let showAll = true;

init();

async function init() {
	const inventoryItems = await getInventoryItems();
	if (inventoryItems) {
		console.log('success!');
		constructAssetObjects(inventoryItems);
		populateAssetArrays();
		sortItems();
		// renderShowAllToggle();
		renderHeaders();
		renderBodies();
	}
	else {
		console.log('fetching inventory items failed.');
		console.log(inventoryItems);
	}
	setLoadingModalVisible(false);
}

async function getInventoryItems() {
	const url = `${CONTROLLER_BASE_URL}?m=dropship_getInventoryItems`;
	const response = await fetch(url).then(response => { return response });
	const responseText = await response.text();
	try {
		return JSON.parse(responseText);
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
		return null;
	}
}

function constructAssetObjects(inventoryItems) {
	for (const data of inventoryItems) {
		const asset = new Asset(data);
		const orderNumberId = asset.getMachineOrderNumberId();
		if (ASSET_DICT[orderNumberId]) {
			delete ASSET_DICT[orderNumberId];
		}
		ASSET_DICT[orderNumberId] = asset;
	}
}

function populateAssetArrays() {
	for (const machineOrderNumberId in ASSET_DICT) {
		/**
		 * @type {Asset}
		 */
		const asset = ASSET_DICT[machineOrderNumberId];
		const inStock = asset.dropshipCount > 0;
		const machineType = asset.getMachineType();
		if (inStock) {
			// ASSET_ARRAYS.in_stock.all.push(asset);
			/**
			 * @type {Asset[]}
			 */
			const typeArray = ASSET_ARRAYS.in_stock[machineType];
			typeArray.push(asset);
		}
	}
}

function renderHeaders() {
	for (const stockType in PAGE_ELEMENTS.headRows) {
		for (const assetType in PAGE_ELEMENTS.headRows[stockType]) {
			/**
			 * @type {ColumnDefinition[]}
			 */
			const headRowColumns = (COLUMNS[stockType][assetType]) ? COLUMNS[stockType][assetType] : COLUMNS[stockType]['generic'];
			/**
			 * @type {HTMLTableRowElement}
			 */
			const currentRow = PAGE_ELEMENTS.headRows[stockType][assetType];
			currentRow.innerHTML = '';

			for (const column of headRowColumns) {
				currentRow.appendChild(column.getHeaderCellElement());
			}
		}
	}
}

function renderBodies() {
	for (const stockType in PAGE_ELEMENTS.tableBodies) {
		for (const assetType in PAGE_ELEMENTS.tableBodies[stockType]) {
			/**
			 * @type {Asset[]}
			 */
			const currentAssetArray = ASSET_ARRAYS[stockType][assetType];
			if (currentAssetArray.length > 0) {
				/**
				 * @type {HTMLTableElement}
				 */
				const currentTable = PAGE_ELEMENTS.tables[stockType][assetType];
				/**
				 * @type {HTMLHeadingElement}
				 */
				const currentTitle = PAGE_ELEMENTS.titles[stockType][assetType];
				/**
				 * @type {HTMLTableSectionElement}
				 */
				const currentTableBody = PAGE_ELEMENTS.tableBodies[stockType][assetType];
				/**
				 * @type {ColumnDefinition[]}
				 */
				const columns = (COLUMNS[stockType][assetType]) ? COLUMNS[stockType][assetType] : COLUMNS[stockType]['generic'];

				currentTableBody.innerHTML = '';

				for (const asset of currentAssetArray) {
					currentTableBody.appendChild(asset.generateRow(columns));
				}
				if (stockType == 'in_stock') {
					currentTitle.hidden = false;
					currentTable.hidden = false;
				}
				else {
					currentTitle.hidden = true;
					currentTable.hidden = true;
				}
			}
		}
	}
}

function toggleShowAll() {
	showAll = !showAll;
	renderShowAllToggle();
	for (const assetOrderNumber in ASSET_DICT) {
		/**
		 * @type {Asset}
		 */
		const currentAsset = ASSET_DICT[assetOrderNumber];
		if (!currentAsset.getTtsActive() && !showAll) {
			currentAsset.setIsValid(false);
		}
		else {
			currentAsset.setIsValid(true);
		}
		currentAsset.conformVisibility();
	}
}

function renderShowAllToggle() {
	/**
	 * @type {HTMLSpanElement}
	 */
	const containerSpan = document.getElementById('showAllToggleSpan');
	while (containerSpan.firstChild) {
		containerSpan.removeChild(containerSpan.firstChild);
	}
	if (showAll) {
		const showAllSpan = document.createElement('span');
		containerSpan.appendChild(showAllSpan);
		showAllSpan.innerText = "(show all)";
	}
	else {
		const showAllLink = document.createElement('a');
		containerSpan.appendChild(showAllLink);
		showAllLink.href = "javascript:void(0)";
		showAllLink.setAttribute('onclick', 'toggleShowAll()');
		showAllLink.innerText = "(show all)";
	}
	const dividerSpan = document.createElement('span');
	containerSpan.appendChild(dividerSpan);
	dividerSpan.innerText = " | ";
	if (!showAll) {
		const showActiveSpan = document.createElement('span');
		containerSpan.appendChild(showActiveSpan);
		showActiveSpan.innerText = "(show TTS active)";
	}
	else {
		const showActiveLink = document.createElement('a');
		containerSpan.appendChild(showActiveLink);
		showActiveLink.href = "javascript:void(0)";
		showActiveLink.setAttribute('onclick', 'toggleShowAll()');
		showActiveLink.innerText = "(show TTS active)";
	}
}

/* filter/sort functions below */

function sortItems() {
	for (const stockType in ASSET_ARRAYS) {
		for (const itemType in ASSET_ARRAYS[stockType]) {
			/**
			 * @type {Asset[]}
			 */
			const currentArray = ASSET_ARRAYS[stockType][itemType];
			currentArray.sort((a,b) => sortFunction(a,b));
		}
	}
	for (const icon of PAGE_ELEMENTS.sortIcons) {
		if (icon.parentElement.getAttribute('sort_order') != sort.by) {
			icon.src = SORT_NONE_URL;
		} else {
			if (sort.order == 1) {
				icon.src = SORT_ASC_URL;
			} else if (sort.order == -1) {
				icon.src = SORT_DESC_URL;
			}
		}
	}

	/**
	 *
	 * @param {Asset} a
	 * @param {Asset} b
	 */
	function sortFunction(a,b) {
		const sortValueA = a.getFieldByName(sort.by);
		const sortValueB = b.getFieldByName(sort.by);
		if(isNaN(sortValueA)) {
			if(sortValueA < sortValueB) {
				return -1 * sort.order;
			}
			if(sortValueA > sortValueB) {
				return 1 * sort.order;
			}
		} else {
			if(parseInt(sortValueA) < parseInt(sortValueB)) {
				return -1 * sort.order;
			}
			if(parseInt(sortValueA) > parseInt(sortValueB)) {
				return 1 * sort.order;
			}
		}
	}
}



function setSort(newSort, invertOrder=true) {
	if (newSort == sort.by && invertOrder) {
		sort.order *= -1;
	} else {
		sort.by = newSort;
	}
	sortItems();
	renderBodies();
}

/**
 *
 * @param {HTMLInputElement} input
 */
function filterInput(input) {
	filter[input.getAttribute('filter')] = input.value;
	const sameNameInputs = document.getElementsByClassName(input.getAttribute('filter') + '_filter');
	for(const elem of sameNameInputs) {
		elem.value = input.value;
	}
	filterResults();
}

function filterResults() {
	for (const machineOrderId in ASSET_DICT) {
		/**
		 * @type {Asset}
		 */
		const currentAsset = ASSET_DICT[machineOrderId];
		let valid = true;
		for (const entry in filter) {
			const assetEntryValue = currentAsset.getFieldByName(entry);
			if (filter[entry] == '') {
				delete filter[entry];
			}
			else if (typeof assetEntryValue == 'string') {
				if (!assetEntryValue.toLowerCase().includes(filter[entry].toLowerCase())) {
					valid = false;
				}
			}
			else if (typeof assetEntryValue == 'number') {
				if (valid) {
					valid = filterNumbers(filter[entry], assetEntryValue);
				}
			}
			else if (typeof assetEntryValue == 'boolean') {
				if (valid) {
					valid = filterBoolean(filter[entry], assetEntryValue);
				}
			}
		}
		currentAsset.setIsValid(valid);
		currentAsset.conformVisibility();
	}
	/**
	 *
	 * @param {string} filterValue Value of the filter
	 * @param {number} inputValue Value of the input
	 */
	function filterNumbers(filterValue, inputValue) {
		let comparator = '';
		if (filterValue.startsWith('>=') || filterValue.startsWith('<=') || filterValue.startsWith('!=')) {
			comparator = filterValue.substring(0, 2);
			filterValue = (isNaN(parseInt(filterValue.substring(2)))) ? 0 : parseInt(filterValue.substring(2));
		} else if (filterValue.startsWith('>') || filterValue.startsWith('<') || filterValue.startsWith('=')) {
			comparator = filterValue.substring(0, 1);
			filterValue = (isNaN(parseInt(filterValue.substring(1)))) ? 0 : parseInt(filterValue.substring(1));
		} else {
			comparator = '=';
		}
		if (comparator == '>=') {
			return (inputValue >= filterValue);
		} else if (comparator == '<=') {
			return (inputValue <= filterValue);
		} else if (comparator == '!=') {
			return (inputValue != filterValue);
		} else if (comparator == '>') {
			return (inputValue > filterValue);
		} else if (comparator == '<') {
			return (inputValue < filterValue);
		} else if (comparator == '=') {
			return (inputValue == filterValue);
		}
		return true;
	}

	/**
	 *
	 * @param {string} filterValue
	 * @param {boolean} inputValue
	 */
	function filterBoolean(filterValue, inputValue) {
		const validTrue = ['y', 'ye', 'yes', 't', 'tr', 'tru', 'true', '1'];
		const validFalse = ['n', 'no', 'f', 'fa', 'fal', 'fals', 'false', '0'];
		if (inputValue === true) {
			return validTrue.includes(filterValue.trim().toLowerCase());
		}
		else if (inputValue === false) {
			return validFalse.includes(filterValue.trim().toLowerCase());
		}
	}
}

/* CSV generation below */

function generateCsv() {
	const rows = [];
	// const rows = [generateCsvHeaderRow()];
	if (hasValid(ASSET_ARRAYS.in_stock.ipad)) {
		rows.push(['"iPads"']);
		rows.push(generateCsvHeaderRow('ipad'));
		for (const asset of ASSET_ARRAYS.in_stock.ipad) {
			if (asset.getIsValid()) {
				const newRow = [
					`"${asset.getMachineOrderNumber()}"`,
					`"${asset.getMachineDescription()}"`,
					`"${asset.getStorage()}"`,
					`"${asset.getFirstStockIndividualPrice()}"`,
					`"${asset.getSecondStockIndividualPrice()}"`,
					`"${asset.getFirstStockTenpackPrice()}"`,
					`"${asset.getSecondStockTenpackPrice()}"`,
					`"${asset.getProcessor()}"`,
					`"${asset.getLatestOs()}"`
				];
				rows.push(newRow);
			}
		}
		rows.push(generateEmptyRow());
	}
	if (hasValid(ASSET_ARRAYS.in_stock.macbook)) {
		rows.push(['"MacBooks"']);
		rows.push(generateCsvHeaderRow('macbook'));
		for (const asset of ASSET_ARRAYS.in_stock.macbook) {
			if (asset.getIsValid()) {
				rows.push(generateItemRow(asset));
			}
		}
		rows.push(generateEmptyRow());
	}
	if (hasValid(ASSET_ARRAYS.in_stock.imac)) {
		rows.push(['"iMacs"']);
		rows.push(generateCsvHeaderRow('imac'));
		for (const asset of ASSET_ARRAYS.in_stock.imac) {
			if (asset.getIsValid()) {
				rows.push(generateItemRow(asset));
			}
		}
		rows.push(generateEmptyRow());
	}
	if (hasValid(ASSET_ARRAYS.in_stock.chromebook)) {
		rows.push(['"Chromebooks"']);
		rows.push(generateCsvHeaderRow('chromebook'));
		for (const asset of ASSET_ARRAYS.in_stock.chromebook) {
			if (asset.getIsValid()) {
				rows.push(generateItemRow(asset));
			}
		}
		rows.push(generateEmptyRow());
	}
	if (hasValid(ASSET_ARRAYS.in_stock.other)) {
		rows.push(['"Other"']);
		rows.push(generateCsvHeaderRow('other'));
		for (const asset of ASSET_ARRAYS.in_stock.other) {
			if (asset.getIsValid()) {
				rows.push(generateItemRow(asset));
			}
		}
		rows.push(generateEmptyRow());
	}

	let csvContent = "data:text/csv;charset=utf-8," + rows.map(e => e.join(",")).join("\n");
	const encodedUri = encodeURI(csvContent);
	const link = document.createElement("a");
	link.setAttribute("href", encodedUri);
	link.setAttribute("download", `inventory_export_${getTimestamp(new Date())}.csv`);
	document.body.appendChild(link); // Required for FF

	link.click();
	delete link;

	/**
	 *
	 * @param {Asset} asset
	 * @param {string} type
	 */
	function generateItemRow(asset, type='') {
		const itemRow = [
			sanitize(asset.machineOrderNumber),
			(asset.dropshipDescription) ? asset.dropshipDescription : asset.machineDescription,
			asset.ram,
			asset.storage,
			asset.processor,
			asset.processorSpeed,
			asset.dropshipCount.toLocaleString(),
			asset.pricing.firstStock.individual.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0}),
			asset.pricing.firstStock.tenpack.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0}),
			(type == "chromebook") ? asset.autoUpdateExpiration : asset.latestOs
		];
		const newRow = [
			`"${asset.getMachineOrderNumber()}"`,
			`"${asset.getMachineDescription()}"`,
			`"${asset.getRam()}"`,
			`"${asset.getStorage()}"`,
			`"${asset.getProcessor()}"`,
			`"${asset.getProcessorSpeed()}"`,
			`"${asset.dropshipCount.toLocaleString()}"`,
			`"${asset.getFirstStockIndividualPrice()}"`,
			`"${asset.getFirstStockTenpackPrice()}"`,
			`"${(type == 'chromebook') ? asset.getAutoUpdateExpiration() : asset.getLatestOs()}"`
		];
		return itemRow;
		/**
		 * Sanitized (escapes) string characters
		 * @param {string} str
		 */
		function sanitize(str) {
			return str.replace("#", ' ');
		}
	}

	function generateCsvHeaderRow(type) {
		if (type == 'ipad') {
			return generateCsvIpadHeaderRow();
		}
		else if (type == 'chromebook') {
			return generateChromebookCsvHeaderRow();
		}
		else {
			return generateOthersCsvHeaderRow();
		}

		function generateCsvIpadHeaderRow() {
			const headerRow = [
				'SKU', 'Description', 'Storage', 'Pricing (First)', 'Pricing (Second)', '10-Pack Pricing (First)', '10-Pack Pricing (Second)', 'Processor', 'Latest OS'
			];
			return headerRow;
		}

		function generateChromebookCsvHeaderRow() {
			const headerRow = [
				'SKU', 'Description', 'RAM', 'Storage', 'Processor', 'Processor Speed', 'Pricing (First)', 'Pricing (Second)', '10-Pack Pricing (First)', '10-Pack Pricing (Second)', 'AUE'
			];
			return headerRow;
		}

		function generateOthersCsvHeaderRow() {
			const headerRow = [
				'SKU', 'Description', 'RAM', 'Storage', 'Processor', 'Processor Speed', 'Pricing (First)', 'Pricing (Second)', '10-Pack Pricing (First)', '10-Pack Pricing (Second)', 'Latest OS'
			];
			return headerRow;
		}
	}

	function generateEmptyRow() {
		const emptyRow = [''];
		return emptyRow;
	}

	/**
	 *
	 * @param {Date} time
	 */
	function getTimestamp(time) {
		let timestamp = time.getFullYear().toString();
		if (time.getMonth()+1 < 10) {
			timestamp += '0';
		}
		timestamp += (time.getMonth()+1).toString();
		if (time.getDate() < 10) {
			timestamp += '0';
		}
		timestamp += time.getDate().toString();
		if (time.getHours() < 10) {
			timestamp += '0';
		}
		timestamp += time.getHours().toString();
		if (time.getMinutes() < 10) {
			timestamp += '0';
		}
		timestamp += time.getMinutes().toString();
		if (time.getSeconds() < 10) {
			timestamp += '0';
		}
		timestamp += time.getSeconds().toString();

		return timestamp
	}

	/**
	 *
	 * @param {Asset[]} arr
	 */
	function hasValid(arr) {
		for (const asset of arr) {
			if (asset.getIsValid()) {
				return true;
			}
		}
		return false;
	}
}
