class Asset {
	constructor(data) {
		// Identification data
		/**
		 * @type {string}
		 */
		this.machineOrderNumberId = data['machineOrderNumberId'];
		/**
		 * @type {string}
		 */
		this.machineId = data['machineId'];
		/**
		 * @type {string}
		 */
		this.machineOrderNumber = data['machineOrderNumber'];
		/**
		 * @type {string}
		 */
		this.machineDescription = data['machineDescription'];
		/**
		 * @type {string}
		 */
		this.machineType = data['machineType'];
		/**
		 * @type {string}
		 */
		this.latestOs = data['latestOs'];
		// this.upgradeable = data['upgradeable'];
		/**
		 * @type {string}
		 */
		this.autoUpdateExpiration = data['autoUpdateExpiration'];
		/**
		 * @type {boolean}
		 */
		this.limitedToInventory = data['limitedToInventory'];
		/**
		 * @type {boolean}
		 */
		this.ttsActive = data['ttsActive'];
		/**
		 * @type {boolean}
		 */
		this.specialOrder = data['specialOrder'];

		// Specification data
		/**
		 * @type {string}
		 */
		this.ram = data['ram'];
		/**
		 * @type {string}
		 */
		this.storage = data['storage'];
		/**
		 * @type {string}
		 */
		this.processor = data['processor'];
		/**
		 * @type {string}
		 */
		this.processorSpeed = data['processorSpeed'];
		/**
		 * @type {boolean}
		 */
		this.touchscreen = data['touchscreen'];
		/**
		 * @type {string}
		 */
		this.color = data['color'];
		/**
		 * @type {string}
		 */
		this.deviceCondition = data['deviceCondition'];

		// Item count data
		const dataItemCounts = data['itemCounts'];
		const dataDemand = data['demand'];
		/**
		 * @type {boolean}
		 */
		this.dropship = data['dropship'];
		/**
		 * @type {number}
		 */
		this.dropshipCount = parseInt(data['dropshipQuantity']);
		/**
		 * @type {number}
		 */
		this.dropshipCost = parseInt(data['dropshipCost']);
		/**
		 * @type {string}
		 */
		this.dropshipDescription = data['dropshipDescription'];

		// Pricing data
		const dataPricing = data['pricing'];
		this.pricing = {
			firstStock : {
				/**
				 * @type {number}
				 */
				individual : dataPricing.firstStock.individual,
				/**
				 * @type {number}
				 */
				tenpack : dataPricing.firstStock.tenpack
			},
			secondStock : {
				/**
				 * @type {number}
				 */
				individual : dataPricing.secondStock.individual,
				/**
				 * @type {number}
				 */
				tenpack : dataPricing.secondStock.tenpack
			}
		};

		// API integration data
		const dataZohoIds = data['zohoIds'];
		this.zohoIds = {
			firstStock : {
				/**
				 * @type {string}
				 */
				individual : dataZohoIds.firstStock.individual,
				/**
				 * @type {string}
				 */
				tenpack : dataZohoIds.firstStock.tenpack
			},
			secondStock : {
				/**
				 * @type {string}
				 */
				individual : dataZohoIds.secondStock.individual,
				/**
				 * @type {string}
				 */
				tenpack : dataZohoIds.secondStock.tenpack
			}
		};
		const dataShopifyIds = data['shopifyIds'];
		this.shopifyIds = {
			firstStock : dataShopifyIds.firstStock,
			secondStock : dataShopifyIds.secondStock
		};

		// Handlers for rendering
		this.isValid = true;
		/**
		 * @type {HTMLTableRowElement}
		 */
		this.assetRow = null;
	}
	/* Generators start */

	conformVisibility() {
		if (this.assetRow) {
			this.assetRow.hidden = !this.getIsValid();
		}
	}

	/**
	 *
	 * @param {ColumnDefinition} columns
	 */
	generateRow(columns) {
		if (this.assetRow) {
			delete this.assetRow;
		}
		const row = document.createElement('tr');
		for (const column of columns) {
			const identifier = column.getIdentifier();
			try {
				row.appendChild(this.generateTableCellByIdentifier(identifier))
			}
			catch (err) {
				console.error(identifier, err);
				row.appendChild(this.generatePlaceholderCell());
			}
		}
		const rowClasses = this.getRowClasses();
		for (const rowClass of rowClasses) {
			row.classList.add(rowClass);
		}
		this.assetRow = row;
		this.hidden = !(this.isValid)
		return row;
	}

	/**
	 *
	 * @param {string} identifier Field name to fetch
	 */
	generateTableCellByIdentifier(identifier) {
		if (identifier == 'asset' || identifier == 'machineOrderNumber') {
			return this.generateAssetCell();
		}
		else if (identifier == 'machine_description') {
			return (this.dropshipDescription) ? this.generateCellWithText(this.dropshipDescription) : this.generateMachineDescriptionCell();
		}
		else if (identifier == 'ram' || identifier == 'ramVal') {
			return this.generateRamCell();
		}
		else if (identifier == 'storage' || identifier == 'storageVal') {
			return this.generateStorageCell();
		}
		else if (identifier == 'total_item_count_less_demand') {
			return this.generateTotalCountCell();
		}
		else if (identifier == 'dropship_quantity') {
			return this.generateTotalCountCell();
		}
		else if (identifier == 'first_stock_count_less_demand') {
			return this.generateFirstStockCountCell();
		}
		else if (identifier == 'second_stock_count_less_demand') {
			return this.generateSecondStockCountCell();
		}
		else if (identifier == 'first_stock_pricing') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getFirstStockIndividualPrice()));
		}
		else if (identifier == 'second_stock_pricing') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getSecondStockIndividualPrice()));
		}
		else if (identifier == 'first_stock_ten_pack_pricing') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getFirstStockTenpackPrice()));
		}
		else if (identifier == 'second_stock_ten_pack_pricing') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getSecondStockTenpackPrice()));
		}
		else if (identifier == 'first_stock_ten_pack_discount') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getFirstStockTenpackDiscount()));
		}
		else if (identifier == 'second_stock_ten_pack_discount') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.getSecondStockTenpackDiscount()));
		}
		else if (identifier == 'dropship_cost') {
			return this.generateCellWithText(this.displayNumberAsCurrency(this.dropshipCost));
		}
		else {
			return this.generateCellWithText(this.getFieldByName(identifier));
		}
	}

	generateAssetCell() {
		const href = `${CONTROLLER_BASE_URL}?m=order_number_edit&order_number_id=${this.getMachineOrderNumberId()}`;
		const cell = this.generateCellWithLink(this.getMachineOrderNumber(), href);

		return cell;
	}

	generateMachineDescriptionCell() {
		const href = `${CONTROLLER_BASE_URL}?m=machine_detail&machine_id=${this.getMachineId()}`;
		const cell = this.generateCellWithLink(this.getMachineDescription(), href);

		return cell;
	}

	generateStorageCell() {
		return this.generateCellWithText(this.getStorage());
	}

	generateRamCell() {
		return this.generateCellWithText(this.getRam());
	}

	generateTotalCountCell() {
		const hasStock = this.dropshipCount > 0;

		const cell = document.createElement('td');
		const span = document.createElement('span');
		span.classList.add('large');
		span.innerText = this.dropshipCount.toLocaleString();
		cell.appendChild(span);
		return cell;
	}

	generateFirstStockCountCell() {
		const hasStock = (this.getFirstStockCount() > 0);

		const cell = document.createElement('td');
		const span = document.createElement('span');
		span.classList.add('large');
		span.innerText = this.getFirstStockLessDemand().toLocaleString();
		if (hasStock) {
			const button = document.createElement('button');
			cell.appendChild(button);
			button.classList.add('link');
			button.classList.add('large');
			button.setAttribute('onclick', `getAssetData('${this.getMachineOrderNumber()}', 'first')`)
			button.appendChild(span);
		}
		else {
			cell.appendChild(span);
		}
		return cell;
	}

	generateSecondStockCountCell() {
		const hasStock = (this.getSecondStockCount() > 0);

		const cell = document.createElement('td');
		const span = document.createElement('span');
		span.classList.add('large');
		span.innerText = this.getSecondStockLessDemand().toLocaleString();
		if (hasStock) {
			const button = document.createElement('button');
			cell.appendChild(button);
			button.classList.add('link');
			button.classList.add('large');
			button.setAttribute('onclick', `getAssetData('${this.getMachineOrderNumber()}', 'second')`)
			button.appendChild(span);
		}
		else {
			cell.appendChild(span);
		}
		return cell;
	}

	/* Generators end */


	/* Getters start */

	/**
	 *
	 * @param {string} fieldName Field name to fetch
	 */
	getFieldByName(fieldName) {
		if (fieldName == 'asset' || fieldName == 'machineOrderNumber') {
			return this.getMachineOrderNumber();
		}
		else if (fieldName == 'machine_description') {
			return this.getMachineDescription();
		}
		else if (fieldName == 'ram') {
			return this.getRam();
		}
		else if (fieldName == 'ramVal') {
			return this.getRamAsNumber();
		}
		else if (fieldName == 'storage') {
			return this.getStorage();
		}
		else if (fieldName == 'storageVal') {
			return this.getStorageAsNumber();
		}
		else if (fieldName == 'processor') {
			return this.getProcessor();
		}
		else if (fieldName == 'processor_speed') {
			return this.getProcessorSpeed();
		}
		else if (fieldName == 'total_item_count_less_demand') {
			return this.getTotalCalculatedStock();
		}
		else if (fieldName == 'first_stock_count_less_demand') {
			return this.getFirstStockLessDemand();
		}
		else if (fieldName == 'second_stock_count_less_demand') {
			return this.getSecondStockLessDemand();
		}
		else if (fieldName == 'first_stock_pricing') {
			return this.getFirstStockIndividualPrice();
		}
		else if (fieldName == 'second_stock_pricing') {
			return this.getSecondStockIndividualPrice();
		}
		else if (fieldName == 'first_stock_ten_pack_pricing') {
			return this.getFirstStockTenpackPrice();
		}
		else if (fieldName == 'second_stock_ten_pack_pricing') {
			return this.getSecondStockTenpackPrice();
		}
		else if (fieldName == 'first_stock_ten_pack_discount') {
			return this.getFirstStockTenpackDiscount();
		}
		else if (fieldName == 'second_stock_ten_pack_discount') {
			return this.getSecondStockTenpackDiscount();
		}
		else if (fieldName == 'latest_os') {
			return this.getLatestOs();
		}
		else if (fieldName == 'color') {
			return this.getColor();
		}
		else if (fieldName == 'device_condition') {
			return this.getDeviceCondition();
		}
		else if (fieldName == 'touchscreen') {
			return this.getTouchscreen();
		}
		else if (fieldName == 'autoUpdateExpiration') {
			return this.getAutoUpdateExpiration();
		}
		else if (fieldName == "dropship_cost") {
			return this.dropshipCost;
		}
	}
	getMachineOrderNumberId() {
		return this.machineOrderNumberId;
	}

	getMachineOrderNumber() {
		return this.machineOrderNumber;
	}

	getMachineId() {
		return this.machineId;
	}

	getMachineDescription() {
		return this.machineDescription;
	}

	getMachineType() {
		return this.machineType;
	}

	getAutoUpdateExpiration() {
		return this.autoUpdateExpiration;
	}

	getLatestOs() {
		return this.latestOs;
	}

	getColor() {
		return this.color;
	}

	getDeviceCondition() {
		return this.deviceCondition;
	}

	getRam() {
		return this.ram;
	}

	getRamAsNumber() {
		const ramString = this.getRam();
		if (isNaN(parseInt(ramString))) {
			return 0;
		}
		else {
			return parseInt(ramString);
		}
	}

	getStorage() {
		return this.storage;
	}

	getStorageAsNumber() {
		const storageString = this.getStorage();
		if (isNaN(parseInt(storageString))) {
			return 0;
		}
		else {
			const multiplier = (storageString.endsWith('TB')) ? 1024 : 1;
			return parseInt(storageString) * multiplier;
		}
	}

	getProcessor() {
		return this.processor;
	}

	getProcessorSpeed() {
		return this.processorSpeed;
	}

	getTouchscreen() {
		return this.touchscreen;
	}

	getCurrentStockCount() {
		return this.getFirstStockCount() + this.getSecondStockCount();
	}

	getFirstStockLessDemand() {
		return this.getFirstStockCount() - this.getFirstStockDemand();
	}

	getSecondStockLessDemand() {
		return this.getSecondStockCount() - this.getSecondStockDemand();
	}

	getTotalCalculatedStock() {
		return this.getFirstStockLessDemand() + this.getSecondStockLessDemand() + this.getInbound();
	}

	getHasStockOrDemand() {
		return (this.getFirstStockCount() || this.getSecondStockCount() || this.getFirstStockDemand() || this.getSecondStockDemand() || this.getInbound());
	}

	getFirstStockIndividualPrice() {
		return this.pricing.firstStock.individual;
	}

	getSecondStockIndividualPrice() {
		return this.pricing.secondStock.individual;
	}

	getFirstStockTenpackPrice() {
		return this.pricing.firstStock.tenpack;
	}

	getSecondStockTenpackPrice() {
		return this.pricing.secondStock.tenpack;
	}

	getFirstStockTenpackDiscount() {
		return Math.max(0, this.getFirstStockIndividualPrice() - this.getFirstStockTenpackPrice() / 10);
	}

	getSecondStockTenpackDiscount() {
		return Math.max(0, this.getSecondStockIndividualPrice() - this.getSecondStockTenpackPrice() / 10);
	}

	getIsLimitedToInventory() {
		return this.limitedToInventory;
	}

	getIsSpecialOrder() {
		return this.specialOrder;
	}

	getIsValid() {
		return this.isValid;
	}

	getTtsActive() {
		return this.ttsActive;
	}

	/**
	 *
	 * @param {boolean} isValid
	 */
	setIsValid(isValid) {
		this.isValid = (isValid);
	}

	getRowClasses() {
		const negativeStock = 'negative-stock';
		const lowStock = 'low-stock';
		const highStock = 'high-stock';
		const limitedToInventory = 'limited-to-inventory';
		const specialOrder = 'special-order';
		const classes = [];

		const totalCalculatedStock = this.dropshipCount;
		if (totalCalculatedStock < 0) {
			classes.push(negativeStock);
		}
		else if (totalCalculatedStock > 0 && totalCalculatedStock <= 10) {
			classes.push(lowStock);
		}
		else if (totalCalculatedStock > 10) {
			classes.push(highStock);
		}

		if (this.getIsLimitedToInventory()) {
			// classes.push(limitedToInventory);
		}
		if (this.getIsSpecialOrder()) {
			// classes.push(specialOrder);
		}
		return classes;
	}
	/* Getters end */

	/* generic generators */

	generatePlaceholderCell() {
		return this.generateCellWithText('placeholder');
	}

	/**
	 *
	 * @param {string|boolean} innerText
	 */
	generateCellWithText(innerText) {
		const cell = document.createElement('td');
		if (typeof innerText == 'boolean') {
			cell.innerText = (innerText) ? 'Yes' : 'No';
		}
		else {
			cell.innerText = (innerText) ? innerText : 'N/A';
		}
		return cell;
	}

	generateCellWithLink(innerText='', href='') {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		if (href) {
			link.href = href;
		}
		link.target = "_blank";
		if (innerText) {
			link.innerText = innerText;
		}
		return cell;
	}

	/**
	 *
	 * @param {number} num
	 */
	displayNumberAsCurrency(num) {
		try {
			return num.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		catch (err) {
			console.error(err);
			return 'N/A';
		}

	}
}
