/**
 * @type {HTMLDivElement}
 */
const assetModal = document.getElementById('asset_modal');
let changed = false;
let assetGrade = '';
const statuses = [];
const modalData = {
	requestedAssetData: [],
	filter: {},
	sortBy: "oNum",
	sortOrder: 1,
	currentPage: 1,
	itemsPerPage: 50,
	maxPages: 0
}
const modalTitles = {
	'all' : 'All',
	'first' : 'First Stock',
	'second' : 'Second Stock',
	'engraved' : 'Engraved',
	'activation' : 'Activation Locked',
	'mdm' : 'MDM Locked',
	'renewed' : "Renewed",
	'certified' : 'Certified Refurbished',
	'scratchdent' : 'Scratch & Dent'
}
const gradeNames = {
	'all' : 'Refurb - In Process',
	'first' : 'Refurb - In Process',
	'second' : 'Refurb - In Process',
	'engraved' : 'Refurb - In Process',
	'activation' : 'Refurb - In Process',
	'mdm' : 'Refurb - In Process',
	'renewed' : 'Picked - Ecomm - Renewed',
	'certified' : 'Picked - Ecomm - Certified Refurbished',
	'scratchdent' : 'Picked - Ecomm - Scratch & Dent',
}

window.onclick = function(event) {
	if (event.target == assetModal) {
		closeAssetModal();
	}
}

function closeAssetModal() {
	const content = document.getElementById('asset_modal_content');
	assetModal.classList.remove('active');
	content.innerHTML = '';
	modalData.currentPage = 1;
	modalData.requestedAssetData.length = 0;
	if (changed) {
		getItems();
	}
}

function buildPageControls() {
	const pageControls = document.getElementById('pages');
	pageControls.innerHTML = "";
	pageControls.innerHTML += `<li onclick="setCurrentPage(${Math.max(modalData.currentPage - 1, 1)})" pnum="prev" class="select-page-button">&laquo;</li>`;
	for (let i = 1; i <= modalData.maxPages; i++) {
		if (i == modalData.currentPage) {
			pageControls.innerHTML += `<li onclick="setCurrentPage(${i})" pnum="${i}" class="select-page-button active">${i}</li>`;
		} else {
			pageControls.innerHTML += `<li onclick="setCurrentPage(${i})" pnum="${i}" class="select-page-button">${i}</li>`;
		}
	}
	pageControls.innerHTML += `<li onclick="setCurrentPage(${Math.min(modalData.currentPage + 1, modalData.maxPages)})" pnum="next" class="select-page-button">&raquo;</li>`;
}

function populateModalData(rebuildPageControls=false) {
	const content = document.getElementById('asset_modal_content');
	const setAll = document.getElementById('set_all_statuses');
	setAll.innerHTML = `Set All: ${generateStatusSelect(gradeNames[assetGrade], 'all')} <img class="small_loading_image" src="../svg/blank.svg" />`;
	if (rebuildPageControls) {
		buildPageControls();
	}
	content.innerHTML = "";
	for (let i = modalData.itemsPerPage * (modalData.currentPage - 1); i < modalData.requestedAssetData.length && i < modalData.itemsPerPage * modalData.currentPage; i++) {
		const row = modalData.requestedAssetData[i];
		let newRow = "<tr>"
		row['marking'] = (parseInt(row['marking']) == 1) ? "Yes" : "No";
		newRow += `<td><a href="${CONTROLLER_BASE_URL}?m=order_detail&oNum=${row['oNum']}" target="_blank">${row['oNum']}</a></td>`;
		newRow += `<td>${row['grade']}</td>`;
		newRow += `<td>${row['marking']}</td>`;
		newRow += `<td>${row['dateMachineReceived']}</td>`;
		newRow += `<td>${generateStatusSelect(row['cStatus'], row['oNum'])}<img class="small_loading_image" src="../svg/blank.svg" /></td>`;
		newRow += `</tr>`;
		content.innerHTML += newRow;
	}
	setLoadingModalVisible(false);
	if(modalData.requestedAssetData.length > 0) {
		assetModal.classList.add('active');
	}
}

function exportAssetModalCsv() {
	const rows = [];
	const headerRow = [
		'"SP Number"',
		'"Grade"',
		'"Engraved"',
		'"Date Received"',
		'"Status"',
	];

	rows.push(headerRow);

	for (const item of modalData.requestedAssetData) {
		const newRow = [
			`"${item['oNum']}"`,
			`"${item['grade']}"`,
			`"${item['marking']}"`,
			`"${item['dateMachineReceived']}"`,
			`"${item['cStatus']}"`
		]
		rows.push(newRow);
	}

	let csvContent = "data:text/csv;charset=utf-8," + rows.map(e => e.join(",")).join("\n");
	const encodedUri = encodeURI(csvContent);
	const link = document.createElement("a");
	link.setAttribute("href", encodedUri);
	link.setAttribute("download", `asset_export_${getTimestamp(new Date())}.csv`);
	document.body.appendChild(link); // Required for FF

	link.click();
	delete link;

	/**
	 *
	 * @param {Date} time
	 */
	function getTimestamp(time) {
		let timestamp = time.getFullYear().toString();
		if (time.getMonth()+1 < 10) {
			timestamp += '0';
		}
		timestamp += (time.getMonth()+1).toString();
		if (time.getDate() < 10) {
			timestamp += '0';
		}
		timestamp += time.getDate().toString();
		if (time.getHours() < 10) {
			timestamp += '0';
		}
		timestamp += time.getHours().toString();
		if (time.getMinutes() < 10) {
			timestamp += '0';
		}
		timestamp += time.getMinutes().toString();
		if (time.getSeconds() < 10) {
			timestamp += '0';
		}
		timestamp += time.getSeconds().toString();

		return timestamp
	}
}


async function getAssetData(asset, grade) {
	setLoadingModalVisible(true);
	if (!statuses.length > 0) {
		await populateStatuses();
	}
	const assetModalTitle = document.getElementById('asset_modal_title');
	assetModalTitle.innerHTML = `${asset} - ${modalTitles[grade]}`;
	assetGrade = grade;
	const postStr = `asset=${asset}&grade=${grade}&page=${page}${(view != '') ? `&view=${view}` : ''}`;
	const url = `${CONTROLLER_BASE_URL}?m=get_asset_data`;
	makeAjaxRequest(url, postStr, "POST", returnAssetData);
	function returnAssetData() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					const data = JSON.parse(http_request.response);
					if (data) {
						const items = data['items'];
						if (items) {
							for (const item of items) {
								modalData.requestedAssetData.push(item);
							}
						}
					}
					if (data && data['items'] && data['items'].length > 0) {
						assetModalTitle.innerHTML += ` - Item Count: ${data['items'].length}`
						updateMaxPages();
						populateModalData(true);
					} else {
						setLoadingModalVisible(false);
					}
					break;
				default:
					alert('Error with AJAX request.');
			}
		}
	}
}

/**
 *
 * @param {HTMLSelectElement} obj
 */
function setStatus(obj) {
	const oNum = obj.getAttribute('onum');
	const setAll = (oNum == 'all');
	const statusInputs = document.getElementsByClassName('status-change-select');

	if (setAll) {
		for(const input of statusInputs) {
			input.value = obj.value;
		}
	}
	// loadingImage.src = loadingSrc;
	// console.log(`Setting ${oNum} to ${newStatus}.`);
	// console.log(setAll);
}

function saveNewStatuses() {
	const loadingSrc = '../svg/loading2.svg';
	const finishedSrc = '../svg/checkmark.svg';
	const originalStatus = gradeNames[assetGrade];
	/**
	 * @type {HTMLCollectionOf<HTMLSelectElement>}
	 */
	const statusInputs = document.getElementsByClassName('status-change-select');
	const oNumsToChange = {};
	/**
	 * @type {HTMLImageElement[]}
	 */
	const loadingImages = [];
	for (const input of statusInputs) {
		if (input.getAttribute('onum') != 'all' && input.value != originalStatus) {
			if (!oNumsToChange[input.value]) {
				oNumsToChange[input.value] = [];
			}
			oNumsToChange[input.value].push(input.getAttribute('onum'));
			loadingImages.push(input.nextSibling);
		}
	}
	// const postStr = JSON.stringify(oNumsToChange);
	// let postStr = 'nums=' + JSON.stringify(oNumsToChange);
	let postStr = '';
	for (const key in oNumsToChange) {
		if (postStr.length > 0) {
			postStr += "&";
		}
		postStr += `${sanitizeString(key)}=${oNumsToChange[key]}`;
	}
	const url = `${CONTROLLER_BASE_URL}?m=inventory_save_new_statuses`;
	for (const img of loadingImages) {
		img.src = loadingSrc;
	}
	console.log(postStr);
	makeAjaxRequest(url, postStr, "POST", returnNewStatuses);
	function returnNewStatuses() {
		if (http_request.readyState == 4) {
			switch (http_request.status) {
				case 200:
					if (parseInt(http_request.response) > 0) {
						changed = true;
					}
					for (const img of loadingImages) {
						img.src = finishedSrc;
					}
					break;
				default:
					alert('Error with AJAX request.');
			}
		}
	}
}

function updateMaxPages() {
	modalData.maxPages = Math.ceil(modalData.requestedAssetData.length / modalData.itemsPerPage);
	if (modalData.currentPage > modalData.maxPages) {
		modalData.currentPage = modalData.maxPages;
	}
}

function generateStatusSelect(status, oNum) {
	return `<select class="borderless status-change-select" onchange="setStatus(this)" onum="${oNum}">${generateStatusOptions(status)}</select>`;
}

function generateStatusOptions(status) {
	let optionClause = "";
	for (const stat of statuses) {
		if (stat == status) {
			optionClause += `<option selected value="${stat}">${stat}</option>`;
		} else {
			optionClause += `<option value="${stat}">${stat}</option>`;
		}
	}
	return optionClause;
}

function setItemsPerPage(num) {
	/**
	 * @type {HTMLCollectionOf<HTMLButtonElement>}
	 */
	const itemsPerPageButtons = document.getElementsByClassName('items-per-page-selection');
	for (const button of itemsPerPageButtons) {
		if (parseInt(button.getAttribute('itemnum')) == num) {
			button.classList.add('active');
			button.disabled = true;
		} else {
			button.classList.remove('active');
			button.disabled = false;
		}
	}
	modalData.itemsPerPage = num;
	updateMaxPages();
	populateModalData(true);
}

function setCurrentPage(num) {
	/**
	 * @type {HTMLCollectionOf<HTMLLIElement}
	 */
	const pageSelectButtons = document.getElementsByClassName('page-select-button');
	modalData.currentPage = num;

	populateModalData(true);
}

function sanitizeString(str) {
	if (str) {
		str = str.replace('#', 'TTS_NUM');
		str = str.replace('&', 'TTS_AMP');
		str = str.replace('?', 'TTS_QST');
	}
	return str;
}

async function populateStatuses() {
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=inventory_getStatuses`)
		.then(response => { return response });

	const responseText = await response.text();
	try {
		const returnedStatuses =  JSON.parse(responseText);
		for(const status of returnedStatuses) {
			if (status) {
				statuses.push(status);
			}
		}
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
		return null;
	}
}
