class Buyback {

	constructor(data) {
		/**
		 * @type {string}
		 */
		const createdDateTimeString = data['createdDateTime'];

		/**
		 * @type {string}
		 */
		this.id = data['dbId'];
		/**
		 * @type {string}
		 */
		this.zohoId = data['zohoId'];
		/**
		 * @type {string}
		 */
		this.mtsNumber = data['mtsNumber'];
		/**
		 * @type {string}
		 */
		this.zohoStatus = data['zohoStatus'];
		/**
		 * @type {string}
		 */
		this.subject = data['subject'];
		/**
		 * @type {boolean}
		 */
		this.fx = data['fx'];
		/**
		 * @type {boolean}
		 */
		this.gd = data['gd'];
		/**
		 * @type {string}
		 */
		this.notes = (data['notes']) ? data['notes'] : '';

		this.createdDateTime = new Date(createdDateTimeString.replace(' ', 'T'));

		this.account = new Account(data['account']);
		this.status = new Status(data['status']);
		this.newStatusId = this.getStatus().getId();

		/**
		 * @type {LineItem[]}
		 */
		this.lineItems = [];

		this.buybackRows = {
			/**
			 * @type {HTMLTableRowElement}
			 */
			main : null,
			/**
			 * @type {HTMLTableRowElement[]}
			 */
			subordinate : []
		}


		this.expanded = false;
		this.isValid = true;
		this.changed = false;


		for (const lineItem of data['lineItems']) {
			this.lineItems.push(new LineItem(lineItem));
		}
	}
	getId() {
		return this.id;
	}
	getZohoId() {
		return this.zohoId;
	}
	getMtsNumber() {
		return this.mtsNumber;
	}
	getZohoStatus() {
		return this.zohoStatus;
	}
	getSubject() {
		return this.subject;
	}
	getNotes() {
		return this.notes;
	}
	getFx() {
		return this.fx;
	}
	getGd() {
		return this.gd;
	}
	getCreatedDateTime() {
		return this.createdDateTime;
	}
	getAccount() {
		return this.account;
	}
	getStatus() {
		return this.status;
	}
	getNewStatusId() {
		return this.newStatusId;
	}
	getLineItems() {
		return this.lineItems;
	}
	getLineItemByIndex(index) {
		return this.lineItems[index];
	}
	getIsValid() {
		return this.isValid;
	}
	getIsChanged() {
		return this.changed;
	}

	setNotes(newValue) {
		if (typeof newValue == 'string') {
			this.notes = newValue;
		}
	}
	setFx(newValue) {
		if (typeof newValue == 'boolean') {
			this.fx = newValue;
		}
	}
	setGd(newValue) {
		if (typeof newValue == 'boolean') {
			this.gd = newValue;
		}
	}
	setStatus(newValue) {
		if (newValue instanceof Status) {
			this.status = newValue;
		}
	}
	setNewStatusId(statusId) {
		this.newStatusId = statusId;
	}
	setIsValid(newValue) {
		if (typeof newValue == 'boolean') {
			this.isValid = newValue;
		}
	}
	setIsChanged(newValue=true) {
		if (typeof newValue == 'boolean') {
			this.changed = newValue;
		}
	}
	addLineItem(lineItem) {
		if (lineItem instanceof LineItem) {
			this.lineItems.push(lineItem);
		}
	}

	getFieldByName(fieldName) {
		if (fieldName == 'mtsNumber') {
			return this.getMtsNumber();
		}
		else if (fieldName == 'accountName') {
			return this.getAccount().getName();
		}
		else if (fieldName == 'createdDateTime') {
			return this.getCreatedDateTime();
		}
	}

	/* getters/setters end */

	conformVisibilityToValid() {
		const valid = this.getIsValid();
		this.buybackRows.main.hidden = !valid;
		for (const row of this.buybackRows.subordinate) {
			row.hidden = !valid;
		}
	}

	getSaveableFieldsAsObject() {
		if (!this.getIsChanged()) {
			return null;
		}
		const toSave = {
			dbId: this.getId(),
			fields: {
				notes : this.getNotes(),
				fx : (this.getFx()) ? '1' : '0',
				gd : (this.getGd()) ? '1' : '0',
			}
		};
		if (this.getNewStatusId() != this.getStatus().getId()) {
			toSave.fields['statusId'] = this.getNewStatusId();
		}
		return toSave;
	}

	/* generators start */
	/**
	 *
	 * @param {ColumnDefinition} columns
	 */
	 generateRows(columns) {
		if (this.buybackRows.main) {
			delete this.buybackRows.main;
			this.buybackRows.main = null;
		}
		this.buybackRows.subordinate.length = 0;

		/**
		 * @type {HTMLTableRowElement[]}
		 */
		const rows = [];
		const mainRow = this.generateMainRow(columns);
		this.buybackRows.main = mainRow;
		rows.push(mainRow);
		for (let i = 1; i < this.getLineItems().length; i++) {
			const subordinateRow = this.generateSubordinateRow(columns, i);
			this.buybackRows.subordinate.push(subordinateRow);
			rows.push(subordinateRow);
		}
		return rows;

	}

	/**
	 *
	 * @param {ColumnDefinition[]} columns
	 */
	generateMainRow(columns) {
		const rowspan = Math.max(1, this.getLineItems().length);
		const row = document.createElement('tr');
		const index = 0;
		for (const column of columns) {
			const identifier = column.getIdentifier();
			if (column.spansMultipleRows) {
				try {
					row.appendChild(this.generateTableCellByIdentifier(identifier, index, rowspan))
				}
				catch(err) {
					console.error(err);
					row.appendChild(this.generatePlaceholderCell(rowspan));
				}
			}
			else {
				try {
					row.appendChild(this.generateTableCellByIdentifier(identifier, index))
				}
				catch(err) {
					console.error(err);
					row.appendChild(this.generatePlaceholderCell());
				}
			}
		}
		row.appendChild(this.generateSaveChangesCell(rowspan));
		return row;
	}

	/**
	 *
	 * @param {ColumnDefinition[]} columns
	 * @param {number} index
	 */
	generateSubordinateRow(columns, index) {
		const row = document.createElement('tr');
		for (const column of columns) {
			if (!column.spansMultipleRows) {
				const identifier = column.getIdentifier();
				try {
					row.appendChild(this.generateTableCellByIdentifier(identifier, index))
				}
				catch(err) {
					console.error(err);
					row.appendChild(this.generatePlaceholderCell());
				}
			}
		}
		return row;
	}

	/**
	 *
	 * @param {string} identifier
	 * @param {number} index
	 * @param {number} rowspan
	 */
	generateTableCellByIdentifier(identifier, index=0, rowspan=null) {
		if (identifier == 'mtsNumber') {
			return this.generateCellWithLink(this.getMtsNumber(), `https://crm.zoho.com/crm/org26161413/tab/Invoices/${this.getZohoId()}`, rowspan);
		}
		else if (identifier == 'accountName') {
			return this.generateCellWithLink(this.getAccount().getName(), `https://crm.zoho.com/crm/org26161413/tab/Accounts/${this.getAccount().getZohoId()}`, rowspan);
		}
		else if (identifier == 'createdDateTime') {
			return this.generateCellWithText(this.getCreatedDateTime().toLocaleDateString(), rowspan);
		}
		else if (identifier == 'lineItemSku') {
			return this.generateCellWithLink(this.getLineItemByIndex(index).getFieldByNameAsText(identifier), `https://crm.zoho.com/crm/org26161413/tab/Products/${this.getLineItemByIndex(index).getZohoId()}`);
		}
		else if (identifier == 'lineItemQuantity') {
			return this.generateCellWithText(this.getLineItemByIndex(index).getFieldByNameAsText(identifier));
		}
		else if (identifier == 'notes') {
			return this.generateNotesCell(rowspan);
		}
		else if (identifier == 'fx') {
			return this.generateFXCell(rowspan);
		}
		else if (identifier == 'gd') {
			return this.generateGDCell(rowspan);
		}
		else if (identifier == 'status') {
			return this.generateStatusCell(rowspan);
		}
		else {
			return this.generatePlaceholderCell(rowspan);
		}
	}

	generateNotesCell(rowspan=1) {
		const cell = document.createElement('td');
		const textArea = document.createElement('textarea');
		textArea.innerText = this.getNotes();
		textArea.placeholder = 'Enter notes';
		textArea.setAttribute('dbId', this.getId());
		textArea.setAttribute('oninput', 'updateNotes(this)');
		cell.appendChild(textArea);
		cell.rowSpan = rowspan;
		return cell;
	}

	generateFXCell(rowspan=null) {
		const cell = document.createElement('td');
		const checkbox = document.createElement('input');
		checkbox.type = 'checkbox';
		cell.appendChild(checkbox);
		checkbox.checked = this.getFx();
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		checkbox.setAttribute('dbId', this.getId());
		checkbox.setAttribute('onchange', 'updateFx(this)');
		return cell;
	}

	generateGDCell(rowspan=null) {
		const cell = document.createElement('td');
		const checkbox = document.createElement('input');
		checkbox.type = 'checkbox';
		cell.appendChild(checkbox);
		checkbox.checked = this.getGd();
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		checkbox.setAttribute('dbId', this.getId());
		checkbox.setAttribute('onchange', 'updateGd(this)');
		return cell;
	}

	generateStatusCell(rowspan=null) {
		const cell = document.createElement('td');
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		const select = document.createElement('select');
		cell.appendChild(select);
		select.classList.add('borderless');
		for (const status of STATUS_ARRAY) {
			const selected = (this.getStatus().getIdentifier() == status.getIdentifier());
			select.appendChild(status.getStatusOption(selected))
		}
		select.setAttribute('dbId', this.getId());
		select.setAttribute('onchange', 'updateStatus(this)');
		return cell;
	}

	generateSaveChangesCell(rowspan) {
		const cell = document.createElement('td');
		cell.rowSpan = rowspan;
		cell.classList.add('halign-center');
		const saveButton = document.createElement('button');
		cell.appendChild(saveButton);
		saveButton.innerText = "Save Changes";
		saveButton.setAttribute('onclick', `saveChanges('${this.getId()}')`);
		return cell;
	}

	/* generic generators */

	generatePlaceholderCell(rowspan=null) {
		return this.generateCellWithText('placeholder', rowspan);
	}

	/**
	 *
	 * @param {string|boolean} innerText
	 */
	generateCellWithText(innerText, rowspan=null) {
		const cell = document.createElement('td');
		if (typeof innerText == 'boolean') {
			cell.innerText = (innerText) ? 'Yes' : 'No';
		}
		else {
			cell.innerText = (innerText) ? innerText : 'N/A';
		}
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	generateCellWithLink(innerText='', href='', rowspan=null) {
		const cell = document.createElement('td');
		const link = document.createElement('a');
		cell.appendChild(link);
		if (href) {
			link.href = href;
		}
		link.target = "_blank";
		if (innerText) {
			link.innerText = innerText;
		}
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	generateCheckboxCell(checked=false, rowspan=null) {
		const cell = document.createElement('td');
		const checkbox = document.createElement('input');
		checkbox.type = 'checkbox';
		cell.appendChild(checkbox);
		checkbox.checked = checked;
		if (rowspan) {
			cell.rowSpan = rowspan;
		}
		return cell;
	}

	/**
	 *
	 * @param {number} num
	 */
	displayNumberAsCurrency(num) {
		try {
			return num.toLocaleString('en-US', {style : 'currency', currency : 'USD', minimumFractionDigits : 0, maximumFractionDigits : 0});
		}
		catch (err) {
			console.warn(err);
			return 'N/A';
		}
	}
}
