const CONTROLLER_BASE_URL = "admin_controller.php";

const SORT_ASC_URL = '../svg/sort_asc.svg';
const SORT_DESC_URL = '../svg/sort_desc.svg';
const SORT_NONE_URL = '../svg/sort.svg';
const DEFAULT_SORT = 'createdDateTime';

const ACTIVE_BUYBACK_DICT = {};
const ACTIVE_BUYBACK_ARRAYS = {

};
const STATUS_DICT = {};
/**
 * @type {Status[]}
 */
const STATUS_ARRAY = [];
const PAGE_ELEMENTS = {
	/**
	 * @type {HTMLDivElement}
	 */
	tableContainer : document.getElementById('tableContainer'),
	/**
	 * @type {HTMLCollectionOf<HTMLImageElement>}
	 */
	sortIcons : document.getElementsByClassName('sort_icon'),
	templates : {}
};

const COLUMN_DEFINITIONS = {
	mtsNumber: new ColumnDefinition('MTS Number', 'mtsNumber', {isSortable: true, isFilterable: true}),
	accountName: new ColumnDefinition('Account Name', 'accountName', {isSortable: true, isFilterable: true}),
	invoiceDate: new ColumnDefinition('Invoice Date', 'createdDateTime', {isSortable: true}),
	sku: new ColumnDefinition('SKU', 'lineItemSku', {spansMultipleRows: false}),
	quantity: new ColumnDefinition('Qty', 'lineItemQuantity', {spansMultipleRows: false}),
	notes: new ColumnDefinition('Notes', 'notes'),
	fx: new ColumnDefinition('FX', 'fx'),
	gd: new ColumnDefinition('GD', 'gd'),
	status: new ColumnDefinition('Status', 'status')
};

const columns = {
	generic : [
		COLUMN_DEFINITIONS.mtsNumber,
		COLUMN_DEFINITIONS.accountName,
		COLUMN_DEFINITIONS.invoiceDate,
		COLUMN_DEFINITIONS.sku,
		COLUMN_DEFINITIONS.quantity,
		COLUMN_DEFINITIONS.notes,
		COLUMN_DEFINITIONS.fx,
		COLUMN_DEFINITIONS.gd,
		COLUMN_DEFINITIONS.status,
	]
};

const filter = {};
const sort = {
	by: DEFAULT_SORT,
	order: 1,
};

/**
 * @type {Response}
 */
let lastResponse;

init();

async function init() {
	clearFilter(); // cleanup for any extraneous data from previous

	const returnedData = await getActiveBuybacks();
	console.log(returnedData);
	setLoadingModalVisible(false);
	if (returnedData && typeof returnedData == 'object') {
		if (returnedData['statuses']) {
			processStatuses(returnedData['statuses']);
			generateTableTemplates();
		}
		if (returnedData['activeBuybacks']) {
			processActiveBuybacks(returnedData['activeBuybacks']);
		}
	}
	generateTableElements();
	sortBuybackArrays();
	renderHeaders();
	renderBody();
}

async function zohoRefreshBuybacks() {
	setLoadingModalVisible(true);
	const response = await fetch(`${CONTROLLER_BASE_URL}?m=buybacks_refreshActiveBuybacks`)
		.then(response => response.text())
		.then(data => { return data }
	);
	init();
}

async function getActiveBuybacks() {
	const response =  await fetch(`${CONTROLLER_BASE_URL}?m=buybacks_fetchActiveBuybacks`)
		.then(response => { return response });
	const responseText = await response.text();
	try {
		const data = JSON.parse(responseText);
		return data;
	}
	catch (err) {
		console.error(err);
		console.log(responseText);
		return null;
	}
}

function generateTableTemplates() {
	emptyDict();
	for(const status of STATUS_ARRAY) {
		const templateData = {
			identifier : status.getIdentifier(),
			name : status.getName()
		};
		const tableTemplate = new TableTemplate(templateData);
		PAGE_ELEMENTS.templates[status.getIdentifier()] = tableTemplate;
	}


	function emptyDict() {
		for (const element in PAGE_ELEMENTS) {
			delete PAGE_ELEMENTS.templates[element];
		}
	}
}

function generateTableElements() {
	PAGE_ELEMENTS.tableContainer.innerHTML = '';
	for(const status of STATUS_ARRAY) {
		/**
		 * @type {TableTemplate}
		 */
		const associatedTemplate = PAGE_ELEMENTS.templates[status.getIdentifier()];
		if (associatedTemplate.active) {
			const templateElements = associatedTemplate.generateSection();
			PAGE_ELEMENTS.tableContainer.appendChild(templateElements.heading);
			PAGE_ELEMENTS.tableContainer.appendChild(templateElements.table);
		}
	}
}

function processStatuses(statusData) {
	emptyDict();
	STATUS_ARRAY.length = 0;
	for (const statusDatum of statusData) {
		const status = new Status(statusDatum);
		STATUS_DICT[status.getIdentifier()] = status;
		STATUS_ARRAY.push(status);
	}

	function emptyDict() {
		for (const element in STATUS_DICT) {
			delete STATUS_DICT[element];
		}
	}
}

function processActiveBuybacks(activeBuybacks) {
	emptyDict();
	for (const buybackData of activeBuybacks) {
		const buyback = new Buyback(buybackData);
		ACTIVE_BUYBACK_DICT[buyback.getId()] = buyback;
		const statusIdentifier = buyback.getStatus().getIdentifier();
		/**
		 * @type {TableTemplate}
		 */
		const associatedTemplate = PAGE_ELEMENTS.templates[statusIdentifier];
		associatedTemplate.active = true;
	}

	function emptyDict() {
		for (const element in ACTIVE_BUYBACK_DICT) {
			delete ACTIVE_BUYBACK_DICT[element];
		}
	}
}

function sortBuybackArrays() {
	emptyArrays();
	for (const buybackId in ACTIVE_BUYBACK_DICT) {
		/**
		 * @type {Buyback}
		 */
		const buyback = ACTIVE_BUYBACK_DICT[buybackId];
		const buybackStatusIdentifier = buyback.getStatus().getIdentifier();
		if (!ACTIVE_BUYBACK_ARRAYS[buybackStatusIdentifier]) {
			ACTIVE_BUYBACK_ARRAYS[buybackStatusIdentifier] = [];
		}
		/**
		 * @type {Buyback[]}
		 */
		const activeArray = ACTIVE_BUYBACK_ARRAYS[buybackStatusIdentifier];
		activeArray.push(buyback);
	}

	function emptyArrays() {
		for (const category in ACTIVE_BUYBACK_ARRAYS) {
			ACTIVE_BUYBACK_ARRAYS[category].length = 0;
			delete ACTIVE_BUYBACK_ARRAYS[category];
		}
	}
}

function renderHeaders() {
	for (const buybackStatus in ACTIVE_BUYBACK_ARRAYS) {
		if (ACTIVE_BUYBACK_ARRAYS[buybackStatus].length > 0) {
			renderHeaderRow(buybackStatus);
		}
	}

	/**
	 *
	 * @param {string} status
	 */
	function renderHeaderRow(status) {
		/**
		 * @type {TableTemplate}
		 */
		const associatedTemplate = PAGE_ELEMENTS.templates[status];
		const headerRow = associatedTemplate.getHeaderRow();

		headerRow.innerHTML = '';
		for (const column of columns.generic) {
			// headerRow.innerHTML += column.buildHeader();
			headerRow.appendChild(column.getHeaderCellElement());
		}
		headerRow.appendChild(generateActionHeader());


		function generateActionHeader() {
			const headerCell = document.createElement('th');
			const button = document.createElement('button');
			headerCell.appendChild(button);
			button.innerText = "Save All Changes";
			button.setAttribute('onclick', 'saveAllChanges(this)');
			return headerCell;
		}
	}
}

function renderBody(refreshValidEntries = true) {
	// if (refreshValidEntries) {
	// 	getValidEntries();
	// }
	for(const category in ACTIVE_BUYBACK_ARRAYS) {
		/**
		 * @type {TableTemplate}
		 */
		const associatedTemplate = PAGE_ELEMENTS.templates[category];
		const tableBody = associatedTemplate.getBody();
		tableBody.innerHTML = '';
		/**
		 * @type {Buyback[]}
		 */
		const activeBuybackArray = ACTIVE_BUYBACK_ARRAYS[category];
		for (const buyback of activeBuybackArray) {
			const buybackRows = buyback.generateRows(columns.generic);
			for (const row of buybackRows) {
				tableBody.appendChild(row);
			}
		}
	}
}

/**
 *
 * @param {string} quoteZohoId Zoho ID of the quote to expand/contract
 */
function toggleQuoteExpanded(quoteZohoId) {
	/**
	 * @type {OpenQuote}
	 */
	const openQuote = ACTIVE_BUYBACK_DICT[quoteZohoId];
	openQuote.toggleExpanded();
}

/**
 *
 * @param {HTMLInputElement} gdCheckbox
 */
function updateGd(gdCheckbox) {
	const dbId = gdCheckbox.getAttribute('dbId');
	/**
	 * @type {Buyback}
	 */
	const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
	const newValue = gdCheckbox.checked;
	activeBuyback.setGd(newValue);
	activeBuyback.setIsChanged(true);
}

/**
 *
 * @param {HTMLInputElement} fxCheckbox
 */
function updateFx(fxCheckbox) {
	const dbId = fxCheckbox.getAttribute('dbId');
	/**
	 * @type {Buyback}
	 */
	const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
	const newValue = fxCheckbox.checked;
	activeBuyback.setFx(newValue);
	activeBuyback.setIsChanged(true);
}

/**
 *
 * @param {HTMLSelectElement} statusSelect
 */
function updateStatus(statusSelect) {
	const dbId = statusSelect.getAttribute('dbId');
	/**
	 * @type {Buyback}
	 */
	const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
	const newStatusId = statusSelect.value;
	activeBuyback.setNewStatusId(newStatusId);
	activeBuyback.setIsChanged(true);
}

/**
 *
 * @param {HTMLTextAreaElement} noteInput
 */
function updateNotes(noteInput) {
	const dbId = noteInput.getAttribute('dbId');
	/**
	 * @type {Buyback}
	 */
	const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
	const newNotes = noteInput.value.trim();
	activeBuyback.setNotes(newNotes);
	activeBuyback.setIsChanged();
}

async function saveAllChanges(input) {
	for (const statusIdentifier in ACTIVE_BUYBACK_ARRAYS) {
		/**
		 * @type {Buyback[]}
		 */
		const activeArray = ACTIVE_BUYBACK_ARRAYS[statusIdentifier];
		for (const buyback of activeArray) {
			if (buyback.getIsChanged()) {
				await saveChanges(buyback.getId(), false);
			}
		}
	}

	init();
}

/**
 *
 * @param {string} dbId
 */
async function saveChanges(dbId, render=true) {
	/**
	 * @type {Buyback}
	 */
	const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
	const dataToSave = activeBuyback.getSaveableFieldsAsObject();
	if (!dataToSave) {
		return false;
	}
	setLoadingModalVisible(true);
	console.log(dataToSave);
	const stringified = JSON.stringify(dataToSave).replaceAll("\\", '').replaceAll("\"[", "[").replaceAll("]\"", "]");
	const rawResponse = await fetch(`${CONTROLLER_BASE_URL}?m=buybacks_saveChanges`, {
		method: 'POST',
		headers: {
			'Accept' : 'application/json',
			'Content-Type': 'application/json'
		},
		body: stringified
	});
	const content = await rawResponse.text();

	console.log(content);

	if (render) {
		init();
	}
	return true;
}

/* Filter functions below */

/**
 *
 * @param {HTMLInputElement} input
 */
function filterInput(input) {
	filter[input.getAttribute('filter')] = input.value;
	const sameNameInputs = document.getElementsByClassName(input.getAttribute('filter') + '_filter');
	for(const elem of sameNameInputs) {
		elem.value = input.value;
	}
	validateEntries();
}

function clearFilter() {
	for (const entry in filter) {
		delete filter[entry];
	}
}

function validateEntries() {
	for (const dbId in ACTIVE_BUYBACK_DICT) {
		let valid = true;
		/**
		 * @type {Buyback}
		 */
		const activeBuyback = ACTIVE_BUYBACK_DICT[dbId];
		for (const entry in filter) {
			/**
			 * @type {string}
			 */
			const filterValue = filter[entry];
			if (filter[entry] == '') {
				delete filter[entry];
			}
			else if (typeof activeBuyback.getFieldByName(entry) == 'string' && !activeBuyback.getFieldByName(entry).toString().toLowerCase().includes(filterValue.toLowerCase())) {
				valid = false;
			}
		}
		activeBuyback.setIsValid(valid);
		activeBuyback.conformVisibilityToValid();
	}

	/**
	 * Filter valid products
	 * @param {Buyback} record Record to check
	 * @param {string} filterValue Value to filter on
	 */
	function validateProducts(record, filterValue) {
		let valid = false;
		for (const product of record.lineItems) {
			if (product.filter.toLowerCase().includes(filterValue.toLowerCase())) {
				product.setIsVisible(true);
				valid = true;
			}
			else {
				product.setIsVisible(false);
			}
		}
		return valid;
	}
}

/* Sort functions below */

function setSort(newSort) {
	if (newSort == sort.by) {
		sort.order *= -1;
	} else {
		sort.by = newSort;
	}
	sortRecords();
	renderBody(false);

	function sortRecords() {
		for (const statusIdentifier in ACTIVE_BUYBACK_ARRAYS) {
			/**
			 * @type {Buyback[]}
			 */
			const currentArray = ACTIVE_BUYBACK_ARRAYS[statusIdentifier];
			currentArray.sort((a,b) => sortFunction(a,b));
		}
		for (const icon of PAGE_ELEMENTS.sortIcons) {
			if (icon.parentElement.getAttribute('sort_order') != sort.by) {
				icon.src = SORT_NONE_URL;
			} else {
				if (sort.order == 1) {
					icon.src = SORT_ASC_URL;
				} else if (sort.order == -1) {
					icon.src = SORT_DESC_URL;
				}
			}
		}

		/**
		 *
		 * @param {Buyback} a
		 * @param {Buyback} b
		 */
		function sortFunction(a,b) {
			const sortValueA = a.getFieldByName(sort.by);
			const sortValueB = b.getFieldByName(sort.by);
			if(isNaN(sortValueA) || (sortValueA instanceof Date && sortValueB instanceof Date)) {
				if(sortValueA < sortValueB) {
					return -1 * sort.order;
				}
				if(sortValueA > sortValueB) {
					return 1 * sort.order;
				}
			} else {
				if(parseInt(sortValueA) < parseInt(sortValueB)) {
					return -1 * sort.order;
				}
				if(parseInt(sortValueA) > parseInt(sortValueB)) {
					return 1 * sort.order;
				}
			}
		}
	}
}
