<?php

class PackingSlips extends Database {
	private $shipFromString = "Source+\n1530 Montague Expwy\nSan Jose, CA 95131-1408\nUnited States";
	private $packingSlipFolder = '';

	function __construct() {
		$this->packingSlipFolder = HTML_FOLDER . "packingslips/";
	}
	public function packingSlipIsGenerated($orderId, $channel) {
		$filename = $this->getPackingSlipPath($orderId, $channel, 'F');
		return file_exists($filename);
	}

	/**
	 * Returns the path for packing slips
	 * @param string $orderId Order ID
	 * @param string $channel Sale channel eg. newegg, ebay
	 * @param string $dest Destination - 'F' returns full file path, else just filename
	 * @return string
	 */
	private function getPackingSlipPath($orderId, $channel, $dest='I') {
		if ($channel == 'back_market') {
			$channel = 'backmarket';
		}
		elseif ($channel == 'ebay_auctions' || $channel == 'ebay_others') {
			$channel = 'ebay';
		}
		if ($dest == 'F') {
			return HTML_FOLDER . "packingslips/" . $orderId . '_' . $channel . '_packingslip.pdf';
		}
		else {
			return $orderId . '_' . $channel . '_packingslip.pdf';
		}
	}

	public function generatePackingSlip($orderId, $channel, $destination='I') {
		$channel = str_replace('_', '', strtolower(trim($channel)));
		if (substr($channel, 0, 4) == 'ebay') {
			$channel = 'ebay';
			$this->generateEbayPackingSlip($orderId, $destination);
		}
		elseif ($channel == 'newegg') {
			$this->generateNeweggPackingSlip($orderId, $destination);
		}
		elseif ($channel == 'backmarket') {
			$this->generateBackMarketPackingSlip($orderId, $destination);
		}
	}

	function generateNeweggPackingSlip($orderId, $destination='I', $debug=false) {
		if ($debug) {
			echo "\t\tIn generateNeweggPackingSlip($orderId, $destination):\n";
		}
		global $neweggApiObj;
		$response = json_decode($neweggApiObj->getOrder($orderId));
		$order = $response->ResponseBody->OrderInfoList[0];
		$channel = 'newegg';
		$pdfData = $this->buildPdfData($order, $orderId, $channel);
		$this->buildPdf($pdfData, $destination);
	}

	function generateEbayPackingSlip($orderId, $destination='I') {
		global $ebayApiObj;
		$order = json_decode($ebayApiObj->getOrder($orderId));
		$channel = "ebay";
		$pdfData = $this->buildPdfData($order, $orderId, $channel);
		$this->buildPdf($pdfData, $destination);
	}

	function generateBackMarketPackingSlip($orderId, $destination='I') {
		global $backmarketObj;
		$order = json_decode($backmarketObj->getOrder($orderId));
		$channel = "backmarket";
		$pdfData = $this->buildPdfData($order, $orderId, $channel);
		$this->buildPdf($pdfData, $destination);
	}

	private function buildPdfData(&$order, $orderId, $channel) {
		$logo = HTML_FOLDER.'img/SPLogo.png';
		$pdfData = new stdClass;
		$pdfData->orderId = $orderId;
		$pdfData->channel = $channel;
		$pdfData->logo = $logo;
		if ($channel == 'ebay') {
			$channelProperName = "eBay";
			$thirdOrderHeader = 'Sales Record #';
			$pdfData->infoUrl = 'www.ebay.com';

			// Parse shipping/address data
			$shippingData = $order->fulfillmentStartInstructions[0]->shippingStep;
			$addressData = $shippingData->shipTo->contactAddress;
			$customerName = $shippingData->shipTo->fullName;
			$shipToString = "$customerName\n$addressData->addressLine1 $addressData->addressLine2\n$addressData->city, $addressData->stateOrProvince $addressData->postalCode\n";
			$shipToString .= $addressData->countryCode == 'US' ? 'United States' : $addressData->countryCode;
			$shipService = $shippingData->shippingServiceCode;
			if (substr($shipService, 0, 3) == 'UPS' && substr($shipService, 0, 4) != 'UPS ') {
				$shipService = substr($shipService, 0, 3) . ' ' . substr($shipService, 3);
			}
			$orderDate = DateTime::createFromFormat('Y-m-d', substr($order->creationDate,0, strpos($order->creationDate, 'T')))->format('m/d/Y');

			// Build item data arrays
			$pdfData->itemData = array();
			foreach ($order->lineItems as $itemInfo) {
				$itemData = new stdClass;
				$itemData->quantity = $itemInfo->quantity;
				$itemData->description = "$itemInfo->title\n\nItem: $itemInfo->legacyItemId\n\nSKU: $itemInfo->sku";
				array_push($pdfData->itemData, $itemData);
			}

		}
		elseif ($channel == 'newegg') {
			$channelProperName = "Newegg";
			$thirdOrderHeader = 'Cust. PO/Ref#';
			$pdfData->infoUrl = 'www.newegg.com';

			// Parse address data
			$shipToString = "$order->CustomerName\n$order->ShipToAddress1 $order->ShipToAddress2\n$order->ShipToCityName, $order->ShipToStateCode $order->ShipToZipCode\n";
			$shipToString .= ucwords(strtolower($order->ShipToCountryCode));
			// $shipService = substr($order->ShipService,0, strpos($order->ShipService, '(')) . "\n" . substr($order->ShipService,strpos($order->ShipService, '('));
			$shipService = $order->ShipService;
			$orderDate = substr($order->OrderDate,0, strpos($order->OrderDate, ' '));

			// Build item data arrays
			$pdfData->itemData = array();
			foreach ($order->ItemInfoList as $itemInfo) {
				$itemData = new stdClass;
				$itemData->quantity = $itemInfo->OrderedQty;
				$itemData->description = "$itemInfo->Description\n\nItem: $itemInfo->NeweggItemNumber\n\nSKU: $itemInfo->SellerPartNumber";
				array_push($pdfData->itemData, $itemData);
			}
		}
		elseif ($channel == 'backmarket') {
			$channelProperName = "Back Market";
			$thirdOrderHeader = 'Sales Record #';
			$pdfData->infoUrl = 'www.backmarket.com';


			$addressData = $order->shipping_address;

			$shipToString = "$addressData->first_name $addressData->last_name\n$addressData->street $addressData->street2\n$addressData->city, $addressData->state_or_province $addressData->postal_code\n";
			$shipToString .= $addressData->country == 'US' ? 'United States' : $addressData->country;
			$shipService = $order->shipper;
			$orderDate = DateTime::createFromFormat(DATE_ATOM, $order->date_creation)->format('m/d/Y');

			// Build item data arrays
			$pdfData->itemData = array();
			foreach ($order->orderlines as $itemInfo) {
				$itemData = new stdClass;
				$itemData->quantity = $itemInfo->quantity;
				$itemData->description = "$itemInfo->product\n\nItem: $itemInfo->product_id\n\nSKU: $itemInfo->listing";
				array_push($pdfData->itemData, $itemData);
			}
		}


		// Insert shipping data into pdfData
		$pdfData->shipTo = $shipToString;
		$pdfData->shipFrom = $this->shipFromString;
		$pdfData->shipService = $shipService;
		$pdfData->orderDate = $orderDate;

		// Construct order data cells
		$pdfData->orderDataCellHeaders = array('Order Number', 'Order Date', $thirdOrderHeader, 'Channel', 'Shipping Method');
		$pdfData->orderDataCellContent = array($orderId, $pdfData->orderDate, $order->salesRecordReference, $channelProperName, $pdfData->shipService);

		// Build item header array
		$pdfData->itemDataCellHeaders = array('Item Info', 'Quantity');


		return $pdfData;
	}

	private function buildPdf($data, $destination='I') {
		$infoString = "For questions or issues regarding your order, including returns, please contact us by visiting $data->infoUrl, logging into your account, and viewing your order details.";
		require_once('pdf/class_packingslip_pdf.php');
		$yStart = 30;
		$yAdded = 0;
		$font = 'Arial';
		$pdf = new PackingSlipPDF();
		$pdf->imgHeader = $data->logo;
		$pdf->SetTopMargin(10);
		$pdf->AddPage('P','Letter');
		$pdf->SetFont($font,'',12);
		$pdf->SetXY(10, $yStart);
		$pdf->Rect(10, $pdf->GetY()+5, 194.8, 36);
		$yAdded += 12;
		$pdf->SetXY(12, $yStart + $yAdded);
		$pdf->Cell(20,3,'Ship From:');
		$pdf->SetXY(100, $yStart + $yAdded);
		$pdf->Cell(20,3,'Ship To:');
		$pdf->SetFont($font,'B',11);
		$yAdded += 5;
		$pdf->SetXY(12, $yStart + $yAdded);
		$pdf->MultiCell(90, 4.8, $data->shipFrom);
		$pdf->SetXY(100, $yStart + $yAdded);
		$pdf->MultiCell(90, 4.8, $data->shipTo);
		$yAdded += 29;
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFillColor(235);
		$pdf->SetFont($font,'',11);
		// Build header row for order data

		$pdf->Cell(32,10,$data->orderDataCellHeaders[0], 1, 0, 'C', true);
		$pdf->Cell(32,10,$data->orderDataCellHeaders[1], 1, 0, 'C', true);
		$pdf->Cell(32,10,$data->orderDataCellHeaders[2], 1, 0, 'C', true);
		$pdf->Cell(38.4,10,$data->orderDataCellHeaders[3], 1, 0, 'C', true);
		$pdf->Cell(60.4,10,$data->orderDataCellHeaders[4], 1, 0, 'C', true);
		$yAdded += 10;
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFont($font,'B',11);
		$pdf->Cell(32,16,$data->orderDataCellContent[0], 1, 0, 'C', false);
		$pdf->Cell(32,16,$data->orderDataCellContent[1], 1, 0, 'C', false);
		$pdf->Cell(32,16,$data->orderDataCellContent[2], 1, 0, 'C', false);
		$pdf->Cell(38.4,16,$data->orderDataCellContent[3], 1, 0, 'C', false);
		if ($pdf->GetStringWidth($data->orderDataCellContent[4]) > 60.4) {
			$pdf->MultiCell(60.4, 8, $data->orderDataCellContent[4], 1, 'C', false);
		}
		else {
			$pdf->Cell(60.4,16,$data->orderDataCellContent[4], 1, 0, 'C', false);
		}
		// $pdf->MultiCell(40.4, 8, $order->ShipService, 1, 'C', false);
		$yAdded += 20;
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFont($font,'',11);
		// $pdf->Cell(40,10,$data->itemDataCellHeaders[0], 1, 0, 'C', true);
		$pdf->Cell(150.8,10,$data->itemDataCellHeaders[0], 1, 0, 'C', true);
		$pdf->Cell(44,10,$data->itemDataCellHeaders[1], 1, 0, 'C', true);
		// $pdf->Cell(22,10,$data->itemDataCellHeaders[3], 1, 0, 'C', true);
		$yAdded += 10;
		$itemsYStart = $yStart + $yAdded;
		$pdf->SetFont($font,'B',11);
		$i = 0;
		$itemsYDiff = 0;
		foreach($data->itemData as $itemInfo) {
			// First determine required height of the cells based on the number of lines in the description cell
			$itemDescriptionFontSize = 14;
			$pdf->SetFont($font,'B',$itemDescriptionFontSize);
			$numLines = ceil($pdf->GetStringWidth($itemInfo->description) / 145) + 3;
			$itemCellHeight = 5 + $numLines * $itemDescriptionFontSize / 1.65;

			// Then write the cells
			$pdf->SetFont($font,'B',11);
			$pdf->SetXY(10, $itemsYStart + $itemsYDiff);
			// $pdf->Cell(40, $itemCellHeight, $itemInfo->itemId, 1, 0, 'C', false);
			$pdf->Cell(150.8,$itemCellHeight,'', 1, 0, 'C', false);
			$pdf->SetFont($font,'B',25);
			$pdf->Cell(44,$itemCellHeight,$itemInfo->quantity, 1, 0, 'C', false);

			// Draw a box around quantities if quantity is greater than 1
			$pdf->SetFont($font,'B',25);
			$rectangleHorizontalWidth = $pdf->GetStringWidth($itemInfo->quantity)+6;
			$rectangleTopLeftX = 182.5 - ($rectangleHorizontalWidth / 2);
			$rectangleTopLeftY = $pdf->GetY() + ($itemCellHeight / 2) - 6.5;
			if ((int)$itemInfo->quantity > 1) {
				$pdf->SetFillColor(0);
				$pdf->Rect($rectangleTopLeftX, $rectangleTopLeftY, $rectangleHorizontalWidth, 1, 'FD'); // Top bar
				$pdf->Rect($rectangleTopLeftX, $rectangleTopLeftY, 1, 12, 'FD'); // Left Bar
				$pdf->Rect($rectangleTopLeftX + $rectangleHorizontalWidth -1, $rectangleTopLeftY, 1, 12, 'FD'); // Right Bar
				$pdf->Rect($rectangleTopLeftX, $rectangleTopLeftY+11.5, $rectangleHorizontalWidth, 1, 'FD'); // Bottom Bar
			}


			// $pdf->Cell(22,$itemCellHeight,$itemInfo->shippedQty, 1, 0, 'C', false);
			$pdf->SetFont($font,'B',$itemDescriptionFontSize);
			$pdf->SetXY(13, $itemsYStart + ($i*$itemCellHeight) + 5);
			// $pdf->MultiCell(100, $itemCellHeight / ($itemDescriptionFontSize / 2), $itemInfo->description, 0, 'L', false);
			$pdf->MultiCell(150, $itemDescriptionFontSize/2.4, $itemInfo->description, 0, 'L', false);
			$pdf->SetFont($font,'B',11);// Add the diff for the next line
			$itemsYDiff += $itemCellHeight;
		}
		// exit;
		$pdf->SetFillColor(240);
		$pdf->SetXY(10, $itemsYStart + $itemsYDiff + 5);
		$pdf->Rect(10, $pdf->GetY(), 194.8, 20, 'FD');
		$pdf->SetFont($font,'',12);
		$pdf->SetXY(12, $itemsYStart + $itemsYDiff + 10);
		$pdf->MultiCell(190, 5, $infoString, 0, 'L', false);
		$barBaseline = 1.4;
		$barHeight = 12;
		$barCodeXPos = 10 + (190 / 2) - ($pdf->GetCode39Width($data->orderDataCellContent[0], $barBaseline, $barHeight) / 2);
		$pdf->Code39($barCodeXPos, $itemsYStart + $itemsYDiff + 40, $data->orderDataCellContent[0], $barBaseline, $barHeight, true, 'C');
		// $pdf->Cell(190, 5, $pdf->GetCode39Width($data->orderDataCellContent[0], 1, 10));

		// echo "Width:";
		// echo $pdf->GetCode39Width($data->orderDataCellContent[0], 1, 10);




		$pdf->Output($destination, $this->getPackingSlipPath($data->orderId, $data->channel, $destination));
	}

	public function generateTtsRmaPackingSlip($orderId, $destination='I') {
		require_once(HTML_FOLDER."classes/RMA/DisplayOrder.php");
		$this->shipFromString = "Tech To School\n1530 Montague Expwy\nSan Jose, CA 95131-1408";
		$sql = "SELECT * FROM ds_orders WHERE `oID`='$orderId';";
		$order = new DisplayOrder($this->db_query($sql, 'select-single', true));
		$pdfData = $this->buildTtsRmaPdfData($order);
		// echo "<pre>";
		// print_r($pdfData);
		// echo "</pre>";
		// exit;
		return $this->buildTtsRmaPdf($pdfData, $destination);
	}

	private function buildTtsRmaPdfData($order) {
		$logo = HTML_FOLDER."img/TTSLogo2020kwhitecenter.png";
		$orderData = $order->getPdfData();
		$pdfData = new stdClass;
		$pdfData->orderId = $order->getONum();
		$pdfData->shipFrom = $this->shipFromString;
		$pdfData->logo = $logo;
		$pdfData->problem = $orderData->problem;
		$pdfData->servicesPerformed = $orderData->servicesPerformed;

		// Parse shipping/address data
		$shippingData = $order->getShippingAddress();
		$shipToString = '';
		$organization = $shippingData->organization;
		$name = $shippingData->name;
		$address1 = $shippingData->address1;
		$address2 = $shippingData->address2;
		$city = $shippingData->city;
		$state = $shippingData->state;
		$zip = $shippingData->zip;
		$country = $shippingData->country;
		if ($organization) {
			$shipToString .= "$organization\n";
		}
		if ($name) {
			$shipToString .= "$name\n";
		}
		if ($address1) {
			$shipToString .= "$address1\n";
		}
		if ($address2) {
			$shipToString .= "$address2\n";
		}
		if ($city) {
			$shipToString .= "$city, ";
		}
		if ($state) {
			$shipToString .= "$state ";
		}
		if ($zip) {
			$shipToString .= "$zip\n";
		}

		$pdfData->shipTo = $shipToString;
		$pdfData->shipService = $order->getPdfData()->shipService;
		$orderDate = DateTime::createFromFormat('Y-m-d', $orderData->dateCreation);
		$completionDate = DateTime::createFromFormat('Y-m-d', $orderData->dateCompleted);
		$pdfData->orderDate = $orderDate ? $orderDate->format('m/d/Y') : '';
		$pdfData->completionDate = $completionDate ? $completionDate->format('m/d/Y') : '';

		// Construct order data cells
		$pdfData->orderDataCellHeaders = array('Order Number', 'Order Date', "", 'Completion Date', 'Shipping Method');
		$pdfData->orderDataCellContent = array($pdfData->orderId, $pdfData->orderDate, '', $pdfData->completionDate, $pdfData->shipService);


		return $pdfData;
	}

	private function buildTtsRmaPdf($data, $destination='I') {
		require_once('pdf/class_packingslip_pdf.php');
		$infoString = "For questions or issues regarding your order, please contact us by visiting techtoschool.com/support";
		$yStart = 30;
		$yAdded = 0;
		$font = 'Arial';
		$pdf = new PackingSlipPDF();
		$pdf->imgHeader = $data->logo;
		$pdf->SetTopMargin(10);
		$pdf->AddPage('P','Letter');
		$pdf->SetFont($font,'',12);
		$pdf->SetXY(10, $yStart);
		$pdf->Rect(10, $pdf->GetY()+5, 194.8, 36);
		$yAdded += 12;
		$pdf->SetXY(12, $yStart + $yAdded);
		$pdf->Cell(20,3,'Ship From:');
		$pdf->SetXY(100, $yStart + $yAdded);
		$pdf->Cell(20,3,'Ship To:');
		$pdf->SetFont($font,'B',11);
		$yAdded += 5;
		$pdf->SetXY(12, $yStart + $yAdded);
		$pdf->MultiCell(90, 4.8, $data->shipFrom);
		$pdf->SetXY(100, $yStart + $yAdded);
		$pdf->MultiCell(90, 4.8, $data->shipTo);
		$yAdded += 30;
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFillColor(235);
		$pdf->SetFont($font,'',11);

		// Build header row for order data
		$pdf->Cell(32,10,$data->orderDataCellHeaders[0], 1, 0, 'C', true);
		$pdf->Cell(32,10,$data->orderDataCellHeaders[1], 1, 0, 'C', true);
		$pdf->Cell(98.8,10,$data->orderDataCellHeaders[2], 1, 0, 'C', true);
		$pdf->Cell(32,10,$data->orderDataCellHeaders[3], 1, 0, 'C', true);
		// $pdf->Cell(60.4,10,$data->orderDataCellHeaders[4], 1, 0, 'C', true);
		$yAdded += 10;
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFont($font,'B',11);
		$pdf->Cell(32,16,$data->orderDataCellContent[0], 1, 0, 'C', false);
		$pdf->Cell(32,16,$data->orderDataCellContent[1], 1, 0, 'C', false);
		$pdf->Cell(98.8,16,$data->orderDataCellContent[2], 1, 0, 'C', false);
		$pdf->Cell(32,16,$data->orderDataCellContent[3], 1, 0, 'C', false);
		// if ($pdf->GetStringWidth($data->orderDataCellContent[4]) > 60.4) {
		// 	$pdf->MultiCell(60.4, 8, $data->orderDataCellContent[4], 1, 'C', false);
		// }
		// else {
		// 	$pdf->Cell(60.4,16,$data->orderDataCellContent[4], 1, 0, 'C', false);
		// }

		$yAdded += 20;

		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFont($font,'',11);
		$pdf->Cell(194.8,10,'Service Requested', 1, 0, 'C', true);
		$yAdded += 10;
		$problemYStart = $yStart + $yAdded;
		$pdf->SetFont($font,'B',11);
		$problemYDiff = 0;

		// First determine required height of the cells based on the number of lines in the description cell
		$problemDescriptionFontSize = 11;
		$pdf->SetFont($font,'',$problemDescriptionFontSize);
		$rawNumLines = count(explode(PHP_EOL, $data->problem));
		$numAdditionalLines = floor($pdf->GetStringWidth($data->problem) / 190);
		$numLines = $rawNumLines + $numAdditionalLines;
		// echo "<p>$rawNumLines $numAdditionalLines $numLines</p>";
		// exit;
		$problemCellHeight = 4 + $numLines * $problemDescriptionFontSize / 1.75;

		// Then write the cell
		$pdf->SetFont($font,'',11);
		$pdf->SetXY(10, $problemYStart + $problemYDiff);
		$pdf->Cell(194.8,$problemCellHeight,'', 1, 0, 'C', false);


		$pdf->SetFont($font,'',$problemDescriptionFontSize);
		$pdf->SetXY(13, $problemYStart + ($i*$problemCellHeight) + 4);
		$pdf->MultiCell(190, $problemDescriptionFontSize/2.4, $data->problem, 0, 'L', false);
		$pdf->SetFont($font,'',11);// Add the diff for the next line
		$problemYDiff += $problemCellHeight;

		$yAdded += $problemYDiff + 5;

		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->SetFont($font,'',11);
		$pdf->Cell(194.8,10,'Service Performed', 1, 0, 'C', true);
		$yAdded += 10;

		$pdf->SetFont($font,'',$problemDescriptionFontSize);
		$numLines = count(explode(PHP_EOL, $data->servicesPerformed));
		// $numAdditionalLines = ceil($pdf->GetStringWidth($data->servicesPerformed) / 194);
		// $numLines += $numAdditionalLines;
		$rawNumLines = count(explode(PHP_EOL, $data->servicesPerformed));
		$numAdditionalLines = floor($pdf->GetStringWidth($data->servicesPerformed) / 194);
		$numLines = $rawNumLines + $numAdditionalLines;
		$servicesPerformedCellHeight = 5 + $numLines * $problemDescriptionFontSize / 1.75;


		// Then write the cell
		$pdf->SetFont($font,'',11);
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->Cell(194.8,$servicesPerformedCellHeight,'', 1, 0, 'C', false);

		$pdf->SetFont($font,'',$problemDescriptionFontSize);
		$pdf->SetXY(13, $yStart + $yAdded + ($i*$servicesPerformedCellHeight) + 4);
		$pdf->MultiCell(190, $problemDescriptionFontSize/2.4, $data->servicesPerformed, 0, 'L', false);
		$pdf->SetFont($font,'',11);// Add the diff for the next line

		$yAdded += $servicesPerformedCellHeight + 5;
		$pdf->SetFillColor(240);
		$pdf->SetXY(10, $yStart + $yAdded);
		$pdf->Rect(10, $pdf->GetY(), 194.8, 15, 'FD');
		$pdf->SetFont($font,'',12);
		$yAdded += 5;
		$pdf->SetXY(12, $yStart + $yAdded);
		$pdf->MultiCell(190, 5, $infoString, 0, 'L', false);
		$barBaseline = 1.4;
		$barHeight = 12;
		$barCodeXPos = 10 + (190 / 2) - ($pdf->GetCode39Width($data->orderDataCellContent[0], $barBaseline, $barHeight) / 2);
		$yAdded += 30;
		$pdf->Code39($barCodeXPos, $yStart + $yAdded, $data->orderDataCellContent[0], $barBaseline, $barHeight, true, 'C');

		if ($destination == 'I') {
			$name = "$data->orderId-packingslip.pdf";
		}
		elseif ($destination == 'F') {
			$name = HTML_FOLDER . "packingslips/$data->orderId-packingslip.pdf";
		}
		$pdf->Output($destination, $name);
		return $name;
	}

	/**
	 * Merges packing slips with tracking numbers.
	 * @param string $orderId Order ID for packing slip
	 * @param string $channel Channel identifier eg. newegg and ebay
	 * @param array $trackingNumbers Array of tracking numbers
	 */
	function mergePackingSlipAndTrackingNumbers($orderId, $channel, $trackingNumbers) {
		// require_once('pdf/class_packingslip_pdf.php');
		// include 'pdf/class_pdfmerger.php';
		require_once('pdf/class_pdfmerger.php');
		$packingSlipPath = $this->getPackingSlipPath($orderId, $channel, 'F');
		foreach($trackingNumbers as $trackingNumber) {
			$pdf = new PDFMerger;
			// echo "test";
			$pdf->addPDF($packingSlipPath);
			$filename = "$trackingNumber.pdf";
			$file = UPS_LABEL_PATH . $filename;
			$pdf->addPDF($file);
			$pdf->merge('browser', "$orderId"."_merged_packingslip_tracking.pdf");
		}
	}

	/**
	 * Merges packing slips with tracking numbers.
	 * @param string $orderId Order ID for packing slip
	 * @param string $channel Channel identifier eg. newegg and ebay
	 * @param array $trackingNumbers Array of tracking numbers
	 */
	function mergePackingSlipAndTrackingNumbersV2($orderId, $channel, $trackingNumbers) {
		$packingSlipPath = $this->getPackingSlipPath($orderId, $channel, 'F');
		foreach($trackingNumbers as $trackingNumber) {
			$shippingLabelPath = UPS_LABEL_PATH . "$trackingNumber.pdf";
			$mergedImage = $this->generateMergedPng($packingSlipPath, $shippingLabelPath, $orderId, $trackingNumber);
			$pdf = new FPDF;
			$pdf->AddPage('L', 'letter');
			$pdf->Image($mergedImage, 1, 1, 280, 280 * 0.6875);
			$pdf->Output('I');
			unlink($mergedImage);
		}
	}

	public function printPackingSlips($orderId, $channel, $trackingNumbers, $verboseLogging=false) {
		if (substr($channel, 0, 4) == 'ebay') {
			$channel = 'ebay';
		}
		if ($verboseLogging) {
			echo "In printPackingSlips function.\n";
			echo "Parameters passed:\n";
			echo "\torderId: $orderId\n\tchannel: $channel\n\ttrackingNumbers: $trackingNumbers\n";
		}
		$returnData = array();
		if (!$trackingNumbers) {
			if ($verboseLogging) {
				echo "No passed tracking numbers found. Preparing to fetch.\n";
			}
			$trackingNumbers = array();
			$sql = "SELECT tracking_number FROM ecomm_open_order_tracking_numbers WHERE open_order_id=(SELECT id FROM ecomm_open_orders WHERE order_id = '$orderId' LIMIT 1);";
			if ($verboseLogging) {
				echo "Query to fetch tracking numbers: $sql\n";
			}
			$results = $this->db_query($sql, 'select-multiple', true);
			if ($verboseLogging) {
				echo "Results:\n";
				print_r($results);
				echo "\n";
			}
			foreach ($results as $result) {
				array_push($trackingNumbers, $result);
			}
			if ($verboseLogging) {
				echo "trackingNumbers array after results pushed:\n";
				print_r($trackingNumbers);
				echo "\n";
			}
		}
		foreach ($trackingNumbers as $trackingNumber) {
			$trackingNum = $trackingNumber['tracking_number'];
			$this->printMergedPackingSlipAndShippingLabel($orderId, $channel, $trackingNum);
		}
		$returnData['printer'] = $printer;
		$returnData['path'] = $path;
		$returnData['filename'] = $filename;
		$returnData['file'] = $file;
		$returnData['url'] = $url;
		$returnData['success'] = true;
		$sql = "UPDATE `ecomm_open_orders` SET `labels_printed`='1', `error`=NULL WHERE `order_id`='$orderId';";
		$this->db_query($sql, 'update', true);
		return $returnData;
	}

	function printMergedPackingSlipAndShippingLabel($orderId, $channel, $trackingNumber) {
		require_once(HTML_FOLDER.'vendor/autoload.php');
		require_once(HTML_FOLDER.'class_fpdf.php');
		$printer = ECOMM_FULFILLMENT_INVENTORY_CAGE_PRINTER_2;

		$packingSlipPath = $this->getPackingSlipPath($orderId, $channel, 'F');
		if (!$this->packingSlipIsGenerated($orderId, $channel)) {
			$this->generatePackingSlip($orderId, $channel, 'F');
		}
		$shippingLabelPath = UPS_LABEL_PATH . "$trackingNumber.pdf";
		$mergedLabelPath = HTML_FOLDER . "tmp/" . $order_id . "_" . $trackingNumber . ".pdf";
		$mergedImage = $this->generateMergedPng($packingSlipPath, $shippingLabelPath, $orderId, $trackingNumber);
		$pdf = new FPDF;
		$pdf->AddPage('L', 'letter');
		$pdf->Image($mergedImage, 5, 5, 278, 278 * 0.6875);
		unlink($mergedImage);
		$pdf->Output('F', $mergedLabelPath);
		$this->send_printer($printer, $mergedLabelPath, array('orientation-requested' => 'landscape'));

		unlink($mergedLabelPath);
	}

	function getMergedPackingSlipAndShippingLabel($orderId, $channel, $trackingNumber) {
		require_once(HTML_FOLDER.'vendor/autoload.php');
		require_once(HTML_FOLDER.'class_fpdf.php');
		if (!$this->packingSlipIsGenerated($orderId, $channel)) {
			$this->generatePackingSlip($orderId, $channel, 'F');
		}
		$packingSlipPath = $this->getPackingSlipPath($orderId, $channel, 'F');
		$shippingLabelPath = UPS_LABEL_PATH . "$trackingNumber.pdf";
		$mergedLabelPath = HTML_FOLDER . "tmp/" . $order_id . "_" . $trackingNumber . ".pdf";
		$mergedImage = $this->generateMergedPng($packingSlipPath, $shippingLabelPath, $orderId, $trackingNumber);
		$pdf = new FPDF;
		$pdf->AddPage('L', 'letter');
		$pdf->Image($mergedImage, 1, 1, 280, 280 * 0.6875);
		$pdf->Output('I');
		unlink($mergedImage);
	}

	/**
	 * Generates merged png and returns the route to it
	 * @param string $packingSlipPath Path to the packing slip file
	 * @param string $shippingLabelPath Path to the shipping label file
	 * @param int $ppi Points per inch for file
	 * @return string image
	 */
	function generateMergedPng($packingSlipPath, $shippingLabelPath, $orderId, $trackingNumber, $ppi = 192) {
		$imagePath = HTML_FOLDER . "tmp/" . $orderId . "_" . $trackingNumber . ".png";
		$ppi = 192;
		$image = new Imagick();
		$image->setResolution($ppi, $ppi);
		$image->newImage(16*$ppi, 11*$ppi, new ImagickPixel('white'));
		$image->setImageFormat('png');
		$packingSlip = new Imagick();
		$packingSlip->setResolution($ppi, $ppi);
		$packingSlip->readImage($packingSlipPath);
		$packingSlip->setImageFormat('png');
		$packingSlip->scaleImage($packingSlip->getImageWidth()*0.9, 0);

		$shippingLabel = new Imagick();
		$shippingLabel->setResolution($ppi, $ppi);
		$shippingLabel->readImage($shippingLabelPath);
		$shippingLabel->setImageFormat('png');
		$shippingLabel->scaleImage(0, $ppi*12);


		$image->compositeImage($packingSlip, Imagick::COMPOSITE_DEFAULT, 0, 0);
		$image->compositeImage($shippingLabel, Imagick::COMPOSITE_DEFAULT, $ppi*8.45, $ppi*0.15);
		$image->writeImage($imagePath);
		return $imagePath;
	}

	public function printBackMarketPackingSlip($url, $orderId, $trackingNumbers, $path='', $verboseLogging=false) {
		require_once(HTML_FOLDER.'vendor/autoload.php');
		require_once(HTML_FOLDER.'class_fpdf.php');
		if ($path == '') {
			$path=HTML_FOLDER.'tmp';
		}
		$printer = ECOMM_FULFILLMENT_INVENTORY_CAGE_PRINTER_2;
		$verboseLogging = false;
		if ($_POST['verboselogging'] == 'yes') {
			$verboseLogging = true;
		}
		if ($verboseLogging) {
			echo "In printBackMarketPackingSlip($url, $orderId)\n";
		}
		$filename = substr($url, strpos($url, 'Shipping_label'), 25);

		$returnData = array();

		if ($url) {
			file_put_contents("$path/$filename", file_get_contents($url));

			$packingSlipPath = "$path/$filename";
			// exit;

			foreach ($trackingNumbers as $trackingNumber) {
				$trackingNum = $trackingNumber['tracking_number'];
				$shippingLabelPath = UPS_LABEL_PATH . "$trackingNum.pdf";
				$mergedLabelPath = HTML_FOLDER . "tmp/" . $orderId . "_" . $trackingNum . ".pdf";
				if ($verboseLogging) {
					echo "In tracking number: $trackingNum\n";
				}
				$mergedImage = $this->generateMergedPng($packingSlipPath, $shippingLabelPath, $orderId, $trackingNum, 300);

				$pdf = new FPDF;
				$pdf->AddPage('L', 'letter');
				$pdf->Image($mergedImage, 5, 5, 278, 278 * 0.6875);
				unlink($mergedImage);
				$pdf->Output('F', $mergedLabelPath);
				$this->send_printer($printer, $mergedLabelPath, array('orientation-requested' => 'landscape'));

				unlink($mergedLabelPath);
			}

			unlink($packingSlipPath);
			$returnData['printer'] = $printer;
			$returnData['path'] = $path;
			$returnData['filename'] = $filename;
			$returnData['file'] = $file;
			$returnData['url'] = $url;
			$returnData['success'] = true;
		}
		else {
			$returnData['success'] = false;
		}
		return $returnData;
	}


}

?>
