<?php

class RequestSigner {
	private $host = 'sellingpartnerapi-na.amazon.com';

	private $method;
	private $uri;
	private $headers;
	private $payload;
	private $algorithm = "AWS4-HMAC-SHA256";
	private $requestDate;
	private $credentials;

	public function __construct($method, $uri, $headers, $payload='', $credentials=null) {
		$this->uri = self::parseURI($uri);
		$this->headers = self::parseHeaders($headers);
		$this->method = $method;
		$this->requestDate = $headers['x-amz-date'];
		$this->credentials = $credentials;
	}

	public function signRequestHeaders($accessKeyId, $secretKey) {
		$authenticationHeader = $this->createAuthenticationHeader($accessKeyId, $secretKey);
		$newHeaders = array(
			$authenticationHeader
		);
		foreach ($this->headers as $key => $value) {
			array_push($newHeaders, "$key: $value");
		}
		return $newHeaders;
	}

	public function signRequest($accessKeyId, $secretKey) {
		// $this->createSignature($secretKey);
		$canonicalRequest = $this->createCanonicalRequest();
		$hashedRequest = hash('sha256', $canonicalRequest);
		$response = new stdClass;
		$response->canonicalRequest = $canonicalRequest;
		$response->hashedRequest = $hashedRequest;
		$response->stringToSign = $this->createStringToSign();
		// $response->signature = $this->calculateSignature($secretKey);
		$response->signature = $this->createSignature($secretKey);
		$response->authenticationHeader = $this->createAuthenticationHeader($accessKeyId, $secretKey);
		return $response;
	}

	private function createCanonicalRequest() {
		$method = $this->method;
		$canonicalURI = $this->getCanonicalURI();
		$canonicalQueryString = $this->getCanonicalQueryString();
		$canonicalHeaders = $this->getCanonicalHeaders();
		$signedHeaders = $this->getSignedHeaders();
		$hashedPayload = $this->getHashedPayload();
		$canonicalRequest = "$method\n$canonicalURI\n$canonicalQueryString\n$canonicalHeaders\n$signedHeaders\n$hashedPayload";
		return $canonicalRequest;
	}

	private function createStringToSign() {
		$algorithm = $this->algorithm;
		$date = $this->requestDate;
		$credentialScope = $this->credentials;
		$credentialString = "$credentialScope->date/$credentialScope->region/$credentialScope->service/$credentialScope->terminationString";
		$hashedCanonicalRequest = hash('sha256', $this->createCanonicalRequest());
		$stringToSign = "$algorithm\n$date\n$credentialString\n$hashedCanonicalRequest";
		return $stringToSign;

	}

	private function calculateSignature($secretKey) {
		$signingKey = $this->createSigningKey($secretKey);
		$signature = hash_hmac('sha256', $signingKey, $this->createStringToSign());
		return $signature;
	}

	private function createSignature($secretKey) { // $secretKey is the AWS secret key from IAM user
		$stringToSign = $this->createStringToSign(); // Derived according to docs: https://docs.aws.amazon.com/general/latest/gr/sigv4-create-string-to-sign.html
		$algorithm = 'sha256';
		$kSecret = "AWS4".$secretKey;
		$date = $this->credentials->date; // Date derived per amazon documentation eg. 20201020 for 20 Oct 2020
		$region = $this->credentials->region; // Region for request eg. us-east-1
		$service = $this->credentials->service; // Service eg. execute-api
		$terminationString = $this->credentials->terminationString; // Signature termination string eg. aws4_request
		$kDate = hash_hmac($algorithm, $date, $kSecret, true);
		$kRegion = hash_hmac($algorithm, $region, $kDate, true);
		$kService = hash_hmac($algorithm, $service, $kRegion, true);
		$kSigning = hash_hmac($algorithm, $terminationString, $kService, true);

		$signature = hash_hmac($algorithm, $stringToSign, $kSigning);
		return trim($signature);
	}

	private function createSigningKey($secretKey) {
		$kSecret = "AWS4".$secretKey;
		$kDate = hash_hmac('sha256', $kSecret, $this->credentials->date, true);
		$kRegion = hash_hmac('sha256', $kDate, $this->credentials->region, true);
		$kService = hash_hmac('sha256', $kRegion, $this->credentials->service, true);
		$kSigning = hash_hmac('sha256', $kService, $this->credentials->terminationString, true);
		return $kSigning;
	}

	private function createAuthenticationHeader($accessKeyId, $secretKey) {
		$credentialScope = $this->getCredentialScope();
		$signedHeaders = $this->getSignedHeaders();
		$signature = $this->createSignature($secretKey);
		$algorithmString = "Authorization: $this->algorithm";
		$credentialString = "Credential=$accessKeyId/$credentialScope,";
		$signedHeadersString = "SignedHeaders=$signedHeaders,";
		$signatureString = "Signature=$signature";
		return "$algorithmString $credentialString $signedHeadersString $signatureString";
	}

	private function getCanonicalURI() {
		if ($this->uri->path) {
			$encodedUri = '/';
			$pathSegments = explode('/', $this->uri->path);
			$i = 0;
			foreach ($pathSegments as $segment) {
				if ($i) {
					$encodedUri .= '/';
				}
				$encodedUri .= rawurlencode(rawurlencode($segment)); // Needs to be URI-encoded *twice*. See https://docs.aws.amazon.com/general/latest/gr/sigv4-create-canonical-request.html step 2
				$i++;
			}
			return $encodedUri;
		}
		else {
			return "/"; // If no path, return just a forward slash.
		}
	}

	private function getCanonicalQueryString() {
		if (!$this->uri->query || $this->uri->query == '') {
			return '';
		}
		$queryArray = self::parseQueryString($this->uri->query);
		$queryString = '';
		$i = 0;
		foreach ($queryArray as $key => $value) {
			if ($i) { // If this is not the first query parameter, append an ampersand sign.
				$queryString .= '&';
			}
			$encodedKey = self::encodeString($key); // Encode parameter name
			$encodedValue = self::encodeString($value); // Encode parameter value
			$queryString .= "$encodedKey=$encodedValue"; // Append key-value pair to query string
			$i++; // Iterate
		}

		return $queryString;
	}

	private function getCanonicalHeaders() {
		$canonicalHeaders = '';
		foreach ($this->headers as $name => $value) {
			$canonicalHeaders .= "$name:$value\n";
		}
		return $canonicalHeaders;
	}

	private function getSignedHeaders() {
		$signedHeaders = '';
		$i = 0;
		foreach ($this->headers as $name => $value) {
			if ($i) {
				$signedHeaders .= ';';
			}
			$signedHeaders .= "$name";
			$i++;
		}
		return $signedHeaders;
	}

	private function getHashedPayload() {
		return hash('sha256', $this->payload);
	}

	private function getCredentialScope() {
		$credentials = $this->credentials;
		return "$credentials->date/$credentials->region/$credentials->service/$credentials->terminationString";
	}

	private static function parseURI($uri) {
		$schemeStart = 0;
		$schemeEnd = strpos($uri, ':');
		$authorityStart = strpos($uri, '//') + 2;
		$authorityEnd = strpos($uri, '/', $authorityStart);
		if ($authorityEnd === false) {
			$authorityEnd = strlen($uri);
		}
		$pathStart = $authorityEnd + 1;
		$pathEnd = strpos($uri, '?');
		if ($pathEnd === false) {
			$pathEnd = strlen($uri);
		}
		$queryStart = $pathEnd + 1;
		$queryEnd = strpos($uri, '#');
		if ($queryEnd === false) {
			$queryEnd = strlen($uri);
		}


		$scheme = substr($uri, $schemeStart, $schemeEnd);
		$authority = substr($uri, $authorityStart, $authorityEnd-$authorityStart);
		$path = substr($uri, $pathStart, $pathEnd-$pathStart);
		$query = substr($uri, $queryStart, $queryEnd-$queryStart);

		$uri = new stdClass;
		$uri->scheme = $scheme;
		$uri->authority = $authority;
		$uri->path = $path;
		$uri->query = $query;
		return $uri;
	}

	private static function parseQueryString($queryString) {
		$queryParams = explode('&', $queryString);
		$queryArray = array();
		foreach ($queryParams as $param) {
			$key = substr($param, 0, strpos($param, '='));
			$value = substr($param, strpos($param, '=') + 1);
			$queryArray[$key] = $value;
		}
		ksort($queryArray);
		return $queryArray;
	}

	private static function parseHeaders($headers) {
		$normalizedHeaders = array();
		foreach ($headers as $name => $value) {
			$normalizedHeaders[strtolower($name)] = trim(preg_replace('/\s+/', ' ', $value));
		}
		ksort($normalizedHeaders);
		return $normalizedHeaders;
	}

	private static function encodeString($string) {
		$unreservedPattern = '/[A-Za-z0-9\-\+\_\.\~]/'; // Regex for unreserved RFC 3986 characters. See https://tools.ietf.org/html/rfc3986#section-2.3
		$stringAsArray = str_split($string);
		$encodedString = '';
		foreach ($stringAsArray as $char) { // Build the new encoded value string
			if (!preg_match($unreservedPattern, $char)) { // Check if it's a nonreserved character in RFC 3986. If it's not, encode it.
				if ($char == '=') {
					$encodedString .= rawurlencode(rawurlencode($char)); // Equals signs in query params need to be double encoded. See https://docs.aws.amazon.com/general/latest/gr/sigv4-create-canonical-request.html step 3b
				}
				else {
					$encodedString .= rawurlencode($char); // Other reserved characters only need to be encoded once.
				}
			}
			else {
				$encodedString .= $char; // If it's an unreserved character, append as normal
			}
		}

		return $encodedString;
	}
}

?>
