<?php

require HTML_FOLDER . "vendor/autoload.php";

use Aws\Sts\StsClient;
use Aws\AwsClient;
use Aws\Exception\AwsException;
use Aws\Credentials\CredentialProvider;
use Aws\Credentials\Credentials;
use Aws\Signature\SignatureProvider;
use Aws\Signature\SignatureV4;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Uri;
use GuzzleHttp\Exception\ClientException;

class AmazonAPIV2 extends Database {
	private static $apiName = 'SourcePlus';
	private static $apiVersion = '0.1';
	private static $rootPath = __DIR__ . '/';

	private static $sandboxEndpoint = "https://sandbox.sellingpartnerapi-na.amazon.com";
	private static $productionEndpoint = "https://sellingpartnerapi-na.amazon.com";
	private static $paths = array(
		'getOrders' => array('path' => '/orders/v0/orders', 'method' => 'GET'),
		'inventory' => array('path' => '/fba/inventory/v1/summaries', 'method' => 'GET'),
		'listCatalogItems' => array('path' => '/catalog/v0/items', 'method' => 'GET'),
		'getMarketplaceParticipations' => array('path' => '/sellers/v1/marketplaceParticipations', 'method' => 'GET')
	);

	private $version;
	private $sandbox;
	private $debug;

	private $clientIdentifier;
	private $clientSecret;
	public function __construct($version='v2', $sandbox=false, $debug=false) {
		$this->version = $version;
		$this->sandbox = $sandbox;
		$this->debug = $debug;
		$this->clientIdentifier = $this->getClientIdentifier();
		$this->clientSecret = $this->getClientSecret();
	}

	public function testAmazonApi() {
		$model = "getMarketplaceParticipations";
		// $request = $this->createRequest(self::getMethod($model), $this->getUrl($model), $this->getHeaders());
		$request = new Request(self::getMethod($model), new Uri($this->getUrl($model)), $this->getHeaders());
		$signer = new SignatureV4('execute-api', 'us-east-1');
		$credentials = $this->getCredentials();
		echo "<pre>";
		echo $request->getMethod();
		echo "\n";
		echo $request->getUri();
		echo "\n";
		print_r($credentials);
		echo "</pre>";
		echo "<h1>Signing request</h1>";
		echo "<pre>";
		$signedRequest = $signer->signRequest($request, $credentials);
		print_r($signedRequest);
		echo "</pre>";
		$clientConfig = [

		];
		$baseUrl = $this->sandbox ? self::$sandboxEndpoint : self::$productionEndpoint;
		$client = new AwsClient([
			'version' => 'latest',
			'region' => 'us-east-1',
			'service' => 'execute-api',
			'credentials' => $credentials,
		]);
		// $client = new Client(['base_url' => $baseUrl]);
		echo "<h1>Sending request</h1>";
		echo "<pre>";
		try {
			echo "</pre>";
			// $client->execute()
			// $response = $client->send($signedRequest);
			echo "<h2>Request succeeded!</h2>";
			echo "<pre>";
			print_r($response);
			echo "</pre>";
		}
		catch(ClientException $e) {
			echo "</pre>";
			echo "<h2>Request failed.</h2>";
			echo "<h3>Request sent:</h3>";
			echo "<pre>";
			print_r($e->getRequest());
			echo "</pre>";
			echo "<h3>Response received:</h3>";
			echo "<pre>";
			print_r($e->getResponse());
			echo "</pre>";
		}
	}

	private function createRequest($method, $uri, array $headers = [], $body = null, $version = "1.1") {
		if (is_string($uri)) {
			$uri = new Uri($uri);
		}
		$request = new Request($method, $uri, $headers, $body, $version);
		return $request;
	}

	private static function getMethod($model) {
		return self::$paths[$model]['method'];
	}

	private function getUrl($model, $pathParameters=null, $queryParameters=null) {
		if ($this->sandbox) {
			return self::$sandboxEndpoint . self::$paths[$model]['path'];
		}
		else {
			return self::$productionEndpoint . self::$paths[$model]['path'];
		}
	}

	private function getCredentials() {
		$roleToAssume = trim(file_get_contents(self::$rootPath . "keys/aws/api_user_role/api_user_role_arn.txt"));
		$stsCredentials = new Aws\Credentials\Credentials(self::getAwsUserAccessKeyId(), self::getAwsUserSecretAccessKey());
		$stsClient = new StsClient([
			'version' => 'latest',
			'region' => 'us-east-1',
			'credentials' => $stsCredentials,
		]);
		try {
			$result = $stsClient->assumeRole([
				'RoleArn' => $roleToAssume,
				'RoleSessionName' => 'testsession1',
				'PolicyArns' => [
					[
						'arn' => "arn:aws:iam::905061731603:policy/SellingPartnerAPI",
					],
				],
			]);
			$credentials = array(
				'key' => $result['Credentials']['AccessKeyId'],
				'secret' => $result['Credentials']['SecretAccessKey'],
				'token'  => $result['Credentials']['SessionToken'],
				'expires' => $result['Credentials']['Expiration']->__toString()
			);
			file_put_contents(self::getStsTokenPath(), json_encode($credentials));
			// return $credentials;
			// return $result;
			$credentialsObj = new Aws\Credentials\Credentials($credentials['key'], $credentials['secret'], $credentials['token'], time()+3500);
			return $credentialsObj;
		}
		catch (AwsException $e) {
			echo "<pre>".$e->getMessage()."</pre>";
		}
	}

	/**
	 * Gets headers for most API calls.
	 */
	private function getHeaders() {
		$dateTime = new DateTime('now', new DateTimeZone('UTC'));//->format('Ymd\THis\Z');
		$requestDateTime = $dateTime->format('Ymd\THis\Z');
		return array(
			'Host' => $this->getHost(),
			'x-amz-access-token' => $this->getAccessToken(), // Fetches token. Will generate new token if already expired.
			'x-amz-date' => $requestDateTime,
			// 'x-amz-security-token' => $this->stsCredentials['token'],
			'user-agent' => self::getUserAgent()
		);
	}

	private function getHost() {
		if ($this->sandbox) {
			return 'sandbox.sellingpartnerapi-na.amazon.com';
		}
		else {
			return 'sellingpartnerapi-na.amazon.com';
		}
	}

	/**
	 * Gets client identifier from storage
	 */
	private function getClientIdentifier($version='v2') {
		return trim(file_get_contents(self::$rootPath . "keys/application/$version/client_identifier.txt"));
	}

	/**
	 * Gets client secret from storage
	 */
	private function getClientSecret($version='v2') {
		return trim(file_get_contents(self::$rootPath . "keys/application/$version/client_secret.txt"));
	}

	private static function getUserAgent() {
		return self::$apiName .'/' . self::$apiVersion . ' (Language=PHP)';
	}

	/**
	 * Gets token access path
	 */
	private static function getAccessTokenPath() {
		return self::$rootPath . "keys/lwa_token.txt";
	}

	/**
	 * Called when token is expired. Generates a new access token, stores it in lwa_token.txt, then returns the new access token to the calling function.
	 */
	private function getFreshAccessToken() {
		$url = 'https://api.amazon.com/auth/o2/token';
		$refreshToken = $this->getRefreshToken();
		$requestHeaders = array(
			'Host' => 'api.amazon.com',
			'Content-Type' => 'application/x-www-form-urlencoded;charset=UTF-8'
		);
		$requestData = "grant_type=refresh_token&refresh_token=$refreshToken&client_id=$this->clientIdentifier&client_secret=$this->clientSecret";
		if ($this->debug) {
			echo "<pre>Request headers:\n";
			print_r($requestHeaders);
			echo "\nRequest data:\n$requestData\nURL:\n$url\n</pre>";
		}
		$response = json_decode($this->curl_post_2($url, $requestData, $requestHeaders));
		if ($this->debug) {
			echo "<pre>New token response:\n";
			print_r($response);
			echo "</pre>";
		}
		if ($response->token_type && $response->token_type == 'bearer') {
			$tokenData = array(
				"access_token" => $response->access_token,
				"expires" => time() + ((int)$response->expires_in - 10)
			);
			$accessTokenPath = self::getAccessTokenPath();
			try {
				if (!file_exists($accessTokenPath)) {
					mkdir($accessTokenPath, 0700, true);
				}
			}
			catch (Exception $e) {
				echo $e;
			}
			$result = file_put_contents($accessTokenPath, json_encode($tokenData));
			return $response->access_token;
		}
	}

	/**
	 * Gets refresh token
	 */
	private function getRefreshToken() {
		if ($this->debug) {
			echo "<pre>";
			echo "Refresh token:\n";
			echo file_get_contents(self::$rootPath . 'keys/refresh_token.txt');
			echo "</pre>";
		}
		return trim(file_get_contents(self::$rootPath . 'keys/refresh_token.txt'));
	}

	/**
	 * Fetches access token for making API calls
	 */
	private function getAccessToken() {
		$tokenData = json_decode(file_get_contents(self::getAccessTokenPath())); // Get access token from storage
		if ($this->isTokenValid($tokenData)) { // Check if the token has expired yet
			return $tokenData->access_token; // If not expired, return token
		}
		else {
			return $this->getFreshAccessToken(); // If expired, get fresh token and return that instead
		}
	}

	/**
	 * Determines if access token has expired yet.
	 */
	private function isTokenValid($tokenData=null) {
		if (!file_exists(self::getAccessTokenPath())) {
			return false;
		}
		if (!$tokenData) {
			$tokenData = json_decode(file_get_contents(self::getAccessTokenPath()));
		}
		return $tokenData->expires > time();
	}

	/**
	 * Gets token STS path
	 */
	private static function getStsTokenPath() {
		return self::$rootPath . "keys/aws/sts/sts_token.txt";
	}

	private static function getAwsUserAccessKeyId() {
		return trim(file_get_contents(self::$rootPath . 'keys/aws/api_user/access_key_id.txt'));
	}
	private static function getAwsUserSecretAccessKey() {
		return trim(file_get_contents(self::$rootPath . 'keys/aws/api_user/secret_access_key.txt'));
	}
}
